package maslab.orc;

/** 
 * Orc wrapper for Infra-red range sensors.
 *
 * The manufacturer (Sharp) suggests that a line can be fit to a
 * function of the form 1/(d + Xd), where d is the distance and Xd is
 * some calibration value. We use parameters Xm and Xb as the
 * parameters of the line.
 *
 * V = 1/(d + Xd) * Xm + Xb
 *
 * Solving for d:
 *
 * d = (Xm/(V-Xb)) - Xd
 **/
public class IRRangeSensor implements RangeSensor
{
    Orc orc;
    int port;

    double Xd, Xm, Xb;
    double arcAngle=0.08;
    double voltageVariance=0.005;

    /** Configure the distance parameters. These parameters can be
     * generated by collecting distance/voltage data and fitting the
     * parameters with the equation listed above. **/
    public void setParameters(double Xd, double Xm, double Xb)
    {
	this.Xd=Xd;
	this.Xm=Xm;
	this.Xb=Xb;
    }

    /** Configure distance and other parameters. **/
    public void setParameters(double Xd, double Xm, double Xb, 
			      double arcAngle, double voltageVariance)
    {
	this.Xd=Xd;
	this.Xm=Xm;
	this.Xb=Xb;
	this.arcAngle=arcAngle;
	this.voltageVariance=voltageVariance;
    }

    /*
    public static IRRangeSensor makeS2Y0A02(Orc orc, int port)
    {
	IRRangeSensor s = new IRRangeSensor(orc, port);
	s.Xd=-0.0618;	        s.Xm=0.7388;                 	s.Xb=-.1141; 
	s.arcAngle=0.08;	s.voltageVariance=.005;

	return s;
    }
    */

    /** Create and return an IRRangeSensor configured with approximate
     * parameters for a Sharp 2Y0A02. **/
    public static IRRangeSensor make2Y0A02(Orc orc, int port)
    {
	IRRangeSensor s = new IRRangeSensor(orc, port);
	s.Xd=-0.0618;	        s.Xm=0.7389;                 	s.Xb=-.1141; 
	s.arcAngle=0.08;	s.voltageVariance=.005;

	return s;
    }

    /** Create and return an IRRangeSensor configured with approximate
     * parameters for a Sharp GP2D12. **/
    public static IRRangeSensor makeGP2D12(Orc orc, int port)
    {
	IRRangeSensor s = new IRRangeSensor(orc, port);
	s.Xd=0.0828;	        s.Xm=0.1384;                 	s.Xb=0.2448; 
	s.arcAngle=0.08;	s.voltageVariance=.005;

	return s;
    }

    /** Create an IRRangeSensor without any parameters-- useful only
	when you provide your own parameters. Otherwise, use (e.g.)
	makeGP2D12.
     **/
    public IRRangeSensor(Orc orc, int port)
    {
	this.orc=orc;
	this.port=port;

	orc.pinModeWrite(port, Orc.PinMode.ANALOG_IN);
    }

    public double getRange()
    {
	double v=orc.analogRead(port);

	double range = Xm/(v-Xb) + Xd;
	if (range<0 || range > 100)
	    return 100;

	return range;
    }

    public double getRangeUncertainty()
    {
	/* We assume some constant uncertainty in voltage, and map this
	   to uncertainty in distance by considering the derivative of
	   our function fit.

	   The derivative is:

	   dd/dV = -Xm / ((V-Xb)^2)

	   This gives us a line around V, a local approximation of the
	   curve:

	   deltaV = dd/dV|V * deltaD

	   We solve for deltaD, which is:

	   deltaD = deltaV / (dd/dV|V)

	   (This is right: big derivatives mean small error.)

	   And the variance goes as the square of any multiplicative
	   factors...
	*/

	double v=orc.analogRead(port);

	double dddV=Math.abs(-Xm/(v-Xb)/(v-Xb));

	return voltageVariance/(dddV*dddV);
    }

    public double getArcAngle()
    {
	return arcAngle;
    }
}
