package maslab.orcspy;

import maslab.util.*;
import maslab.orc.*;

import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;
import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.event.*;
import java.io.*;
import java.util.*;

public class GyroPanel extends JPanel implements OrcUpdateListener
{
    public static final long serialVersionUID=1001;

    Orc orc;

    SmallButton calibratebutton=new SmallButton("Calibrate");

    SignalPlot plot=new SignalPlot();
    SignalPlot detailplot=new SignalPlot();

    Color      lightBlue=new Color(200, 200, 255);
    Color      lightRed=new Color(255, 100, 100);
    Color      darkRed=new Color(100, 0, 0);
    Color      yellow=new Color(255, 255, 0);

    static final int INDENT_PX=20;

    CompassWidget cw = new CompassWidget();

    Gyro  gyro;

    public GyroPanel(Orc orc)
    {
	super(new GridBagLayout());

	this.orc=orc;

	gyro=new Gyro(orc);

	Font smallfont=new Font("Dialog",Font.PLAIN, 9);
	setBackground(lightRed);

	GridBagConstraints gbc=new GridBagConstraints();

	///////// ROW ONE
	gbc.gridx=0; gbc.gridy=0; gbc.gridwidth=5; 
	gbc.weightx=1; gbc.weighty=1;  gbc.fill=GridBagConstraints.BOTH;

	JPanel nameholder=new JPanel(new FlowLayout(FlowLayout.LEFT));
	JLabel namelabel=new JLabel(" Gyro ");
	nameholder.add(namelabel);
	namelabel.setForeground(yellow);
	nameholder.setBackground(darkRed);
	namelabel.setFont(smallfont);
	add(nameholder, gbc);

	///////// ROW TWO, including column 0 spacer
	gbc.gridx=0; gbc.gridy=1; gbc.gridwidth=1; 
	gbc.weightx=1; gbc.weighty=1;  gbc.fill=GridBagConstraints.BOTH;
	add(Box.createHorizontalStrut(INDENT_PX), gbc);

	/* Add the detailplot */
	detailplot.setSize(new Dimension(200,15));
	detailplot.setMinimumSize(new Dimension(100,15));
	detailplot.setPreferredSize(new Dimension(200,15));

	gbc.insets=new Insets(1, 1, 1, 1);
	gbc.gridx=1; gbc.gridy=1; gbc.gridwidth=1; gbc.gridheight=2; 
	gbc.weightx=999; gbc.weighty=999;  gbc.fill=GridBagConstraints.BOTH;
	add(detailplot, gbc);

	/* Add the plot */
	plot.setSize(new Dimension(200,15));
	plot.setMinimumSize(new Dimension(100,15));
	plot.setPreferredSize(new Dimension(200,15));

	gbc.insets=new Insets(1, 4, 1, 1);
	gbc.gridx=2; gbc.gridy=1; gbc.gridwidth=1; gbc.gridheight=2;
	gbc.weightx=999; gbc.weighty=999;  gbc.fill=GridBagConstraints.BOTH;
	add(plot, gbc);

	/* add the compass */
	gbc.weightx=1; gbc.weighty=1; gbc.fill=GridBagConstraints.NONE;
	gbc.gridheight=1;
	gbc.insets=new Insets(4, 2, 0, 2);
	gbc.gridx=3; gbc.gridy=1; gbc.gridwidth=1; 
	add(cw, gbc);

	/* calibrate button */
	gbc.gridx=3; gbc.gridy=2; gbc.gridwidth=2; 
	gbc.weightx=1; gbc.weighty=1;  gbc.fill=GridBagConstraints.NONE;
	gbc.anchor=GridBagConstraints.EAST;
	add(calibratebutton, gbc);
	calibratebutton.addActionListener(new CalibrateActionListener());



	orc.addUpdateListener(this);
    }

    public void orcUpdated(Orc orc)
    {
	double theta=gyro.getRadians();
	double volts=orc.analogRead(15);

	plot.addData(Color.yellow, (theta+Math.PI)/(2*Math.PI), 
		     StringUtil.formatDouble(180.0*theta/Math.PI,2));
	
	detailplot.addData(Color.blue, 0.5, null,
			   Color.yellow, 0.5+60*(volts-gyro.gyroCorrectionPerTick*5.0/65535.0-2.5), 
			   StringUtil.formatDouble(volts,4));

	cw.setTheta(theta);
    }

    public class CalibrateActionListener implements ActionListener
    {
	public void actionPerformed(ActionEvent e)
	{
	    gyro.calibrate(4000);
	}
    }

    class CompassWidget extends JComponent
    {
	public static final long serialVersionUID=1001;	

	double theta;

	public void setTheta(double theta)
	{
	    this.theta=theta;
	    repaint();
	}
	public Dimension getMaximumSize()
	{
	    return new Dimension(100,100);
	}

	public Dimension getMinimumSize()
	{
	    return new Dimension(60,60);
	}

	public Dimension getPreferredSize()
	{
	    return getMinimumSize();
	}

	public synchronized void paint(Graphics gin)
	{
	    Graphics2D g=(Graphics2D) gin;
	    g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);

	    int width=getWidth();
	    int height=getHeight();

	    int cx=width/2, cy=height/2;

	    int diam=width<height ? width : height;

	    g.translate(cx, cy);

	    g.setColor(Color.black);
	    g.fillOval(-diam/2,-diam/2, diam, diam);

	    g.setColor(Color.darkGray);
	    g.drawLine(-diam/2, 0, diam/2, 0);
	    g.drawLine(0, -diam/2, 0, diam/2);

	    g.rotate(-theta);

	    g.setColor(Color.white);
	    int arrowlength=(int) Math.min(diam*.9, diam-2);
	    int arrowthick=(int) Math.min(diam*.1, 5);
	    int headlength=(int) Math.min(diam/3-1,20);
	    int headheight=headlength;

	    Rectangle2D.Double rect=new Rectangle2D.Double(-arrowlength/2, -arrowthick/2, 
							   arrowlength-headlength+1, arrowthick);
	    g.fill(rect);

	    g.setColor(Color.blue);
	    Polygon p=new Polygon();
	    p.addPoint(arrowlength/2, 0); // point of arrow
	    p.addPoint(arrowlength/2-headlength, headheight/2);
	    p.addPoint(arrowlength/2-headlength, -headheight/2);

	    g.fill(p);
	}

    }

    public static void main(String[] args)
    {

	Orc orc;

	try {
	    orc=new Orc();
	} catch (Exception ex) {
	    return;
	}


	GetOpt opts=new GetOpt();
	
	JFrame frame=new JFrame("GyroPanel");
	frame.getContentPane().setLayout(new BorderLayout());

	GyroPanel   gyropanel=new GyroPanel(orc);

	frame.add(gyropanel,BorderLayout.CENTER);
	frame.setVisible(true);
	frame.setSize(600,300);
    }

}

