package maslab.telemetry.botclient;

import maslab.telemetry.channel.*;
import maslab.util.*;
import maslab.telemetry.*;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import java.io.*;
import java.util.*;
import java.util.zip.*;

public class VectorPlugin extends JPanel implements Plugin, StatusListener
{
    public static String types = "vectorgraphics";
    public static Logger log = new Logger("VectorPlugin");
    
    JInternalFrame frame;
    HashMap<Integer,Drawable> items;
    HashSet<Drawable> grid;
    Color currentColor;
    
    int originX = 0;
    int originY = 0;
    int translateX = 0;
    int translateY = 0;
    double zoomfactor = 1.0;
    double zoomfactordiff = 0;

    public static final long serialVersionUID=1001;

    public VectorPlugin(JInternalFrame frame)
    {
	this.frame = frame;
	frame.getContentPane().add(this);
	frame.setSize(320,240);
	frame.setResizable(true);
	frame.setVisible(true);
	items = new HashMap<Integer, Drawable>();
	currentColor = Color.GREEN;
	Transformer t = new Transformer();
	addMouseMotionListener(t);
	addMouseListener(t);

	grid = new HashSet<Drawable>();
	int gridlimit = 1000;
	for (int x = -gridlimit; x < gridlimit+1; x+=10)
	    {
		Line l = new Line();
		l.color = Color.DARK_GRAY;
		l.x1 = x;
		l.y1 = -gridlimit;
		l.x2 = x;
		l.y2 = gridlimit;
		grid.add(l);

		l = new Line();
		l.color = Color.DARK_GRAY;
		l.y1 = x;
		l.x1 = -gridlimit;
		l.y2 = x;
		l.x2 = gridlimit;
		grid.add(l);
	    }
    }

    public boolean addChannel(String name,String type)
    {
	frame.setTitle(name);
	return true;
    }

    public void removeChannel(String name)
    {

     }

     public synchronized void messageReceived(String channel, byte[] data)
     {
	 try{
	     DataInputStream dis = new DataInputStream(new ByteArrayInputStream(data));
	     int msg = dis.readInt();

	     switch(msg)
		 {
		 case VectorChannel.LINE:
		     doLine(dis);
		     break;
		 case VectorChannel.DELETE:
		     doDelete(dis);
		     break;
		 case VectorChannel.CIRCLE:
		     doCircle(dis);
		     break;
		 default:
		     log.warn("Unknown message type: " + msg);
		 }
	 }
	 catch(IOException ioe){
	     log.log(Logger.WARN,"Unexpected IOException",ioe);
	 }


     }


     void doLine(DataInputStream dis) throws IOException
     {
	 Line l = new Line();
	 l.color = readColor(dis);
	 l.x1 = dis.readInt();
	 l.y1 = dis.readInt();
	 l.x2 = dis.readInt();
	 l.y2 = dis.readInt();
	 CRC32 crc = new CRC32();
	 crc.update(l.color.getRGB());
	 crc.update(l.x1);
	 crc.update(l.y1);
	 crc.update(l.x2);
	 crc.update(l.y2);
	 Integer key = new Integer((int)crc.getValue());
	 items.put(key,l);
	 repaint();
     }

     void doCircle(DataInputStream dis) throws IOException
     {
	 Circle l = new Circle();
	 l.color = readColor(dis);
	 l.x = dis.readInt();
	 l.y = dis.readInt();
	 l.r = dis.readInt();
	 CRC32 crc = new CRC32();
	 crc.update(l.color.getRGB());
	 crc.update(l.x);
	 crc.update(l.y);
	 crc.update(l.r);
	 Integer key = new Integer((int)crc.getValue());
	 items.put(key,l);
	 repaint();
     }


     Color readColor(DataInputStream dis) throws IOException
     {
	 int r = dis.readInt();
	 int g = dis.readInt();
	 int b = dis.readInt();
	 return new Color(r,g,b);
     }


     void doDelete(DataInputStream dis) throws IOException
     {
	 int handle = dis.readInt();
	 items.remove(new Integer(handle));
	 repaint();
     }

     public synchronized void paint(Graphics g)
     {
	 Graphics2D g2 = (Graphics2D)g;
	 g.setColor(Color.BLACK);
	 g.fillRect(0,0,getWidth(),getHeight());


	 g2.scale(zoomfactor+zoomfactordiff,zoomfactor+zoomfactordiff);
	 g.translate(originX+translateX,originY+translateY);


	 Iterator i = grid.iterator();
	 while (i.hasNext())
	     {
		 Drawable d = (Drawable)i.next();	
		 d.draw(g2);
	     }


	 i = items.values().iterator();
	 while (i.hasNext())
	     {
		 Drawable d = (Drawable)i.next();
		 d.draw(g2);
	     }

     }


     public void statusChanged(StatusEvent e)
     {
	 if (e.name.equals("connected"))
	     {
		 if (e.booleanValue == true)
		     {
			 items = new HashMapBuffer<Integer, Drawable>();
			 repaint();
		     }
	     }
     }

     interface Drawable
     {
	 public void draw(Graphics2D g);
     }

     class Line implements Drawable
     {
	 int x1,y1,x2,y2;
	 Color color;

	 public void draw(Graphics2D g)
	 {
	     g.setColor(color);
	     g.drawLine(x1,y1,x2,y2);
	 }

     }

    class Circle implements Drawable
    {
	int x,y,r;
	Color color;
	public void draw(Graphics2D g)
	{
	    g.setColor(color);
	    g.drawArc(x-r,y-r,2*r,2*r,0,360);
	}
    }

    class HashMapBuffer<K,V> extends LinkedHashMap<K,V>
    {
	static final int MAX_SIZE = 1024;
	static final long serialVersionUID = 1001;
	
	protected boolean removeEldestEntry(Map.Entry entry)
	{
	    return size() > MAX_SIZE;
	}
    }
    
    class Transformer extends MouseAdapter implements MouseMotionListener
     {
	 int dragOriginX, dragOriginY;
	 boolean zmode = false;

	 public void mousePressed(MouseEvent e)
	 {
	     dragOriginX = e.getX();
	     dragOriginY = e.getY();
	 }

	 public void mouseDragged(MouseEvent e)
	 {
	     if (zmode || (e.getModifiersEx() & e.CTRL_DOWN_MASK) != 0)
		 {
		     zoomfactordiff = (e.getX() - dragOriginX)/200.0;
		     zmode = true;
		     repaint();
		}
	    else
		{
		    translateX = (int)((e.getX() - dragOriginX)/zoomfactor);
		    translateY = (int)((e.getY() - dragOriginY)/zoomfactor);
		    repaint();
		}
	}

	public void mouseReleased(MouseEvent e)
	{
	    originX += translateX;
	    originY += translateY;
	    translateX = 0;
	    translateY = 0;

	    zmode = false;

	    zoomfactor += zoomfactordiff;
	    zoomfactordiff = 0;
	    repaint();
	}

	public void mouseMoved(MouseEvent e)
	{}

    }




}
