/*
 * LZMA compressed kernel decompressor for bcm947xx boards
 *
 * Copyright (C) 2005 by Oleg I. Vdovikin <oleg@cs.msu.su>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *
 * Please note, this was code based on the bunzip2 decompressor code
 * by Manuel Novoa III  (mjn3@codepoet.org), although the only thing left
 * is an idea and part of original vendor code
 *
 *
 * 12-Mar-2005  Mineharu Takahara <mtakahar@yahoo.com>
 *   pass actual output size to decoder (stream mode
 *   compressed input is not a requirement anymore)
 *
 * 24-Apr-2005 Oleg I. Vdovikin
 *   reordered functions using lds script, removed forward decl
 *
 * ??-Nov-2005 Mike Baker
 *   reorder the script as an lzma wrapper; do not depend on flash access
 */

#include "LzmaDecode.h"
#include "loader.h"
#ifdef AR531X
#include "../ar531x/flashlayout.h"
#endif

#ifdef BRCM
#define BCM4710_FLASH		0x9c000000
/* 2.0.X layout:
   0x60000 for CFE
   0x20000 for uhh, config?
   0x80000 flash:os (openwrt loader.elf is here, it loads an lzma compressed
           image (that expands to at most 3MB) out of a trx file that is at 0xa0000)
   0xa0000 trx file, needs to have 2 things in it, best plan is to just have a first
           thing that is small and unused, and the second is an LZMA image.  In 2.0.6 
           this is kernel+fs, where kernel < 3MB, and fs is quite big.
*/

/* for something just upgraded from netgear web thing (or from 2.0.x)
   piece1 = 0xa0000-0x180000 (size = 0xe0000)

   0x60000 for CFE
   0x20000 for uhh, config?
   0x80000 flash:os (start execution of an elf file here)
             if 2.0.x then this is the openwrt decompress.c program, which will
               open up the trx file at 0xa0000
             if 
   0xa0000 trx file containing a lzma compressed version of this program
   0xb0000 first 0x100000 of LZMA compressed kernel (1MB)
  0x1b0000 random ascii crap - for netgear web thing (we reserve a whole 0x10000 for it)
  0x1c0000 remainder of LZMA compressed kernel (at most (4MB - 1MB) more)
*/
#define BCM4710_FIRST_CHUNK		(BCM4710_FLASH + 0xb0000)
#define BCM4710_FIRST_LEN		0x100000
#define BCM4710_SECOND_CHUNK	(BCM4710_FLASH + 0x1c0000)
#define BCM4710_SECOND_LEN		0x300000
#endif


#define KSEG0			0x80000000
#define KSEG1			0xa0000000

#define KSEG1ADDR(a)		((((unsigned)(a)) & 0x1fffffffU) | KSEG1)

#define Index_Invalidate_I	0x00
#define Index_Writeback_Inv_D   0x01

#define cache_unroll(base,op)	\
	__asm__ __volatile__(		\
		".set noreorder;\n"		\
		".set mips3;\n"			\
		"cache %1, (%0);\n"		\
		".set mips0;\n"			\
		".set reorder\n"		\
		:						\
		: "r" (base),			\
		  "i" (op));

int printf(const char *fmt, ...);

static __inline__ void blast_icache(unsigned long size, unsigned long lsize)
{
	unsigned long start = KSEG0;
	unsigned long end = (start + size);

	while(start < end) {
		cache_unroll(start, Index_Invalidate_I);
		start += lsize;
	}
}

static __inline__ void blast_dcache(unsigned long size, unsigned long lsize)
{
	unsigned long start = KSEG0;
	unsigned long end = (start + size);

	while(start < end) {
		cache_unroll(start, Index_Writeback_Inv_D);
		start += lsize;
	}
}

unsigned char *data = (unsigned char*)1;

static int read_byte(void *object, const unsigned char **buffer, UInt32 *bufferSize)
{
	*bufferSize = 1;
	*buffer = data;
	++data;
	return LZMA_RESULT_OK;
}

static __inline__ unsigned char get_byte(void)
{
	const unsigned char *buffer;
	UInt32 fake;
	
	read_byte(0, &buffer, &fake);

	return *buffer;
}

/* I have no idea how big this really might need to be... 
   but 4096 was too small*/
unsigned char probsBuffer[4*1024*1024];

#if defined(BRCM)
unsigned long vmlinux_buffer[1024*1024]; // 4mb
#else
unsigned long vmlinux_buffer[PARTITION_SIZE/sizeof(unsigned long)];
#endif

unsigned char stack[8192];
unsigned long linux_args[3];

extern long bss_start[];
extern long bss_end[];

#if defined(AR531X)
static int read_part_from_flash(void *flash_addr, int dochecksum) {
	struct partition_data *pd = (struct partition_data *)flash_addr;
	int i;
	long *fromPtr;
	int wordcount;
	int crc;
	int compressedlength = pd->lzma_compressed_length;

	printf("reading flash at %p - %p... ", flash_addr,
		   (char*)flash_addr + compressedlength);

	/* do the copy word by word */
	fromPtr = (long *)pd->lzma_compressed_data;
	wordcount = (compressedlength + sizeof(long)-1) / sizeof(long);
	if (wordcount > PARTITION_SIZE/sizeof(unsigned long)) {
		printf("partition too big!");
	}
	for (i=0; i<wordcount; i++) {
		vmlinux_buffer[i] = fromPtr[i];
	}

	printf("done\nCalculating CRC... ");

	crc = update_crc(0, (unsigned char*)vmlinux_buffer,
		    pd->lzma_compressed_length);
	if (crc == pd->lzma_compressed_crc32) {
		printf("0x%x - matches\n", crc);
		return 1;
	} else {
		printf("calculated 0x%x, expected 0x%x\n", crc,
		    pd->lzma_compressed_crc32);
		return 0;
	}
}
#endif

static int do_decompress(unsigned char * lzma)
{
	unsigned int i;  /* temp value */
	unsigned int osize; /* uncompressed size */

	ILzmaInCallback callback;
	CLzmaDecoderState vs;

	callback.Read = read_byte;

	data = lzma;

	/* lzma args */
	i = get_byte();
	vs.Properties.lc = i % 9, i = i / 9;
	vs.Properties.lp = i % 5, vs.Properties.pb = i / 5;

	vs.Probs = (CProb *)probsBuffer;

	/* skip rest of the LZMA coder property */
	for (i = 0; i < 4; i++)
		get_byte();

	/* read the lower half of uncompressed size in the header */
	osize = ((unsigned int)get_byte()) +
		((unsigned int)get_byte() << 8) +
		((unsigned int)get_byte() << 16) +
		((unsigned int)get_byte() << 24);

	/* skip rest of the header (upper half of uncompressed size) */
	for (i = 0; i < 4; i++) 
		get_byte();

	/* decompress kernel */
	return LzmaDecode(&vs, &callback, (unsigned char*)KERNEL_ENTRY,
		osize, &osize);
}

extern unsigned char lzma_start[];
extern unsigned char lzma_end[];

/* should be the first function */
void entry(unsigned long icache_size, unsigned long icache_lsize, 
		unsigned long dcache_size, unsigned long dcache_lsize)
{
	long *bssp;

	/* clear the bss */
	for (bssp = bss_start; bssp != bss_end; bssp++)
		*bssp = 0;

	printf("starting stage2\n");

	if (lzma_start != lzma_end) {
		printf("decompressing embedded kernel image... ");
		do_decompress(lzma_start);
		printf("done\n");
	} else {
#if defined(BRCM)
		int i;
		long *fromPtr = (long*) BCM4710_FIRST_CHUNK;
		for (i=0; i<BCM4710_FIRST_LEN/sizeof(long); i++) {
			vmlinux_buffer[i] = fromPtr[i];
		}
		fromPtr = (long*) BCM4710_SECOND_CHUNK;
		for (; i < (BCM4710_FIRST_LEN + BCM4710_SECOND_LEN)/sizeof(long); i++) {
			vmlinux_buffer[i] = fromPtr[i-(BCM4710_FIRST_LEN/sizeof(long))];
		}
#elif defined(AR531X)
		// we are a stage2 bootloader, do the flash checks
		if (!read_part_from_flash((void*)PART1_ADDR, 1)) {
			read_part_from_flash((void*)PART2_ADDR, 0);
		}
#endif
		printf("decompressing... ");
		do_decompress((unsigned char *)vmlinux_buffer);
		printf("done\n");
	}

	blast_dcache(dcache_size, dcache_lsize);
	blast_icache(icache_size, icache_lsize);

	printf("starting linux\n");

	/* call linux kernel */
	((void (*)(unsigned long, unsigned long, unsigned long)) KERNEL_ENTRY)
		(linux_args[0], linux_args[1], linux_args[2]);
}
