/*
 *  Misc useful routines to access NIC SROM/OTP .
 *
 * Copyright 2005, Broadcom Corporation      
 * All Rights Reserved.      
 *       
 * THIS SOFTWARE IS OFFERED "AS IS", AND BROADCOM GRANTS NO WARRANTIES OF ANY      
 * KIND, EXPRESS OR IMPLIED, BY STATUTE, COMMUNICATION OR OTHERWISE. BROADCOM      
 * SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS      
 * FOR A SPECIFIC PURPOSE OR NONINFRINGEMENT CONCERNING THIS SOFTWARE.      
 * $Id$
 */

#include <typedefs.h>
#include <osl.h>
#include <bcmutils.h>
#include <bcmsrom.h>
#include <bcmdevs.h>
#include <bcmendian.h>
#include <pcicfg.h>
#include <sbutils.h>

#include <proto/ethernet.h>	/* for sprom content groking */

#define	VARS_MAX	4096	/* should be reduced */

#define WRITE_ENABLE_DELAY	500	/* 500 ms after write enable/disable toggle */
#define WRITE_WORD_DELAY	20	/* 20 ms between each word write */

static int initvars_srom_pci(void *sbh, void *curmap, char **vars, int *count);
static int sprom_read_pci(uint16 *sprom, uint wordoff, uint16 *buf, uint nwords, bool check_crc);

static int initvars_table(osl_t *osh, char *start, char *end, char **vars, uint *count);

/*
 * Initialize local vars from the right source for this platform.
 * Return 0 on success, nonzero on error.
 */
int
srom_var_init(void *sbh, uint bustype, void *curmap, osl_t *osh, char **vars, int *count)
{
	ASSERT(bustype == BUSTYPE(bustype));
	if (vars == NULL || count == NULL)
		return (0);

	switch (BUSTYPE(bustype)) {

	case PCI_BUS:
		ASSERT(curmap);	/* can not be NULL */
		return initvars_srom_pci(sbh, curmap, vars, count);

	default:
		return 0;
	}
	return (-1);
}

/* support only 16-bit word read from srom */
int
srom_read(uint bustype, void *curmap, osl_t *osh, uint byteoff, uint nbytes, uint16 *buf)
{
	void *srom;
	uint off, nw;

	ASSERT(bustype == BUSTYPE(bustype));

	/* check input - 16-bit access only */
	if (byteoff & 1 || nbytes & 1 || (byteoff + nbytes) > (SPROM_SIZE * 2))
		return 1;

	off = byteoff / 2;
	nw = nbytes / 2;

	if (BUSTYPE(bustype) == PCI_BUS) {
		if (!curmap)
			return 1;
		srom = (uchar*)curmap + PCI_BAR0_SPROM_OFFSET;
		if (sprom_read_pci(srom, off, buf, nw, FALSE))
			return 1;
	} else {
		return 1;
	}

	return 0;
}

/* support only 16-bit word write into srom */
int
srom_write(uint bustype, void *curmap, osl_t *osh, uint byteoff, uint nbytes, uint16 *buf)
{
	uint16 *srom;
	uint i, off, nw, crc_range;
	uint16 image[SPROM_SIZE], *p;
	uint8 crc;
	volatile uint32 val32;

	ASSERT(bustype == BUSTYPE(bustype));

	/* check input - 16-bit access only */
	if (byteoff & 1 || nbytes & 1 || (byteoff + nbytes) > (SPROM_SIZE * 2))
		return 1;

	crc_range = (((BUSTYPE(bustype) == SDIO_BUS)) ? SPROM_SIZE : SPROM_CRC_RANGE) * 2;

	/* if changes made inside crc cover range */
	if (byteoff < crc_range) {
		nw = (((byteoff + nbytes) > crc_range) ? byteoff + nbytes : crc_range) / 2;
		/* read data including entire first 64 words from srom */
		if (srom_read(bustype, curmap, osh, 0, nw * 2, image))
			return 1;
		/* make changes */
		bcopy((void*)buf, (void*)&image[byteoff / 2], nbytes);
		/* calculate crc */
		htol16_buf(image, crc_range);
		crc = ~hndcrc8((uint8 *)image, crc_range - 1, CRC8_INIT_VALUE);
		ltoh16_buf(image, crc_range);
		image[(crc_range / 2) - 1] = (crc << 8) | (image[(crc_range / 2) - 1] & 0xff);
		p = image;
		off = 0;
	} else {
		p = buf;
		off = byteoff / 2;
		nw = nbytes / 2;
	}

	if (BUSTYPE(bustype) == PCI_BUS) {
		srom = (uint16*)((uchar*)curmap + PCI_BAR0_SPROM_OFFSET);
		/* enable writes to the SPROM */
		val32 = OSL_PCI_READ_CONFIG(osh, PCI_SPROM_CONTROL, sizeof(uint32));
		val32 |= SPROM_WRITEEN;
		OSL_PCI_WRITE_CONFIG(osh, PCI_SPROM_CONTROL, sizeof(uint32), val32);
		bcm_mdelay(WRITE_ENABLE_DELAY);
		/* write srom */
		for (i = 0; i < nw; i++) {
			W_REG(&srom[off + i], p[i]);
			bcm_mdelay(WRITE_WORD_DELAY);
		}
		/* disable writes to the SPROM */
		OSL_PCI_WRITE_CONFIG(osh, PCI_SPROM_CONTROL, sizeof(uint32), val32 & ~SPROM_WRITEEN);
	} else {
		return 1;
	}

	bcm_mdelay(WRITE_ENABLE_DELAY);
	return 0;
}


/*
 * Read in and validate sprom.
 * Return 0 on success, nonzero on error.
 */
static int
sprom_read_pci(uint16 *sprom, uint wordoff, uint16 *buf, uint nwords, bool check_crc)
{
	int err = 0;
	uint i;

	/* read the sprom */
	for (i = 0; i < nwords; i++)
		buf[i] = R_REG(&sprom[wordoff + i]);

	if (check_crc) {
		/* fixup the endianness so crc8 will pass */
		htol16_buf(buf, nwords * 2);
		if (hndcrc8((uint8*)buf, nwords * 2, CRC8_INIT_VALUE) != CRC8_GOOD_VALUE)
			err = 1;
		/* now correct the endianness of the byte array */
		ltoh16_buf(buf, nwords * 2);
	}
	
	return err;
}	

/*
* Create variable table from memory.
* Return 0 on success, nonzero on error.
*/
static int
initvars_table(osl_t *osh, char *start, char *end, char **vars, uint *count)
{
	int c = (int)(end - start);

	/* do it only when there is more than just the null string */
	if (c > 1) {
		char *vp = MALLOC(osh, c);
		ASSERT(vp);
		if (!vp)
			return BCME_NOMEM;
		bcopy(start, vp, c);
		*vars = vp;
		*count = c;
	}
	else {
		*vars = NULL;
		*count = 0;
	}
	
	return 0;
}

/*
 * Initialize nonvolatile variable table from sprom.
 * Return 0 on success, nonzero on error.
 */
static int
initvars_srom_pci(void *sbh, void *curmap, char **vars, int *count)
{
	uint16 w, b[64];
	uint8 sromrev;
	struct ether_addr ea;
	char eabuf[32];		     
	uint32 w32;
	int woff, i;
	char *vp, *base;
	osl_t *osh = sb_osh(sbh);
	int err;

	/*
	* Apply CRC over SROM content regardless SROM is present or not,
	* and use variable <devpath>sromrev's existance in flash to decide
	* if we should return an error when CRC fails or read SROM variables
	* from flash.
	*/
	sprom_read_pci((void*)((int8*)curmap + PCI_BAR0_SPROM_OFFSET), 0, b, sizeof(b)/sizeof(b[0]), TRUE);

	/* top word of sprom contains version and crc8 */
	sromrev = b[63] & 0xff;
	/* bcm4401 sroms misprogrammed */
	if (sromrev == 0x10)
		sromrev = 1;
	
	/* srom version check */
	if (sromrev > 3)
		return (-2);

	ASSERT(vars);
	ASSERT(count);

	base = vp = MALLOC(osh, VARS_MAX);
	ASSERT(vp);
	if (!vp)
		return -2;

	vp += sprintf(vp, "sromrev=%d", sromrev);
	vp++;

	if (sromrev >= 3) {
		/* New section takes over the 3th hardware function space */

		/* Words 22+23 are 11a (mid) ofdm power offsets */
		w32 = ((uint32)b[23] << 16) | b[22];
		vp += sprintf(vp, "ofdmapo=%d", w32);
		vp++;

		/* Words 24+25 are 11a (low) ofdm power offsets */
		w32 = ((uint32)b[25] << 16) | b[24];
		vp += sprintf(vp, "ofdmalpo=%d", w32);
		vp++;

		/* Words 26+27 are 11a (high) ofdm power offsets */
		w32 = ((uint32)b[27] << 16) | b[26];
		vp += sprintf(vp, "ofdmahpo=%d", w32);
		vp++;

		/*GPIO LED Powersave duty cycle (oncount >> 24) (offcount >> 8)*/
		w32 = ((uint32)b[43] << 24) | ((uint32)b[42] << 8);
		vp += sprintf(vp, "gpiotimerval=%d", w32);

		/*GPIO LED Powersave duty cycle (oncount >> 24) (offcount >> 8)*/
		w32 = ((uint32)((unsigned char)(b[21] >> 8) & 0xFF) << 24) |  /* oncount*/
			((uint32)((unsigned char)(b[21] & 0xFF)) << 8); /* offcount */
		vp += sprintf(vp, "gpiotimerval=%d", w32);

		vp++;
	}

	if (sromrev >= 2) {
		/* New section takes over the 4th hardware function space */

		/* Word 29 is max power 11a high/low */
		w = b[29];
		vp += sprintf(vp, "pa1himaxpwr=%d", w & 0xff);
		vp++;
		vp += sprintf(vp, "pa1lomaxpwr=%d", (w >> 8) & 0xff);
		vp++;

		/* Words 30-32 set the 11alow pa settings,
		 * 33-35 are the 11ahigh ones.
		 */
		for (i = 0; i < 3; i++) {
			vp += sprintf(vp, "pa1lob%d=%d", i, b[30 + i]);
			vp++;
			vp += sprintf(vp, "pa1hib%d=%d", i, b[33 + i]);
			vp++;
		}
		w = b[59];
		if (w == 0)
			vp += sprintf(vp, "ccode=");
		else
			vp += sprintf(vp, "ccode=%c%c", (w >> 8), (w & 0xff));
		vp++;

	}

	/* parameter section of sprom starts at byte offset 72 */
	woff = 72/2;

	/* first 6 bytes are il0macaddr */
	ea.octet[0] = (b[woff] >> 8) & 0xff;
	ea.octet[1] = b[woff] & 0xff;
	ea.octet[2] = (b[woff+1] >> 8) & 0xff;
	ea.octet[3] = b[woff+1] & 0xff;
	ea.octet[4] = (b[woff+2] >> 8) & 0xff;
	ea.octet[5] = b[woff+2] & 0xff;
	woff += ETHER_ADDR_LEN/2 ;
	bcm_ether_ntoa((uchar*)&ea, eabuf);
	vp += sprintf(vp, "il0macaddr=%s", eabuf);
	vp++;

	/* next 6 bytes are et0macaddr */
	ea.octet[0] = (b[woff] >> 8) & 0xff;
	ea.octet[1] = b[woff] & 0xff;
	ea.octet[2] = (b[woff+1] >> 8) & 0xff;
	ea.octet[3] = b[woff+1] & 0xff;
	ea.octet[4] = (b[woff+2] >> 8) & 0xff;
	ea.octet[5] = b[woff+2] & 0xff;
	woff += ETHER_ADDR_LEN/2 ;
	bcm_ether_ntoa((uchar*)&ea, eabuf);
	vp += sprintf(vp, "et0macaddr=%s", eabuf);
	vp++;

	/* next 6 bytes are et1macaddr */
	ea.octet[0] = (b[woff] >> 8) & 0xff;
	ea.octet[1] = b[woff] & 0xff;
	ea.octet[2] = (b[woff+1] >> 8) & 0xff;
	ea.octet[3] = b[woff+1] & 0xff;
	ea.octet[4] = (b[woff+2] >> 8) & 0xff;
	ea.octet[5] = b[woff+2] & 0xff;
	woff += ETHER_ADDR_LEN/2 ;
	bcm_ether_ntoa((uchar*)&ea, eabuf);
	vp += sprintf(vp, "et1macaddr=%s", eabuf);
	vp++;

	/*
	 * Enet phy settings one or two singles or a dual
	 * Bits 4-0 : MII address for enet0 (0x1f for not there)
	 * Bits 9-5 : MII address for enet1 (0x1f for not there)
	 * Bit 14   : Mdio for enet0
	 * Bit 15   : Mdio for enet1
	 */
	w = b[woff];
	vp += sprintf(vp, "et0phyaddr=%d", (w & 0x1f));
	vp++;
	vp += sprintf(vp, "et1phyaddr=%d", ((w >> 5) & 0x1f));
	vp++;
	vp += sprintf(vp, "et0mdcport=%d", ((w >> 14) & 0x1));
	vp++;
	vp += sprintf(vp, "et1mdcport=%d", ((w >> 15) & 0x1));
	vp++;

	/* Word 46 has board rev, antennas 0/1 & Country code/control */
	w = b[46];
	vp += sprintf(vp, "boardrev=%d", w & 0xff);
	vp++;

	if (sromrev > 1)
		vp += sprintf(vp, "cctl=%d", (w >> 8) & 0xf);
	else
		vp += sprintf(vp, "cc=%d", (w >> 8) & 0xf);
	vp++;

	vp += sprintf(vp, "aa0=%d", (w >> 12) & 0x3);
	vp++;

	vp += sprintf(vp, "aa1=%d", (w >> 14) & 0x3);
	vp++;

	/* Words 47-49 set the (wl) pa settings */
	woff = 47;

	for (i = 0; i < 3; i++) {
		vp += sprintf(vp, "pa0b%d=%d", i, b[woff+i]);
		vp++;
		vp += sprintf(vp, "pa1b%d=%d", i, b[woff+i+6]);
		vp++;
	}

	/*
	 * Words 50-51 set the customer-configured wl led behavior.
	 * 8 bits/gpio pin.  High bit:  activehi=0, activelo=1;
	 * LED behavior values defined in wlioctl.h .
	 */
	w = b[50];
	if ((w != 0) && (w != 0xffff)) {
		/* gpio0 */
		vp += sprintf(vp, "wl0gpio0=%d", (w & 0xff));
		vp++;

		/* gpio1 */
		vp += sprintf(vp, "wl0gpio1=%d", (w >> 8) & 0xff);
		vp++;
	}
	w = b[51];
	if ((w != 0) && (w != 0xffff)) {
		/* gpio2 */
		vp += sprintf(vp, "wl0gpio2=%d", w & 0xff);
		vp++;

		/* gpio3 */
		vp += sprintf(vp, "wl0gpio3=%d", (w >> 8) & 0xff);
		vp++;
	}
	
	/* Word 52 is max power 0/1 */
	w = b[52];
	vp += sprintf(vp, "pa0maxpwr=%d", w & 0xff);
	vp++;
	vp += sprintf(vp, "pa1maxpwr=%d", (w >> 8) & 0xff);
	vp++;

	/* Word 56 is idle tssi target 0/1 */
	w = b[56];
	vp += sprintf(vp, "pa0itssit=%d", w & 0xff);
	vp++;
	vp += sprintf(vp, "pa1itssit=%d", (w >> 8) & 0xff);
	vp++;

	/* Word 57 is boardflags, if not programmed make it zero */
	w32 = (uint32)b[57];
	if (w32 == 0xffff) w32 = 0;
	if (sromrev > 1) {
		/* Word 28 is the high bits of boardflags */
		w32 |= (uint32)b[28] << 16;
	}
	vp += sprintf(vp, "boardflags=%d", w32);
	vp++;

	/* Word 58 is antenna gain 0/1 */
	w = b[58];
	vp += sprintf(vp, "ag0=%d", w & 0xff);
	vp++;

	vp += sprintf(vp, "ag1=%d", (w >> 8) & 0xff);
	vp++;

	if (sromrev == 1) {
		/* set the oem string */
		vp += sprintf(vp, "oem=%02x%02x%02x%02x%02x%02x%02x%02x",
			      ((b[59] >> 8) & 0xff), (b[59] & 0xff),
			      ((b[60] >> 8) & 0xff), (b[60] & 0xff),
			      ((b[61] >> 8) & 0xff), (b[61] & 0xff),
			      ((b[62] >> 8) & 0xff), (b[62] & 0xff));
		vp++;
	} else if (sromrev == 2) {
		/* Word 60 OFDM tx power offset from CCK level */
		/* OFDM Power Offset - opo */
		vp += sprintf(vp, "opo=%d", b[60] & 0xff);
		vp++;
	} else {
		/* Word 60: cck power offsets */
		vp += sprintf(vp, "cckpo=%d", b[60]);
		vp++;

		/* Words 61+62: 11g ofdm power offsets */
		w32 = ((uint32)b[62] << 16) | b[61];
		vp += sprintf(vp, "ofdmgpo=%d", w32);
		vp++;
	}

	/* final nullbyte terminator */
	*vp++ = '\0';

	ASSERT((vp - base) <= VARS_MAX);
	
	err = initvars_table(osh, base, vp, vars, count);
	
	MFREE(osh, base, VARS_MAX);
	return err;
}

