/*                     __                                               *\
**     ________ ___   / /  ___     Scala API                            **
**    / __/ __// _ | / /  / _ |    (c) 2003, LAMP/EPFL                  **
**  __\ \/ /__/ __ |/ /__/ __ |                                         **
** /____/\___/_/ |_/____/_/ | |                                         **
**                          |/                                          **
** $Id: SynchronizedPriorityQueue.scala,v 1.3 2004/06/04 08:51:07 zenger Exp $
\*                                                                      */

package scala.collection.mutable;


/** This class implements synchronized priority queues using a heap.
 *  The elements of the queue have to be ordered in terms of the
 *  <code>Ordered[T]</code> trait.
 *  
 *  @author  Matthias Zenger
 *  @version 1.0, 03/05/2004
 */
class SynchronizedPriorityQueue[A <% Ordered[A]] extends PriorityQueue[A] {
    
    /** Checks if the queue is empty.
     *
     *  @return true, iff there is no element in the queue.
     */
    override def isEmpty: Boolean = synchronized { super.isEmpty; }
    
    /** Inserts a single element into the priority queue.
     *
     *  @param  elem        the element to insert
     */
    override def +=(elem: A): Unit = synchronized { super.+=(elem); }
    
    /** Adds all elements provided by an <code>Iterable</code> object
     *  into the priority queue.
     *
     *  @param  iter        an iterable object
     */
    override def ++=(iter: Iterable[A]): Unit = synchronized { super.++=(iter); }
    
    /** Adds all elements provided by an iterator into the priority queue.
     *
     *  @param  it        an iterator
     */
    override def ++=(it: Iterator[A]): Unit = synchronized { super.++=(it); }
    
    /** Adds all elements to the queue.
     *
     *  @param  elems       the elements to add.
     */
    override def enqueue(elems: A*): Unit = synchronized { super.++=(elems); }
    
    /** Returns the element with the highest priority in the queue,
     *  and removes this element from the queue.
     *
     *  @return   the element with the highest priority.
     */
    override def dequeue: A = synchronized { super.dequeue; }
    
    /** Returns the element with the highest priority in the queue,
     *  or throws an error if there is no element contained in the queue.
     *  
     *  @return   the element with the highest priority.
     */
    override def max: A = synchronized { super.max; }
    
    /** Removes all elements from the queue. After this operation is completed,
     *  the queue will be empty.
     */
    override def clear: Unit = synchronized { super.clear; }
    
    /** Returns an iterator which yiels all the elements of the priority
     *  queue in descending priority order.
     *  
     *  @return  an iterator over all elements sorted in descending order.
     */
    override def elements: Iterator[A] = synchronized { super.elements; }
    
    /** Checks if two queues are structurally identical.
     *   
     *  @return true, iff both queues contain the same sequence of elements.
     */
    override def equals(that: Any): Boolean = synchronized { super.equals(that); }
    
    /** The hashCode method always yields an error, since it is not
     *  safe to use mutable queues as keys in hash tables.
     *
     *  @return never.
     */
    override def hashCode(): Int = synchronized { super.hashCode(); }
    
    /** Returns a textual representation of a queue as a string.
     *
     *  @return the string representation of this queue.
     */
    override def toString(): String = synchronized { super.toString(); }
}
