/*
 *     X Version 11 - S Graphics Device Driver
 *
 *     Copyright 1988 by the Massachusetts Institute of Technology
 *
 *     Permission to use, copy, modify, and distribute this
 *     software and its documentation for any purpose and without
 *     fee is hereby granted, provided that the above copyright
 *     notice appear in all copies and that both that copyright
 *     notice and this permission notice appear in supporting
 *     documentation, and that the name of M.I.T. not be used in
 *     advertising or publicity pertaining to distribution of the
 *     software without specific, written prior permission.
 *     M.I.T. makes no representations about the suitability of
 *     this software for any purpose.  It is provided "as is"
 *     without express or implied warranty.
 *
 *     Enhancements, improvements Copyright 1989 by Michael M. Meyer
 *     Permission to use, copy, modify and distribute is free granted
 *     to all.
 */

#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/cursorfont.h>
#include <X11/Xutil.h>
#include <signal.h>
#include <stdio.h>
#include <signal.h>
#include <math.h>

#undef Complex
#include "S.h"
#include "device.h"

/* X11 stuff */

static	Display		*display = NULL;
static	Window		rootWindow, window;
static	Cursor		cursor_readpen = None, cursor_normal = None;
static	GC		xgc;
static	int		screen;
static	int		char_color = -1;
static	int		pen_color = -1;
static	XFontStruct	*fontstruct;
static	int		blackpixel, whitepixel;

	/* Fonts */

struct SFont {
	double		cex;
	unsigned long	psize;
	char		*name;
	int		width;
	int		height;
	Font		font;
};

struct SFontHdr {
	int	nfonts[4];
	struct SFont *fonts[4];
};

static struct SFont Q0f[4] = {
	{ 0.5,  60, "6x10-Rotation0", 6, 10, None },
	{ 1.0, 120, "6x13-Rotation0", 6, 13, None },
	{ 1.5, 180, "8x13-Rotation0", 8, 13, None },
	{ 2.0, 240, "9x15-Rotation0", 9, 15, None } };

static struct SFont Q1f[4] = {
	{ 0.5,  60, "6x10-Rotation90", 10, 6, None },
	{ 1.0, 120, "6x13-Rotation90", 13, 6, None },
	{ 1.5, 180, "8x13-Rotation90", 13, 8, None },
	{ 2.0, 240, "9x15-Rotation90", 15, 9, None } };

static struct SFont Q2f[4] = {
	{ 0.5,  60, "6x10-Rotation180", 6, 10, None },
	{ 1.0, 120, "6x13-Rotation180", 6, 13, None },
	{ 1.5, 180, "8x13-Rotation180", 8, 13, None },
	{ 2.0, 240, "9x15-Rotation180", 9, 15, None } };

static struct SFont Q3f[4] = {
	{ 0.5,  60, "6x10-Rotation270", 10, 6, None },
	{ 1.0, 120, "6x13-Rotation270", 13, 6, None },
	{ 1.5, 180, "8x13-Rotation270", 13, 8, None },
	{ 2.0, 240, "9x15-Rotation270", 15, 9, None } };

struct SFontHdr SDefaultFont = { {4, 4, 4, 4}, {Q0f, Q1f, Q2f, Q3f} };

static struct SFontHdr SFontInfo;

static struct SFontHdr *SFonts;


static struct SFont *cur_font;
static int cur_font_quadrant;	/* quadrant index */
static int cur_font_cex;	/* an index computed from the real cex */

	/* Colors */

#define MAX_COLORS	128

static char *color_names[MAX_COLORS];
static unsigned long color_cookies[MAX_COLORS];
static int n_colors = 2;

#define background_color	color_cookies[0]
#define foreground_color	color_cookies[1]

	/* Definitions for Shading Patterns */

#define Bitmap_width 16
#define Bitmap_height 16
typedef char Shade16[32];
static Shade16 Shades[] = {
#include "x11texture.h"
};

#define NSHADES		sizeof(Shades)/sizeof(Shade16)

Pixmap ShadePixmap[MAX_COLORS+1];

/* End X11 stuff */

#define DNAME x11
#include "disp_list.h"

static void readpen();
static void ColorShade();
static void FreeShade();
static void SetOp();
static int processOrdinaryEvent();
static void NewColor();
static void GetColors();
static void GetTextures();
static void BWColors();
static void GetNumColors();
static char *dupstr();
static int quadrant();
static double reduce();
static void rotate();
static void drawchar();
static void revstring();
static int ExpandIndex();
static void FindFonts();
static void DefaultFont();
static void FindAFont();

/*
 *     Tunable Defaults
 */

#define WINDOW_X_ORIGIN		0		/* default window upper left */
#define WINDOW_Y_ORIGIN		0
#define WINDOW_WIDTH	400
#define WINDOW_HEIGHT	400



static int windowWidth = WINDOW_WIDTH, windowHeight = WINDOW_HEIGHT;
static int trueWidth = WINDOW_WIDTH, trueHeight = WINDOW_HEIGHT;
static int ask;
static double pixelAspectRatio;
static double xMapFudge = 1.0;
static double yMapFudge = 1.0;
#define xUserToRaster(i)       xMapFudge*((i)*am(37)+am(36)+0.5)
#define yUserToRaster(i)       yMapFudge*(windowHeight-((i)*am(39)+am(38)+0.5))


vector *DNAME(iask,xgeom,xreverse,xdisplay,xcloseDisplay)
	long *iask;
	char **xgeom;		/* The X geometry spec */
	long *xreverse;
	char **xdisplay;	/* Name of display device */
	long *xcloseDisplay;	/* Force close of display first */
{
	device *d, *new_device();
	int alarmCatcher();
	double pixelWidth(), pixelHeight();
	int i;

	ask = *iask;

	signal(SIGALRM,alarmCatcher);
	set_device(new_device(&d_DNAME, 0L)->which);
	ALARM_NO;
	if(display == NULL || *xcloseDisplay) {
		if( *xcloseDisplay )
			closeX11(0);
		openX11(xgeom,xreverse,xdisplay);
	} else {
		XMapWindow(display,window);
	}

	for( i=0 ; i<40 ; i++ ) am(i) = 0.0;

	am(20) = 6.0;		/* character size in rasters (A lie, */
	am(21) = 13.0;		/* but it works) */

	am(22) = 0.0;		/* display extents */
	am(23) = windowWidth;
	am(24) = 0.0;
	am(25) = windowHeight;

	am(26) = 0.5;		/* character addressing offsets */
	am(27) = 0.5;

	am(28) = pixelWidth();	/* raster size in inches */
	am(29) = pixelHeight();

	am(30) = X11;

	am(1) = 1.0;		/* variable size characters */
	am(31) = 1.0;		/* allow character rotation */

	F77_SUB(defltz)();
	ALARM_YES;
}


static void
setfont(q, x)
	int q;
	int x;
{
	if( q < 0 || q > 3 ) q = 0;
	if( x < 0 ) x = 0;
	else if(x >= SFonts->nfonts[q] ) x = SFonts->nfonts[q] - 1;
	if( cur_font != NULL && cur_font_quadrant == q && cur_font_cex == x )
/**/		return;
	cur_font = &SFonts->fonts[q][x];
	if(cur_font->font == None) {
		cur_font->font = XLoadFont(display,cur_font->name);
	}
	cur_font_quadrant = q;
	cur_font_cex = x;
	XSetFont(display,xgc,cur_font->font);
}    

/* Different Line Styles */

#define NUMD 9
#define D31   2
#define D3431 4
#define D44   2
#define D47   2
#define D123  3
#define D42   2
#define D1234 4
#define D414  3
#define D444  3

int dash_list_length[NUMD] = {
	D31, D3431, D44, D47, D123, D42, D1234, D414, D444};

unsigned char d31[D31] = {3,1} ;
unsigned char d3431[D3431] = {3,4,3,1};
unsigned char d44[D44] = {4,4};
unsigned char d47[D47] = {4,7};
unsigned char d123[D123] = {1,2,3};
unsigned char d42[D42] = {4,2};
unsigned char d1234[D1234] = {1,2,3,4};
unsigned char d414[D414] = {4,1,4};
unsigned char d444[D444] = {4,4,4};

unsigned char *dash_list[NUMD] = {
	d31, d3431, d44, d47, d123, d42, d1234, d414, d444};


int dash_offset= 0;


static void USRamodes(ii)
	int ii;
{
	int j;
	switch(ii) {
	case 8:
		j = ((int)am(8)-1)%10;
		if(j == 0) {
			XSetLineAttributes(display,xgc,0,LineSolid,
				CapButt,JoinMiter);
		} else {
			XSetLineAttributes(display,xgc,0,LineOnOffDash,
				CapButt,JoinMiter);
			XSetDashes(display, xgc, dash_offset, dash_list[j-1],
				dash_list_length[j-1]);
		}
		break;
	case 10:		/* col= */
		j = am(10);
		NewColor(j);
		break;
	case 18:		/* character expansion */
		setfont(cur_font_quadrant, ExpandIndex(am(18)));
		break;
	case 14:		/* character rotation */
		setfont(quadrant(am(14)), cur_font_cex);
		break;
#if 0
	case 9:
		fprintf(stderr,"am(9) is %d\n", (int)am(9));/* line width */
		break;
	case 15:
		fprintf(stderr,"am(15) is %d\n", (int)am(15));/* plot char */
		break;
#endif
	}
}

#define ROTATE	(cur_font_quadrant&1)
#define REVERSE	(cur_font_quadrant&2)

static void USRtext(x,y,str,adj)
float *x;
float *y;
char *str;
float adj;
{
	char *s;
	double srt, nm1;
	int i, ix, iy, ix0, iy0, ix1, iy1, inc, n, length;

	processEvents();
	updateClipRegion();
	n = strlen(str);
	ix = xUserToRaster(*x);
	iy = yUserToRaster(*y);
	srt = reduce(am(14));
	if( srt >= -0.5 && srt <= 0.5 ) srt = 0.0;
	if( srt <= -179.5 || srt >= 179.5 ) srt = 180.;

	if( ! ROTATE ) {
		length = n*cur_font->width;
		ix0 = ix-adj*length;
		ix1 = ix0+length;
		iy0 = iy;
		iy1 = iy;
		rotate(&ix0,&iy0,&ix1,&iy1,ix,iy,srt);
		iy0 += cur_font->height/2;
		iy1 += cur_font->height/2;
	}
	else {
		length = n*cur_font->height;
		iy0 = iy-adj*length;
		iy1 = iy0+length;
		ix0 = ix;
		ix1 = ix;
		rotate(&ix0,&iy0,&ix1,&iy1,ix,iy,srt-90.0);
		ix0 -= cur_font->width/2;
		ix1 -= cur_font->width/2;
	}
	/* Something is not quite right here.  Try:
		text(x,y,"A")
		text(x,y,"A",srt=180)	# OK so far
		text(x,y,"A",srt=90)	# huh?
	 */

	if (srt == 0.0){
		XDrawString(display,window,xgc,ix0,iy0,str,n);
	}
	else if (srt == 180.0 ) {
		/* arcane knowledge of display.h => we have write
		 * permission on the string
		 */
		revstring(str);
		XDrawString(display,window,xgc,ix1,iy1,str,n);
		revstring(str);
	}
	else if( n<=1 ) {
		drawchar(ix,iy,str[0]);
	}
	else {
		nm1 = n-1;
		for( i=0 ; i<n ; i++ ) {
			ix = ix0+(ix1-ix0)*(i/nm1);
			iy = iy0+(iy1-iy0)*(i/nm1);
			drawchar(ix,iy,*str++);
		}
	}
	flushX11();
}

static void USRmrks(xx, yy, n) /* THis is really the points command.  Bates
calls it marks */
	float *xx, *yy;
	int n;
{
	pntsX11(xx,yy,n);
}

static void USRlins(x,y,n)
float *x;
float *y;
int n;
{
	XPoint *points;
	int i, col, lty;

	processEvents();
	updateClipRegion();
	col = am(10);
	NewColor(col);	/* not necessary, but prudent */
	lty = am(8);
	if( n<=1 )
		return;
	points = (XPoint *)emalloc(n*sizeof(XPoint));
	for( i=0 ; i<n ; i++ ) {
		points[i].x = xUserToRaster(x[i]);
		points[i].y = yUserToRaster(y[i]);
	}
	XDrawLines(display,window,xgc,points,n,CoordModeOrigin);
	free(points);
	flushX11();
}

static void USRsegs(x0,y0,x1,y1,n)
float *x0;
float *y0;
float *x1;
float *y1;
int n;
{
	int col, lty, i, ix0, iy0, ix1, iy1;

	processEvents();
	updateClipRegion();
	col = am(10);
	NewColor(col);	/* not necessary, but prudent */
	lty = am(8);
	if( n<1 )
		return;
	for( i=0 ; i<n ; i++ ) {
		ix0 = xUserToRaster(x0[i]);
		iy0 = yUserToRaster(y0[i]);
		ix1 = xUserToRaster(x1[i]);
		iy1 = yUserToRaster(y1[i]);
		XDrawLine(display,window,xgc,ix0,iy0,ix1,iy1);
	}
	flushX11();
}

static void USRpoly(x,y,n)
float *x;
float *y;
int n;
{
	XPoint *points;
	int i, col, lty;

	processEvents();
	updateClipRegion();
	col = am(10);
	NewColor(col);	/* not necessary, but prudent */
	lty = am(8);
	if( n<=1 )
		return;

	points = (XPoint *)emalloc(n*sizeof(XPoint));
	for( i=0 ; i<n ; i++ ) {
		points[i].x = xUserToRaster(x[i]);
		points[i].y = yUserToRaster(y[i]);
	}

	XFillPolygon(display,window,xgc,points,n,Nonconvex,
			CoordModeOrigin);

	XDrawLines(display,window,xgc,points,n,CoordModeOrigin);
	XDrawLine(display,window,xgc,
		points[n-1].x,points[n-1].y,
		points[0].x,points[0].y);
	free(points);
	flushX11();
}

static void
NewColor(col)
	int col;
{
	if( col == pen_color )
/**/		return;
	if( col < 0 || col >= n_colors )
		col = 1;

	pen_color = col;
	XSetForeground(display,xgc,color_cookies[col]);
	if( ShadePixmap[col] == None ) {
		XSetFillStyle(display,xgc,FillSolid);
	} else {
		XSetTile(display,xgc,ShadePixmap[col]);
		XSetFillStyle(display,xgc,FillTiled);
	}
}


/* OTHER ROUTINES CALLED BY S
 *
 * flush() brings the output up to date
 * clear() prints the current page
 * wrap() wraps up the job
 * signalled() fixes up if interrupt or other signal occurs
 */

static void USRflush()
{
	flushX11();
}

static void USRclear()
{
	if( ask ) {
		Eprintf("GO? ");
		while( getchar()!='\n' );
	}
	clearX11();
}

static void USRamReset()
{
	grzReset();
}

static void USRwrap()
{
	closeX11(1);
}

static void USRsignalled()
{
	fprintf(stderr,"Signalled\n");
}


static void USRinput(x, y, n, nmax)
	float x[], y[];
	long *n, *nmax;
{
	readpen(x,y,n,nmax);
}

static vector *USRmenu(ent,arglist)
	vector *ent, *arglist;
{
	fprintf(stderr,"Menu\n");
}

static void USRhook()
{
	fprintf(stderr,"Hook\n");
}

static void USRlength()
{
	fprintf(stderr,"Length\n");
}
/*
 *     X VERSION 11 ROUTINES
 *
 */

static openX11(xgeom,xreverse,xdisplay)
	char **xgeom;
	int *xreverse;
	char **xdisplay;
{
	char *option;
	int xrev2;
	int window_x=WINDOW_X_ORIGIN,
	    window_y=WINDOW_Y_ORIGIN,
	    window_width=WINDOW_WIDTH,
	    window_height=WINDOW_HEIGHT;
	XSizeHints		xsize_hints;
	XSetWindowAttributes	attributes;
	XGCValues		values;
	XEvent			event;
	Visual			*visual;
	unsigned long		blackpixel, whitepixel;
	int screen_depth;
	int i, screen_width;

		/* open the default display */

	option = *xdisplay;
	if(strcmp(option,"")==0)
		option = NULL;
	if((display=XOpenDisplay(option))==NULL) {
		Recover("Couldn't open X11 graphics window",S_void);
	}

	/*
	 *      Default Screen and Root Window
	 */

	screen = XDefaultScreen(display);
	rootWindow = XDefaultRootWindow(display);
	screen_width = DisplayWidth(display,screen);
	screen_depth = DisplayPlanes(display,screen);
	visual = DefaultVisual(display,screen);

	xrev2 = *xreverse;
	if (*xreverse == -1) {
		option = XGetDefault(display,"S","ReverseVideo");
		if( option == NULL )
			option = XGetDefault(display,"S","reversevideo");
		if( option == NULL )
			option = "missing!!";
		xrev2 = ((strcmp(option,"on")==0) || (strcmp(option,"yes")==0)
			|| (strcmp(option,"1")==0) || (strcmp(option,"true") == 0));
	}


	option = *xgeom;
	if ( option == NULL || strlen(option) == 0) {
		option=XGetDefault(display,"S","geometry");
		if(option == NULL)
			option="";
	}

	if ( strlen(option) != 0)
		XParseGeometry(option,&window_x,&window_y,&window_width,&window_height);

	/*
	 *	Try to figure out colors and textures.  Begin by finding
	 *	out what white and black are, and assigning them to
	 *	foreground and background.
	 */

	blackpixel = BlackPixel(display,screen);
	whitepixel = WhitePixel(display,screen);

	if (xrev2){
		background_color = blackpixel;
		foreground_color = whitepixel;
	} else {
		background_color = whitepixel;
		foreground_color = blackpixel;
	}
	/* Find out how many "colors" the X options database says we have.
	 *
	 * Based on the screen depth and whatever "visual" says about the
	 * screen, set up the mapping between the integers 0:n_colors-1
	 * and pixel values.  We have to wait to figure out textures until
	 * we get the window open.
	 *
	 * Also, decide on the logical op to use to paint things on the
	 * screen.  This is easy in black and white - just use Xand or Xor,
	 * depending on the background color.  In color, it's not so clear
	 * what to do.  The "right" thing to do is Xcopy.  This can wipe
	 * out pieces of lines and text when there is partial overlap.
	 * Using rotated fonts for text would solve the text problem; it's
	 * not clear what to do about the line problem.
	 *
	 * Looking at visual->class is a hack recommended by the X
	 * documentation - doing this right would be painful.
	 */

	GetNumColors();
	if( screen_depth == 1) {
			/* Monochrome - easy */
		SetOp(xrev2, whitepixel, blackpixel, &values);
		BWColors();
	} else
	switch(visual->class) {		/* THIS MAY BREAK SOMEDAY */
	case PseudoColor:
	case StaticColor:
	case DirectColor:
	case TrueColor:
		values.function = GXcopy;
		GetColors();
		break;
	case GrayScale:
	case StaticGray:
		SetOp(xrev2, whitepixel, blackpixel, &values);
		GetColors();
		break;
	default:
		/* HUH??? */
		values.function = GXcopy;
		BWColors();
		break;
	}


	/*
	 *	Try to create a simple window.
	 *	Want to know about exposures
	 *	and window-resizes and locations
	 */

	attributes.event_mask = ButtonPressMask
				| ExposureMask
				| StructureNotifyMask;
	attributes.background_pixel = background_color;
	attributes.border_pixel = blackpixel;

	if((window = XCreateWindow(display,rootWindow,
		window_x,window_y,
		window_width, window_height, 1,
		DefaultDepth(display,screen),
		InputOutput, visual,
		CWEventMask | CWBackPixel | CWBorderPixel, &attributes ))==0) {
			Recover("Couldn't Create X11 graphics window",S_void);
	}
	XChangeProperty(display,window,XA_WM_NAME,XA_STRING,
		8,PropModeReplace,"S Graphics",10);

	GetTextures();

	xsize_hints.width=window_width;
	xsize_hints.height=window_height;
	xsize_hints.x=window_x;
	xsize_hints.y=window_y;
	xsize_hints.flags=(PSize | PPosition);
	XSetNormalHints(display,window,&xsize_hints);

		/* Need two cursors */
	cursor_readpen = XCreateFontCursor(display,XC_crosshair);
	cursor_normal = XCreateFontCursor(display,XC_pencil);
	XDefineCursor(display,window,cursor_normal);

	values.plane_mask = AllPlanes;
	values.foreground = foreground_color;
	values.background = background_color;

		/* Get Graphics Context and Reset State */

	xgc = XCreateGC( display,
			window,
			GCFunction|GCPlaneMask|GCForeground|GCBackground,
			&values);

	option = XGetDefault(display,"S","Font");
	if( option == NULL )
		option = XGetDefault(display,"S","font");
	if( option == NULL )
		SFonts = &SDefaultFont;
	else
		FindFonts(option);

	cur_font_quadrant = 0;
	cur_font_cex = ExpandIndex(1.0);
	setfont(cur_font_quadrant, cur_font_cex);

		/* Map The Window */

	XMapWindow(display, window);

	XNextEvent(display,&event);
	if( event.xany.type==Expose ) {
		while( event.xexpose.count )
			XNextEvent(display,&event);
	}
	else if( event.type==ConfigureNotify ) {
		trueWidth = windowWidth = event.xconfigure.width;
		trueHeight = windowHeight = event.xconfigure.height;
	}

	XSelectInput(display,window,
		ExposureMask|ButtonPressMask|StructureNotifyMask);
}

static void
ColorShade(i, texture)
	int i;
	int texture;
{
	int prev;
	if( i < 0 || i >= n_colors )
/**/		return;
	if( texture < 0 || texture >= NSHADES )
/**/		return;

	if( ShadePixmap[i] != None )
		FreeShade(i);

	ShadePixmap[i] = XCreatePixmapFromBitmapData(display,window,
				Shades[texture],Bitmap_width,Bitmap_height,
				color_cookies[i], background_color,
				DefaultDepth(display,screen));

}

static void
FreeShade(i)
	int i;
{
	if( i < 0 || i >= n_colors || ShadePixmap[i] == None )
/**/		return;
	XFreePixmap(display, ShadePixmap[i]);
	ShadePixmap[i] = None;
}

static void
SetOp(rev, whitepixel, blackpixel, v)
	int rev;
	unsigned long whitepixel, blackpixel;
	XGCValues *v;
{
	/* AllPlanes instead of 1 here?? 1/0 is correct for DEC hardware */

	if( blackpixel == 1 && whitepixel == 0 ) {
		if( rev )
			v->function = GXand;
		else
			v->function = GXor;
	} else
	if( blackpixel == 0 && whitepixel == 1 ) {
		if( rev )
			v->function = GXor;
		else
			v->function = GXand;
	} else {
		/* HUH ??? */
		v->function = GXcopy;
	}
}

#define MM_PER_INCH 25.4

static double pixelWidth()
{
	double width, widthMM;
	width = DisplayWidth(display,screen);
	widthMM = DisplayWidthMM(display,screen);
	return (widthMM/width)/MM_PER_INCH;
}

static double pixelHeight()
{
	double height, heightMM;
	height = DisplayHeight(display,screen);
	heightMM = DisplayHeightMM(display,screen);
	return (heightMM/height)/MM_PER_INCH;
}

static void
rotate(x0, y0, x1, y1, xc, yc, theta)
	int *x0,*y0,*x1,*y1,xc,yc;
	double theta;
{
	double cos(), sin(), sintheta, costheta;
	int x, y;

	costheta = cos( PI * theta / 180.0 );
	sintheta = sin( PI * theta / 180.0 );
	x = *x0-xc;
	y = *y0-yc;
	*x0 = xc+x*costheta-y*sintheta;
	*y0 = yc-y*costheta-x*sintheta;	/* y runs backwards */
	x = *x1-xc;
	y = *y1-yc;
	*x1 = xc+x*costheta-y*sintheta;
	*y1 = yc-y*costheta-x*sintheta; /* y runs backwards */
}

/*
 * Zero-origin quadrant
 *
 *	|
 *   1  |   0
 *	|
 *------|------
 *	|
 *   2  |   3
 *	|
 */
static int
quadrant(srt)
	double srt;
{
	int isrt = reduce(srt);
	if( -80<=isrt && isrt<=80 )
		return 0;
	if( 80<isrt && isrt<=100 )
		return 1;
	if( -100<isrt && isrt<-80 )
		return 3;
	return 2;
}

static double
reduce(srt)
	double srt;
{
	while(srt<=-180.) srt+=360.;
	while(srt>180.) srt-=360.;
	return srt;
}

static pntsX11(x,y,n)
float *x;
float *y;
int n;
{
	int col, pch, i, ix, iy;
	long lpch, ln;
	XPoint *points, *p;

	processEvents();
	updateClipRegion();
	col = am(10);

	pch = (long) am(15);
	if( pch<32 ) {
		lpch = (long) pch;
		ln = (long)n;
		F77_SUB(dmarkz)(x,y,&ln,&lpch);
	}
	/* use pixel if pch==46 */
	else if( pch==46 ) {
		points=(XPoint *) calloc((unsigned)2*n, sizeof(XPoint));
		if(points==NULL) {
			Recover("Cannot allocate memory for points",S_void);
		}
		p = points;
		for( i=0 ; i< n ; i+=1, p+=2 ) {
			ix = xUserToRaster(x[i]);
			iy = yUserToRaster(y[i]);
			p->x = ix;
			p->y = iy;
			(p+1)->x = ix;
			(p+1)->y = iy+1;
		}
		XDrawPoints(display,window,xgc,points,2*n,CoordModeOrigin);
		free(points); /* return memory */
	}
	else {
		for( i=0 ; i<n ; i++ ) {
			ix = xUserToRaster(x[i]);
			iy = yUserToRaster(y[i]);
			iy += cur_font->height/2;
			ix -= cur_font->width/2;
			drawchar(ix,iy,pch);
		}
	}
	flushX11();
}

static void
drawchar(ix,iy,ich)
	int ix;
	int iy;
	int ich;
{
	char s[2];
	s[0]=(char)ich;
	s[1]=(char)0;
	XDrawString(display,window,xgc,ix,iy,s,1);
}


static clearX11() {
	XClearWindow(display,window);
	flushX11();
}


static flushX11() {
	XSync(display,0);
}


static
closeX11(unmap_only)
	int unmap_only;
{
		/* Try to clean up everything before we exit */
	int i, j;

	ALARM_NO;
	if( display == NULL )
/**/		return;
	if( unmap_only ) {
		XUnmapWindow(display,window);
		XSync(display,0);
	} else {
		for(i = 0; i < n_colors; i++ ) {
			if( ShadePixmap[i] != None ) {
				FreeShade(i);
			}
			if( color_names[i] != NULL ) {
				free(color_names[i]);
				color_names[i] = NULL;
			}
		}
		for(i=0; i<4; i++ ) {
			for(j=0; j<SFonts->nfonts[i]; j++) {
				if( SFonts->fonts[i][j].font != None ) {
					XUnloadFont(display, SFonts->fonts[i][j].font);
					SFonts->fonts[i][j].font = None;
				}
			}
			if( SFontInfo.fonts[i] != NULL ) {
				for(j=0; j<SFonts->nfonts[i]; j++) {
					free(SFontInfo.fonts[i][j].name);
					SFontInfo.fonts[i][j].name = NULL;
				}
				free((char *) SFontInfo.fonts[i]);
				SFontInfo.fonts[i] = NULL;
				SFontInfo.nfonts[i] = 0;
			}
		}
		cur_font = NULL;
		XFreeCursor(display,cursor_readpen);
		XFreeCursor(display,cursor_normal);
		cursor_readpen = None;
		cursor_normal = None;
		XFreeGC(display,xgc);
		XCloseDisplay(display);
		display = NULL;
		pen_color = -1;
		char_color = -1;
		n_colors = 2;
	}
}


static void readpen(x,y,n,nxy)
float *x, *y;
long *n, *nxy;
{
	int	i;
	int	done;
	XEvent	event;

	/* Discard pending events (in case user fidgets with mouse).
	 * This code used to raise the window, but this sometimes resulted in
	 * the window not being re-drawn properly (a race condition?).  If
	 * we force the user to raise the window, the problem seems to go
	 * away (sigh).
	 */

	/*XRaiseWindow(display,window);*/
	processEvents();
	XSync(display,1);

	XDefineCursor(display,window,cursor_readpen);  /* Give a crosshair cursor */
	for( i=0, done=0 ; i<*nxy && !done ; ) {
		XNextEvent(display,&event);
		switch(event.xany.type) {
		case Expose:
		case ConfigureNotify:
				/* Oops - user may have gone away & done
				 * something else.  Make no attempt to
				 * optimize redraws.
				 */
			if( processOrdinaryEvent(&event))
				redrawDisplayList();
			break;
		case ButtonPress:
			switch(event.xbutton.button) {
			case Button1:
			case Button2:
				x[i] = ((event.xbutton.x-am(36))/am(37))/xMapFudge;
				y[i] = ( (windowHeight-event.xbutton.y-am(38))
					/am(39) ) / yMapFudge;
				fprintf(stderr,"\07");
				fflush(stderr);	/* BSD line buffers this */
				i++;
				break;
			default:
				done = 1;
				break;
			}
		}
	}

	*n = i;
	XDefineCursor(display,window,cursor_normal);  /*  Reset the cursor */
	flushX11();
}






long F77_COM(reshap);

static grzReset()
{
	long x0, x1, y0, y1;
	extern F77_SUB(zzdevz)();
	am(22) = x0 = 0;
	am(23) = x1 = windowWidth = trueWidth;
	am(24) = y0 = 0;
	am(25) = y1 = windowHeight = trueHeight;
	xMapFudge = 1.0;
	yMapFudge = 1.0;
	F77_COM(reshap) = 1;
	F77_SUB(zzdevz)(&x0,&x1,&y0,&y1);
	updateClipRegion();
}

static int alarmCatcher()
{
	ALARM_NO;
#ifndef Berkeley
	signal(SIGALRM,alarmCatcher);
#endif
	processEvents();
	ALARM_YES;
}

static processEvents()
{
	int flag=0;
	XEvent	event;

	while( XPending(display) ) {
		XNextEvent(display,&event);
		flag |= processOrdinaryEvent(&event);
	}
	if( flag )
		redrawDisplayList();
}

static int
processOrdinaryEvent(event)
	XEvent *event;
{
	int flag = 0;
	switch(event->xany.type) {
	case Expose:
		if( event->xexpose.count == 0 )
			flag = 1;
		break;

	case ConfigureNotify:
		trueWidth = event->xconfigure.width;
		trueHeight = event->xconfigure.height;
		xMapFudge = ((double)trueWidth)/((double)windowWidth);
		yMapFudge = ((double)trueHeight)/((double)windowHeight);
		updateClipRegion();
		flag = 1;
		break;
	}
	return flag;
}

static updateClipRegion()
{
	XRectangle cliprect;
	double drw, drh;

	if( am(65)>0.5 ) {
		cliprect.x = 0;
		cliprect.width = trueWidth;
		cliprect.y = 0;
		cliprect.height = trueHeight;
	}
	else {
		cliprect.x = xUserToRaster(am(61));
		cliprect.width = xUserToRaster(am(62))-cliprect.x+1.0;
		cliprect.y = yUserToRaster(am(64));
		cliprect.height = yUserToRaster(am(63))-cliprect.y+1.0;
	}
	XSetClipRectangles(display,xgc,0,0,&cliprect,1,Unsorted);
}

static void
BWColors()
{
	int i;

	for(i=2; i<n_colors; i++ )
		color_cookies[i] = foreground_color;
}

static void
GetColors()
{
	char *option;
	int i;
	char buf[256];
	XColor xcol;
	Colormap cmap;

	cmap = DefaultColormap(display, screen);

	for( i = 0; i < n_colors; i++ ) {
		sprintf(buf, "color%d", i);
		option = XGetDefault(display, "S", buf);
		if( option == NULL ) {
			if( i > 1 ) {
				sprintf(buf,"no color given for %d", i);
				Warning(buf,NULL_ENTRY);
			}
	/**/		continue;
		}
		color_names[i] = dupstr(option);
		if( !XParseColor(display, cmap, option, &xcol) ) {
			/* need a warning? */
			if( i > 1 ) {
				sprintf(buf, "color %d was invalid", i);
				Warning(buf, NULL_ENTRY);
				color_cookies[i] = foreground_color;
			}
	/**/		continue;
		}
		if( !XAllocColor(display, cmap, &xcol) ) {
			sprintf(buf, "could not allocate color %d", i);
			Warning(buf, NULL_ENTRY);
		}
		color_cookies[i] = xcol.pixel;
	}
}

static void
GetTextures()
{
	char *option;
	int i;
	char buf[256];
	int texture;

	for( i = 0; i < n_colors; i++ ) {
		sprintf(buf, "texture%d", i);
		option = XGetDefault(display, "S", buf);
		if( option == NULL ) {
			FreeShade(i);
	/**/		continue;
		}
		texture = atoi(option);
		if( texture < 0 || texture >= NSHADES ) {
			sprintf(buf,"texture %d out of range", i);
			Warning(buf, NULL_ENTRY);
			FreeShade(i);
	/**/		continue;
		}
		ColorShade(i, texture);
	}
}

static void
GetNumColors()
{
	char *option;
	char buf[256];

	option = XGetDefault(display, "S", "nColors");
	if( option == NULL )
		option = XGetDefault(display, "S", "ncolors");
	if( option == NULL ) {
		n_colors = 2;
/**/		return;
	}

	n_colors = atoi(option);
	if( n_colors <= 1 ) {
		Warning("ncolors must be > 1", NULL_ENTRY);
		n_colors = 2;
/**/		return;
	} else
	if( n_colors >= MAX_COLORS ) {
		sprintf(buf,"ncolors must be < %d", MAX_COLORS);
		Warning(buf,NULL_ENTRY);
		n_colors = MAX_COLORS-1;
	}
}

static char *
dupstr(s)
	char *s;
{
	char *t;
	t = emalloc(strlen(s)+1);
	(void)strcpy(t, s);
	return t;
}

static void
revstring(s)
	char *s;
{
	char *t;
	int c, l;
	l = strlen(s);
	t = s + l;
	l /= 2;
	for(; l > 0; --l) {
		c = *--t;
		*t = *s;
		*s++ = c;
	}
}

static int
ExpandIndex(cex)
	float cex;
{
	int x;
	for(x=0; x<SFonts->nfonts[cur_font_quadrant]; x++)
		if(cex < SFonts->fonts[cur_font_quadrant][x].cex)
			return x;
	return SFonts->nfonts[cur_font_quadrant] - 1;
}

static void
FindFonts(fname)
	char *fname;
{
	int q;

	for(q = 0; q < 4; q++ ) {
		FindAFont(fname, q, &SFontInfo);
	}
	SFonts = &SFontInfo;
}

static int fcmp(p1, p2)
	struct SFont *p1, *p2;
{
	return p1->psize - p2->psize;
}

static void
FindAFont(fname, quadrant, fhdr)
	char *fname;
	int quadrant;
	struct SFontHdr *fhdr;
{
	char *pat, *q;
	int nfound;
	char **flist;
	struct SFont *list0;
	int i;
	XFontStruct *fp;

	pat = "*%s*-fixed-rotation%d";
	q = malloc(strlen(fname)+strlen(pat)+4);
	if( q == NULL )
		S_terminate("Out of memory in X11 driver");
	sprintf(q, pat, fname, 90*quadrant);
	flist = XListFonts(display, q, 15, &nfound);
	free((char *)q);

	if( flist == NULL || nfound == 0 ) {
		DefaultFont("no such font");
/**/		return;
	}
	list0 = (struct SFont *)calloc(nfound, sizeof(struct SFont));
	if( list0 == NULL )
		S_terminate("Out of memory in X11 driver");
	for(i=0; i<nfound; i++) {
		list0[i].name = dupstr(flist[i]);
		list0[i].font = XLoadFont(display, flist[i]);
		fp = XQueryFont(display, list0[i].font);
		if( fp == NULL ) {
			DefaultFont("no info for font (botch?)");
/**/			return;
		}
		list0[i].height = fp->ascent + fp->descent;
		list0[i].width = fp->max_bounds.rbearing -
			fp->min_bounds.lbearing;
		if(!XGetFontProperty(fp, XA_POINT_SIZE, &list0[i].psize)) {

			DefaultFont("No point size for font");
/**/			return;
		}
		list0[i].cex = (double)list0[i].psize / 120.;

		XFreeFontInfo((char **)NULL, fp, 1);
	}
	XFreeFontNames(flist);
	qsort((char *)list0, nfound, sizeof(struct SFont), fcmp);

	fhdr->fonts[quadrant] = list0;
	fhdr->nfonts[quadrant] = nfound;
}

static void
DefaultFont(s)
	char *s;
{
	fprintf(stderr,"X11 driver:%s\nDefault font used\n", s);
	SFonts = &SDefaultFont;
}
