/*
 * frag.c
 *
 * Keep a cache of recent "fragment leaders" to allow us
 * to do the right thing with later fragments
 *
 * Modification history:
 *
 * August 1993		Jeffrey Mogul
 *	data type cleanup
 *
 * June(?) 1993		David Cherkus
 *	Ported from screend to statspy.
 *
 * 28 December 1988	Jeffrey Mogul/DECWRL
 *	Created.
 *
 */

/*
 * Portions Copyright (c) 1993 by Digital Equipment Corporation
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies, and that
 * the name of Digital Equipment Corporation not be used in advertising or
 * publicity pertaining to distribution of the document or software without
 * specific, written prior permission.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND DIGITAL EQUIPMENT CORP. DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS.   IN NO EVENT SHALL DIGITAL EQUIPMENT
 * CORPORATION BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/*
 * DC: The variable PARSE_FRAG controls whether or not the contents of 
 *  this file are visible to the NNstat programs.  It also is used
 *  to mark the deltas between this version of frag.c and the one
 *  from screend.
 */

#ifdef PARSE_FRAG  /* include this file? */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <stdio.h>

#include "stat.h" 
#include "sobj.h"
#include "packet.h"
    
extern int debug;

/*
 * Whenever we get the first fragment of a packet, we record its
 * identity and the unpacked value.  Subsequent fragments are
 * matched with cache entries and, if present, the unpacked value
 * is returned.
 *
 * The database is a simple hash table.  Entries are removed when
 * their TTL expires (but we limit the TTL to a brief distance
 * into the future, since we should not be seeing too much delay
 * skew).
 *
 * "Identification" is a triple: (src, dst, id) since a sender
 * is allowed to use the same fragment ID for different dests
 * (they probably don't, but better safe than sorry).
 */

#define	TTL_MAX		5	/* seconds before trashing entry */

struct frag_entry {
	struct in_addr src;	/* IP source */
	struct in_addr dst;	/* IP destination */
	u_short	id;		/* IP datagram ID */
        struct ip pk_iphdr;       /* the IP header */
        union ip_data pk_ipdata;  /* the transport layer header */
	time_t expiration;	/* time (tv_sec) when this entry is stale */
	struct frag_entry *next;	/* free (or hash) list chain */
};

#define	HASH_SLOTS	(1<<12)
#define	HASH_MASK	(HASH_SLOTS - 1)
#define	HASH_FUNC(id)	(((id)&HASH_MASK)^(((id)/HASH_SLOTS)&HASH_MASK))

struct frag_entry *frag_hashtab[HASH_SLOTS];
			/* ASSUMPTION: initialized to all NULLs */

struct frag_entry *frag_freelist = NULL;

InitFragCache()
{
	register struct frag_entry *fstore;
	register int count;
	
	count = HASH_SLOTS * 2;		/* arbitrary */
	
	fstore = (struct frag_entry *)malloc(sizeof(*fstore) * count);
	if (fstore == NULL) {
	    perror("InitFragCache/malloc");
	    exit(1);
	}
	
	/* Break up the storage and put it on the free list */
	while (count-- > 0) {
	    fstore->next = frag_freelist;
	    frag_freelist = fstore;
	    fstore++;
	}
}

/* hack */
int nobufs=0;

RecordFragLeader(ipp, unpp)
register struct ip *ipp;
register union ip_data *unpp;
{
	register struct frag_entry *fep;
	register int i;
	
	if (frag_freelist == NULL) {
	    PurgeFragCache();	/* see if there is some space */
	    if (frag_freelist == NULL) {
		if (debug) {
		    fprintf(stderr, "all fragment entries busy\n");
		}
if ((++nobufs % 100) == 0)
 fprintf(stderr,"buf full %d times \n",nobufs);
		return;		/* still no space, tough luck */
	    }
	}
	fep = frag_freelist;
	frag_freelist = fep->next;
	
#ifdef SUNOS4
	bcopy (&ipp->ip_src, &fep->src, sizeof (fep->src));
	bcopy (&ipp->ip_dst, &fep->dst, sizeof (fep->dst));
#else
	fep->src = ipp->ip_src;
	fep->dst = ipp->ip_dst;
#endif
	fep->id = ipp->ip_id;
        bcopy(ipp, &fep->pk_iphdr, sizeof(struct ip));
        bcopy(unpp, &fep->pk_ipdata, sizeof(union ip_data));
	
	i = ipp->ip_ttl;
	if (i > TTL_MAX)
		i = TTL_MAX;
	fep->expiration = time(0) + i + 1;	/* add 1 for roundoff error */
	
	i = HASH_FUNC(fep->id);
	/* insert into that hash chain */
	fep->next = frag_hashtab[i];
	frag_hashtab[i] = fep;
	
	if (debug) {
            struct in_addr x;
            bcopy (&fep->src, &x, sizeof(x));
	    fprintf(stderr, "FragLeader (hash %d) id %x [%s",
			i, fep->id, inet_ntoa(x));
            bcopy (&fep->dst, &x, sizeof(x));
	    fprintf(stderr, "->%s] life %d\n",
			inet_ntoa(x), fep->expiration - time(0));
	}
}


FindFragLeader(ipp, pk_iphdr_out, pk_ipdata_out)
register struct ip *ipp;
register struct ip **pk_iphdr_out;
register union ip_data **pk_ipdata_out;
{
	register int i;
	register struct frag_entry *fep;
	
	i = HASH_FUNC(ipp->ip_id);
	
	if (debug) {
            struct in_addr x;
            bcopy (&ipp->ip_src, &x, sizeof(x));
	    fprintf(stderr, "Fragment id %x [%s",
			ipp->ip_id, inet_ntoa(x));
            bcopy (&ipp->ip_dst, &x, sizeof(x));
	    fprintf(stderr, "->%s]\n", inet_ntoa(x));
	}

	fep = frag_hashtab[i];
	while (fep) {
	    if (debug) {
                struct in_addr x;
                bcopy (&fep->src, &x, sizeof(x));
		fprintf(stderr, "\tChecking #%d id %x [%s",
			    i, fep->id, inet_ntoa(x));
                bcopy (&fep->dst, &x, sizeof(x));
		fprintf(stderr, "->%s]\n", inet_ntoa(x));
	    }
	    if ((fep->id == ipp->ip_id)
#ifdef SUNOS4
			&& bcmp (&fep->src.s_addr, &ipp->ip_src.s_addr,
				 sizeof (fep->src.s_addr))
			&& bcmp (&fep->dst.s_addr, &ipp->ip_dst.s_addr,
				 sizeof (fep->dst.s_addr))) {
#else
			&& (fep->src.s_addr == ipp->ip_src.s_addr)
			&& (fep->dst.s_addr == ipp->ip_dst.s_addr)) {
#endif
		/* match */
                *pk_iphdr_out  = &fep->pk_iphdr;
                *pk_ipdata_out = &fep->pk_ipdata;
		if (debug)
		    fprintf(stderr, "match\n");
		return(1);
	    }
	    /* no match yet */
	    fep = fep->next;
	}
	/* no match at all */
	return(0);
}

PurgeFragCache()
{
	register struct frag_entry *fep;
	register int now;
	register int i;
	struct frag_entry *PurgeFragEntry();
	
	now = time(0);
	
	if (debug) {
	    fprintf(stderr, "Purging fragment cache:\n");
	}
	
	for (i = 0; i < HASH_SLOTS; i++) {
	    fep = frag_hashtab[i];
	    while (fep) {
		if (debug) {
		    fprintf(stderr, "slot %d id %x life %d\n",
				i, fep->id, fep->expiration - now);
		}
		if (fep->expiration < now) {
		    fep = PurgeFragEntry(i, fep);
		}
		else
		    fep = fep->next;
	    }
	}
}

struct frag_entry *
PurgeFragEntry(slot, fep)
register int slot;
register struct frag_entry *fep;
{
	register struct frag_entry *targp;
	register struct frag_entry *nextp = fep->next;
	
	if (debug) {
	    fprintf(stderr, "purging %x from slot %d\n", fep, slot);
	}

	targp = frag_hashtab[slot];
	
	if (targp == fep) {
	    /* unlink from head of hash chain */
	    frag_hashtab[slot] = nextp;
	    /* link into free list */
	    fep->next = frag_freelist;
	    frag_freelist = fep;
	    return(nextp);
	}

	/* must be past front of chain */
	while (targp) {
	    if (targp->next == fep) {
		/* unlink from hash chain */
		targp->next = nextp;
		/* link into free list */
		fep->next = frag_freelist;
		frag_freelist = fep;
		return(nextp);
	    }
	    targp = targp->next;
	}
	
	/* Didn't find it, but that is "impossible" */
	fprintf(stderr, "PurgeFragEntry: entry not found\n");
	exit(1);
}

#endif
