

/****************************************************************************/
/*                                                                          */
/*      NNstat -- Internet Statistics Collection Package                    */
/*                                                                          */
/*            Written by: Bob Braden & Annette DeSchon                      */
/*            USC Information Sciences Institute                            */
/*            Marina del Rey, California                                    */
/*                                                                          */
/*      Copyright (c) 1991, 1992 University of Southern California.         */
/*      All rights reserved.                                                */
/*                                                                          */
/*      Redistribution and use in source and binary forms are permitted     */
/*      provided that the above copyright notice and this paragraph are     */
/*      duplicated in all such forms and that any documentation,            */
/*      advertising materials, and other materials related to such          */
/*      distribution and use acknowledge that the software was              */
/*      developed by the University of Southern California, Information     */
/*      Sciences Institute.  The name of the University may not be used     */
/*      to endorse or promote products derived from this software           */
/*      without specific prior written permission.                          */
/*      THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR        */
/*      IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED      */
/*      WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR          */
/*      PURPOSE.                                                            */
/*                                                                          */
/****************************************************************************/

static char rcsid[] =
	"$Header: parse.c,v 1.7 93/09/21 11:06:23 mogul Exp $";


/*                   parse.c
 *     PARSE PACKETS COLLECTED FROM ETHERNET
 */
 
/* CHANGES:
 *    Aug93/DECWRL:
 *	  Cleaned up fragment-handling code.  Added IP.fraglen field.
 *	  Untangled horribly nested #ifdefs around definitions for
 *	  FI_xxx values.  Ported to Alpha/OSF port.
 *    11Jul93 DC: add IP frag reassembly
 *    18Feb93 JD: add new field Ether.cast, which indicates if we have a
 *                broadcast (2), Multicast (1) or single-cast (0) dest-address
 *    27Jan93 JD: add fields to support IEEE 802.2 header, SNAP-Header etc.
 *    18Nov88 ISI: Time preamble more efficient, and remove unneeded rlen.
 *                       INTERVAL field conditionally compiled.
 *    19Nov88 ISI: Add reference count and cumulative invocation count to FPT.
 *    22Nov88 ISI: Add Ethernet Filter Table 
 *    15Oct89 ISI: Changes for Sun4 (SPARC) -- with thanks to C.Philip Wood(LANL)
 *    17Oct89 AKS/ISI: Changes for subnets
 *          Rel 2.4:
 *     1Nov89 ISI: Change handling of packet length -- new field "length"
 *     9Nov89 ISI: Add symmetric-other field to support symif statement
 *          Rel 3.0:
 *     DEC: Little-endian code.
 *     ISI: Move InitFields() here.
 *     ISI: Allow full snaplen in packet  virtual field
 *     ISI: Define experimental field CurrTimeMS
 *     ISI: Make experimental fields invisible when not included.
 *     ISI: Remove RT-dependent code to etherifrt.c
 *     ISI: New experimental virtual fields: host+port for UDP, TCP
 *     SGI: Add ICMP code field
 *     SGI: Handle Class D network addresses
 *          Rel 3.02:
 *     ISI: Fix more little-endian bugs.
 */
  
#include <stdio.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include "stat.h" 
#include "sobj.h"
#include "packet.h"
    
#define Ether_hdr_ptr ((struct ether_header *) pp)

    /*        Fields
     *
     *  NOTE: THESE MUST BE IN PARSE ORDER... i.e., as a packet is
     *  parsed, the sequence of defined field numbers MUST! be MONTONIC.
     *
     *  Compilation flags:
     *    #define FALIGN:  Include code for SPARC chip; harmless but inefficient
     *               for big-endian hardware.
     *    #define SUBNETS: Include subnet fields SSBNET and DSBNET
     *    #define IBMRTPC: Modify parse for Network Tap header of PC RT
     *    #define LITTLEEND:     Little-endian byte ordering
     *    #define XPARSE:  Include experimental virtual field INTERVAL, hostports.
     *    #define LLC:     Decodes IEEE 802.2 headers too
     *
     */
#define FI_ROOT        1
#define FI_PACKET      1  /* Virtual field ... actually, entire packet! */
#define FI_LENGTH      2  /* Length of packet (excl. of Ethernet hdr) */
#define FI_INTERVAL    3  /* Time in ms since previous packet */
#define FI_PERSEC      4  /* Count of packets received in last second */
#define FI_ETHER_SRC   5
#define FI_ETHER_DST   6

#if defined LLC

#define FI_ETHER_CAST  (FI_ETHER_DST+1)  /*Type of destination address */
#define FI_ETHER_TYPE  (FI_ETHER_CAST+1)
#define FI_ETHER_TYPE2 (FI_ETHER_TYPE+1)
			  /*Ethernet-Type field, with length recoded to 0 */
#define FI_LLC_DSAP    (FI_ETHER_TYPE2+1)  /*LLC2 Destination SAP*/
#define FI_LLC_SSAP    (FI_LLC_DSAP+1)    /*LLC2 Source SAP*/
#define FI_LLC_C       (FI_LLC_SSAP+1)    /*LLC2 Control field*/
#define FI_SNAP        (FI_LLC_C+1)	/* SNAP Type Field*/
#define FI_IP_VERSION  (FI_SNAP+1)
			/*IP version: Defined only if NOT current version */

#else	/* not LLC */

#define FI_ETHER_TYPE  (FI_ETHER_DST+1)
#define FI_IP_VERSION  (FI_ETHER_TYPE+1)
			/*IP version: Defined only if NOT current version */

#endif	/* LLC */

#define FI_IP_PKTLEN   (FI_IP_VERSION+1)
			/* (This is really obsolete, with 'length' field) */
#define FI_IP_FRAGLEN  (FI_IP_PKTLEN+1)	/* length of IP fragment */
#define FI_IP_OPTION   (FI_IP_FRAGLEN+1)
			/*IP option code: defined only if there is option*/
#define FI_IP_TOS      (FI_IP_OPTION+1)
#define FI_IP_OFFSET   (FI_IP_TOS+1)
			/*IP reas'bly offset: Defined only if pkt fragm'ted*/
#define FI_IP_PROT     (FI_IP_OFFSET+1)
#define FI_IP_SRCNET   (FI_IP_PROT+1)	/* Network of source address */
#define FI_IP_DSTNET   (FI_IP_SRCNET+1)	/* Network of dest address */

#if defined SUBNETS        /* subnets */

#define FI_IP_SSBNET   (FI_IP_DSTNET+1)	/* Subnetwork of source address */
#define FI_IP_DSBNET   (FI_IP_SSBNET+1)	/* Subnetwork dest address */
#define FI_IP_SRCHST   (FI_IP_DSBNET+1)

#else   /* no subnets */

#define FI_IP_SRCHST   (FI_IP_DSTNET+1)

#endif	/* SUBNETS */

#define FI_IP_DSTHST   (FI_IP_SRCHST+1)
#define FI_TCP_SPORT   (FI_IP_DSTHST+1)
#define FI_TCP_DPORT   (FI_TCP_SPORT+1)
#define FI_UDP_SPORT   (FI_TCP_DPORT+1)
#define FI_UDP_DPORT   (FI_UDP_SPORT+1)
#define FI_ICMP_TYPE   (FI_UDP_DPORT+1)
#define FI_ICMP_CODE   (FI_ICMP_TYPE+1)

#if defined XPARSE        /* hostports */

#define FI_TCP_SHOSTPORT (FI_ICMP_CODE+1)
#define FI_TCP_DHOSTPORT (FI_TCP_SHOSTPORT+1)
#define FI_UDP_SHOSTPORT (FI_TCP_DHOSTPORT+1)
#define FI_UDP_DHOSTPORT (FI_UDP_SHOSTPORT+1)
#define MAXFI_NO FI_UDP_DHOSTPORT

#else	/* no hostports */

#define MAXFI_NO FI_ICMP_CODE

#endif	/* XPARSE */

  /* Virtual fields... */
static int zero = 0;
static u_int32 snet, dnet, ssbnet, dsbnet;
static u_char ip_version;
static short  ip_offset;
static int32   intervalMS, secrun;
       short  packetlen;  /* length of Ethernet data ...
                           * THIS VAR. IS ACCESSED DIRECTLY BY -bytes OBJECTS
                           */
#ifdef	PARSE_FRAG
short	fraglen;  /* len of datagram frag (or unfrag'd dgram), net order */
#endif	/* PARSE_FRAG */

#ifdef LITTLEEND
static short lilend_pktlen; /* Little-endian version of packetlen */
			    /* i.e., we keep this in big-endian order */
#endif LITTLEEND

#ifdef LLC
static u_char ether_cast;
static u_short ether_type2;
static u_char llc_dsap,llc_ssap;
static u_short llc_control;
static u_short snap_type;
#endif LLC

#ifdef XPARSE
    /* source and destination host+port structs */
 struct {
    u_int32  SHPhost;
    u_short SHPport;
    } Shostport;
struct {
    u_int32  DHPhost;
    u_short DHPport;
    } Dhostport;
#endif
                          
Subnet SubHdr, *SubTail = &SubHdr;  /* Dummy Subnet entry to head
   subnet table, and ptr to last entry in list */

#define FIELD_DEF(num, parent, so, name, ptr, len, type) \
 num, parent, so, 0, name, NULL, (char *)ptr, len, len, type, 0 , 0   

/*            PACKET PARSE TREE:
 *
 *packet
 *  length
 *    Ether.src
 *      Ether.dst
 *        Ether.type
 *          IP.version
 *   ----- Stop parse here if IP.version != 4 ...
 *            IP.length  (Same value as "length", but only for IP packets)
 *              IP.option
 *                IP.TOS
 *                  IP.offset
 *   ----- Stop parse here if offset != 0 (not first fragment) ...
 *                    IP.protocol
 *                      IP.srcnet
 *                        IP.dstnet
 *                          IP.ssubnet
 *                            IP.dsubnet
 *                              IP.srchost
 *                                IP.dsthost
 *                                  TCP.srcport
 *                                    TCP.dstport
 *                                  UDP.srcport
 *                                    UDP.dstport
 *                                  ICMP.type
 *                                    ICMP.code
 *
 */     
FIELD Fields[MAXFI_NO+2] = {
 {NULL},  /* unused */      
 {FIELD_DEF(FI_PACKET, 0, 0, "packet", NULL, 0, DTYP_packet)},
 {FIELD_DEF(FI_LENGTH, FI_PACKET, 0, "length", &packetlen, 2, DTYP_int)},
#ifdef XPARSE  /* Make experimental fields visible */
 {FIELD_DEF(FI_INTERVAL, FI_PACKET, 0, "interval.ms", &intervalMS, 4, DTYP_int)},
 {FIELD_DEF(FI_PERSEC, FI_PACKET, 0, "per.sec", &secrun, 4, DTYP_int)},
#else /* invisible */
 {FIELD_DEF(0, FI_PACKET, 0, "()", NULL, 0, 0)},
 {FIELD_DEF(0, FI_PACKET, 0, "()", NULL, 0, 0)},
#endif   
 {FIELD_DEF(FI_ETHER_SRC, FI_LENGTH, FI_ETHER_DST,
                                        "Ether.src", NULL, 6, DTYP_EtherA)},
                                     
 {FIELD_DEF(FI_ETHER_DST, FI_ETHER_SRC, FI_ETHER_SRC, 
                                        "Ether.dst", NULL, 6, DTYP_EtherA)},
#ifdef LLC
 {FIELD_DEF(FI_ETHER_CAST, FI_ETHER_DST, 0, 
                                  "Ether.cast", &ether_cast, 1, DTYP_bits)},
 {FIELD_DEF(FI_ETHER_TYPE, FI_ETHER_CAST, 0, 
                                        "Ether.type", NULL, 2, DTYP_bits)},
 {FIELD_DEF(FI_ETHER_TYPE2, FI_ETHER_CAST, 0, 
                                  "Ether.type2", &ether_type2, 2, DTYP_bits)},
 {FIELD_DEF(FI_LLC_DSAP, FI_ETHER_TYPE2, FI_LLC_SSAP, 
                                  "LLC.DSAP", &llc_dsap, 1, DTYP_bits)},
 {FIELD_DEF(FI_LLC_SSAP, FI_LLC_DSAP, FI_LLC_DSAP, 
                                  "LLC.SSAP", &llc_ssap, 1, DTYP_bits)},
 {FIELD_DEF(FI_LLC_C, FI_LLC_SSAP, 0, 
                                  "LLC.Control", &llc_control, 2, DTYP_bits)},
 {FIELD_DEF(FI_SNAP, FI_LLC_C, 0, 
                                  "SNAP.type", &snap_type, 2, DTYP_bits)},
 {FIELD_DEF(FI_IP_VERSION, FI_ETHER_TYPE, 0, 
                                      "IP.version", &ip_version, 1, DTYP_int)},

#else

 {FIELD_DEF(FI_ETHER_TYPE, FI_ETHER_DST, 0, 
                                        "Ether.type", NULL, 2, DTYP_bits)},
 {FIELD_DEF(FI_IP_VERSION, FI_ETHER_TYPE, 0, 
                                      "IP.version", &ip_version, 1, DTYP_int)},
#endif

 {FIELD_DEF(FI_IP_PKTLEN, FI_IP_VERSION, 0, 
                                        "IP.length", &packetlen, 2, DTYP_int)},
#ifdef PARSE_FRAG
 {FIELD_DEF(FI_IP_FRAGLEN, FI_IP_PKTLEN, 0, 
                                        "IP.fraglen", &fraglen, 2, DTYP_int)},
#endif
 {FIELD_DEF(FI_IP_OPTION, FI_IP_FRAGLEN, 0, "IP.option", NULL, 1, DTYP_int)},
 {FIELD_DEF(FI_IP_TOS, FI_IP_OPTION, 0, "IP.TOS", NULL, 1, DTYP_bits)},
 {FIELD_DEF(FI_IP_OFFSET, FI_IP_TOS, 0, "IP.offset", &ip_offset, 2, DTYP_int)},
 {FIELD_DEF(FI_IP_PROT, FI_IP_OFFSET, 0, "IP.protocol", NULL, 1, DTYP_int)},
 {FIELD_DEF(FI_IP_SRCNET, FI_IP_PROT, FI_IP_DSTNET, 
                                        "IP.srcnet", &snet, 4, DTYP_IP)},
 {FIELD_DEF(FI_IP_DSTNET, FI_IP_SRCNET, FI_IP_SRCNET, 
                                        "IP.dstnet",  &dnet, 4, DTYP_IP)},
#ifdef SUBNETS
 {FIELD_DEF(FI_IP_SSBNET, FI_IP_DSTNET, FI_IP_DSBNET, 
                                        "IP.srcsubn", &ssbnet, 4, DTYP_IP)},
 {FIELD_DEF(FI_IP_DSBNET, FI_IP_SSBNET, FI_IP_SSBNET, 
                                        "IP.dstsubn", &dsbnet, 4, DTYP_IP)},
 {FIELD_DEF(FI_IP_SRCHST, FI_IP_DSBNET, FI_IP_DSTHST, 
                                        "IP.srchost", NULL, 4, DTYP_IP)},
#else
 {FIELD_DEF(FI_IP_SRCHST, FI_IP_DSTNET, FI_IP_DSTHST, 
                                        "IP.srchost", NULL, 4, DTYP_IP)},
#endif
 {FIELD_DEF(FI_IP_DSTHST, FI_IP_SRCHST, FI_IP_SRCHST, 
                                        "IP.dsthost", NULL, 4, DTYP_IP)},
 {FIELD_DEF(FI_TCP_SPORT, FI_IP_DSTHST, FI_TCP_DPORT, 
                                        "TCP.srcport", NULL, 2, DTYP_int)},
 {FIELD_DEF(FI_TCP_DPORT, FI_TCP_SPORT, FI_TCP_SPORT, 
                                        "TCP.dstport", NULL, 2, DTYP_int)},
 {FIELD_DEF(FI_UDP_SPORT, FI_IP_DSTHST, FI_UDP_DPORT,
                                        "UDP.srcport", NULL, 2, DTYP_int)},
 {FIELD_DEF(FI_UDP_DPORT, FI_UDP_SPORT, FI_UDP_SPORT, 
                                        "UDP.dstport", NULL, 2, DTYP_int)},
 {FIELD_DEF(FI_ICMP_TYPE, FI_IP_DSTHST, 0, "ICMP.type", NULL, 1, DTYP_int)},
 {FIELD_DEF(FI_ICMP_CODE, FI_ICMP_TYPE, 0, "ICMP.code", NULL, 1, DTYP_int)},
 
#ifdef XPARSE
 {FIELD_DEF(FI_TCP_SHOSTPORT, FI_TCP_DPORT, FI_TCP_DHOSTPORT, "TCP.shstport", 
                                  &Shostport, 6, DTYP_IPsock)},
 {FIELD_DEF(FI_TCP_DHOSTPORT, FI_TCP_SHOSTPORT, FI_TCP_SHOSTPORT, "TCP.dhstport", 
                                  &Dhostport, 6, DTYP_IPsock)},
 {FIELD_DEF(FI_UDP_SHOSTPORT, FI_UDP_DPORT, FI_UDP_DHOSTPORT, "UDP.shstport", 
                                  &Shostport, 6, DTYP_IPsock)},
 {FIELD_DEF(FI_UDP_DHOSTPORT, FI_UDP_SHOSTPORT, FI_UDP_SHOSTPORT, "UDP.dhstport", 
                                  &Dhostport, 6, DTYP_IPsock)},
#endif
 {NULL}  /* END OF FIELDS */
};

/* Ethernet Filter Table.  Has one entry for each Ethernet type value
 *  that the parse subroutine defines.
 */
struct FilterTE EtherFTable[] =
    { {FI_IP_VERSION, 0x0800}, 
      {0, 0}  /* flag end of list */
};
   
    
struct timeval last_time = {0, 0};
long   CurrTimeMS, LastTimeMS = -1;
extern time_t StarTime;
            
int    packetno;   /* Max # of original packets to receive (0-1 => no limit)*/
u_long packetcnt;  /* Number received */

u_long StartSecCnt= 0;  /* Packet count at start of second interval */
u_long StartTickCnt= 0; /* Packet count at start of tick interval */
u_long MaxSecRun = 0;   /* Max # packets arriving within one second */
u_long MaxTickRun = 0;  /* Max # packets arriving within one clock tick */
   
#if defined(LITTLEEND)  
    /* Little-endian host.  Note that field values passed to interpreter
     * have NETWORK byte order.
     */
int32 netmask[16] = {     /* Network mask in network byte order... */
    0x000000FF, 0x000000FF, 0x000000FF, 0x000000FF,
    0x000000FF, 0x000000FF, 0x000000FF, 0x000000FF,
    0x0000FFFF, 0x0000FFFF, 0x0000FFFF, 0x0000FFFF,
    0x00FFFFFF, 0x00FFFFFF,
    0xFFFFFFFF, 0xFFFFFFFF,
};
#else           
    /* Big-Endian: Network byte order == host byte order */
int32 netmask[16] = {
    0xFF000000, 0xFF000000, 0xFF000000, 0xFF000000,    /* Class A */
    0xFF000000, 0xFF000000, 0xFF000000, 0xFF000000,    /* Class A */
    0xFFFF0000, 0xFFFF0000, 0xFFFF0000, 0xFFFF0000, /* Class B */
    0xFFFFFF00, 0xFFFFFF00,             /* Class C */
    0xFFFFFFFF, 0xFFFFFFFF,             /* Class D, E */
};
#endif

#define INTERPRET(FIno, p)  { Fields[FIno].field_ptr= (char *) p;  \
                              if (Fields[FIno].field_opcs) \
                                 interpret(Fields[FIno].field_opcs);  }  

#define INTERPRETV(FIno, p)  {  if (Fields[FIno].field_opcs) \
                                 interpret(Fields[FIno].field_opcs);  }
         /* Version of INTERPRET for virtual-fields */  


    /*
     *
     *  PACKET PARSER
     *
     */   
parse( pp, clen, rlen, capture_time)
    register u_char *pp ;
    int clen ;   /* amount of data captured */
    int rlen ;   /* actual packet length */
    struct timeval capture_time;
    {
    register struct ip *ip_h;
    register u_char *optp;         
    register union ip_data *ipdp;     
    register x; 
#ifdef LLC
    struct llc_hdr {
      u_char dsap;
      u_char ssap;
      u_char control;
      u_char snap1;
      u_char snap2;
      u_char snap3;
      u_short type;
    };
    register struct llc_hdr *llc_h;
    u_short ether_typex;
#endif
#ifdef PARSE_FRAG
    struct ip      *l_iphdr;
    union  ip_data *l_ipdata;
    union  ip_data c_ipdata;
#endif
    union {  /* place to full-word align two fullword IP addresses*/
        char ip_addrs_c[8];
        struct {
            struct in_addr saddr; /* src addr */
            struct in_addr daddr; /* dst addr */
        } ip_addrs_s;
    } ip_addrs;
#define SADDR ip_addrs.ip_addrs_s.saddr.s_addr 
#define DADDR ip_addrs.ip_addrs_s.daddr.s_addr 
    Subnet *FindSubnet();
        
    packetcnt++;
    packetlen = rlen - sizeof (struct ether_header);
    if (packetno >= 0 && (packetcnt >= packetno))  {
            if (packetno) exit(0); /* For overhead measurements */
            else return;   /* For local testing */
    }

    Fields[FI_PACKET].field_len = clen ;  /* save len captured */
    INTERPRET(FI_PACKET, pp);

#ifdef LITTLEEND
    lilend_pktlen = htons(packetlen);  /* virtual field to network order */
    INTERPRET(FI_LENGTH, &lilend_pktlen);
#else LITTLEEND
    INTERPRET(FI_LENGTH, &packetlen);
#endif LITTLEEND
     
    CurrTime =  capture_time.tv_sec;  /* Time NIT captured the packet */
    
#ifdef XPARSE   /* Experimental fields */
    CurrTimeMS = capture_time.tv_usec/1000 + (CurrTime-StarTime)*1000;
    if (CurrTimeMS > 0x40000000) CurrTimeMS = 0;
       /* Time since start in milliseconds */
          
       /* INTERVAL: special field, interval in ms; cannot be combined with
        *   any later field (would have no meaning).  INTERVAL was defined
        *   to gather histogram of interarrival intervals; not generally
        *   useful. 
        */
    if (Fields[FI_INTERVAL].field_opcs) { 
        intervalMS =  CurrTimeMS - LastTimeMS;
#ifdef LITTLEEND
        intervalMS = htonl(intervalMS);  /* 'cuz this read by LFETCH [Mogul] */
#endif
        if (LastTimeMS >= 0)
            interpret(Fields[FI_INTERVAL].field_opcs);
    }
    LastTimeMS = CurrTimeMS;
     
#endif XPARSE
  
    if (capture_time.tv_usec != last_time.tv_usec) {
            /*** This assumes microsec clock advances in discrete steps of
            /***    one clock tick (20 ms on Sun 3).
            /*** If new clock tick... ***/
        if (MaxTickRun < (x = packetcnt - StartTickCnt)) 
               MaxTickRun = x;
        StartTickCnt = packetcnt;
    }
    
    if (capture_time.tv_sec != last_time.tv_sec) {
            /*** Maximum # packets arriving in same second ***/
        if (MaxSecRun < (secrun = packetcnt - StartSecCnt))
            MaxSecRun = secrun;
#ifdef XPARSE
            /* # in last second */
        INTERPRETV(FI_PERSEC, &secrun);
#endif XPARSE
        StartSecCnt = packetcnt;
    } 
    last_time = capture_time;
    
        /* BEGIN THE PACKET PARSING... */
    
#if defined (ULTRIX) || defined (sgi) || defined(DECOSF)
    INTERPRET(FI_ETHER_SRC, Ether_hdr_ptr->ether_shost) ;
    INTERPRET(FI_ETHER_DST, Ether_hdr_ptr->ether_dhost) ;
#else
    INTERPRET(FI_ETHER_SRC, &Ether_hdr_ptr->ether_shost) ;
    INTERPRET(FI_ETHER_DST, &Ether_hdr_ptr->ether_dhost) ;
#endif ULTRIX
    INTERPRET(FI_ETHER_TYPE, &Ether_hdr_ptr->ether_type) ;
#ifdef LLC
    ether_type2 = Ether_hdr_ptr->ether_type;
    ether_typex = ntohs(Ether_hdr_ptr->ether_type);
    if (ether_typex < 1501) ether_type2 = 0;
    INTERPRETV(FI_ETHER_TYPE2, &ether_type2);

    /* set type of destination address */
    ether_cast = 0;
#if defined(sgi) || defined(ULTRIX) || defined(DECOSF)
    if (Ether_hdr_ptr->ether_dhost[0] & 1) {
      ether_cast = 1;
      if ((Ether_hdr_ptr->ether_dhost[0] == 0xff)
       && (Ether_hdr_ptr->ether_dhost[0] == 0xff))
	ether_cast = 2;
    }
#else
    if (Ether_hdr_ptr->ether_dhost.ether_addr_octet[0] & 1) {
      ether_cast = 1;
      if ((Ether_hdr_ptr->ether_dhost.ether_addr_octet[0] == 0xff)
       && (Ether_hdr_ptr->ether_dhost.ether_addr_octet[0] == 0xff))
	ether_cast = 2;
    }
#endif
    INTERPRETV(FI_ETHER_CAST, &ether_cast);

    if (ether_type2 == 0) {
      llc_h = (struct llc_hdr *) (pp+sizeof(struct ether_header));
      llc_dsap = llc_h->dsap;
      if (llc_dsap != 0xff) llc_dsap &= 0xfe;
      INTERPRETV(FI_LLC_DSAP, &llc_dsap);

      llc_ssap = llc_h->ssap;
      if (llc_ssap != 0xff) llc_ssap &= 0xfe;
      INTERPRETV(FI_LLC_SSAP, &llc_ssap);

      llc_control = llc_h->control;
      if ((llc_control & 0x01) == 0) llc_control = 0;
      INTERPRETV(FI_LLC_C, &llc_control);

             /* check for SNAP Header */
      if (llc_dsap == 0xaa) {
	snap_type = ntohs ( llc_h->type );
	ether_typex = snap_type;
	INTERPRETV(FI_SNAP, &snap_type);
      }
    }
#endif

#ifdef LLC
    switch (ether_typex) {
#else
    switch (ntohs(Ether_hdr_ptr->ether_type)) {
#endif
    
/*    Statistics on ARP or RARP contents unimplemented...
 *
 *  case ETHERTYPE_ARP:
 *  case ETHERTYPE_REVARP:
 *      break ;
 */
            
    case ETHERTYPE_IP:
        /********************** IP datagram *************************/
        
        ip_h = (struct ip *)(pp+sizeof(struct ether_header));
#ifdef LLC
	       /* skip SNAP-Header */
	if ((ether_type2 ==0)) ip_h = (struct ip *)(pp+8);
#endif
            
            /* Since Butterfly gateways use IP Version 3 for 
             * inter-gateway encapsulation, we are not interested
             * in the version number.  But need to test it, to avoid
             * illegal parses.
             *
             * Note that IP_VERSION virtual field is defined ONLY if the 
             * version is not 4.
             */
        ip_version = ip_h->ip_v ;                
        if (ip_version != IPVERSION)  {             
            INTERPRETV(FI_IP_VERSION, &ip_version) ;    
            return;                                 
        }   
#ifdef  LITTLEEND
        ip_h->ip_off = ntohs(ip_h->ip_off);
#endif  LITTLEEND

#ifndef PARSE_FRAG
            /*
             *  Must test for fragments... assume all headers will fit into
             *   first fragment, just ignore later fragments.  Note that the
             *   IP length field that is recorded is the fragment length.
             *   There is no simple way to get the true packet length; would
             *   essentially have to build reassembly mechanism to match
             *   first and last fragments of each packet.
             */

        if (ip_offset = ip_h->ip_off)   
            if ((ip_offset = ((ip_h->ip_off&0x1fff)<<3)) ||
                                       (ip_h->ip_off&IP_MF)) {
                    /* 
                     * This is a fragment... define IP-OFFSET virtual field 
                     */
#ifdef  LITTLEEND
                ip_offset = htons(ip_offset);
#endif  LITTLEEND
                INTERPRETV(FI_IP_OFFSET, &ip_offset);
                if (ip_offset) return;
            }
                                             
        packetlen = ntohs(ip_h->ip_len);  
                              /* Avoid funny length when small (??) */
        INTERPRET(FI_IP_PKTLEN, &ip_h->ip_len) ;     
        INTERPRET(FI_IP_FRAGLEN, &ip_h->ip_len) ;     
                     
            /* Compute ptr to IP data */
        ipdp = (union ip_data *) ((u_char *) ip_h + (ip_h->ip_hl<<2 ));
#else  /* PARSE_FRAG */
            /*
             *  Attempt IP fragment reassembly
             *  Code is somewhat contorted to try to retain the structure
             *  of this module.
             */
        ipdp = l_ipdata = (union ip_data *) 0;
        l_iphdr = (struct ip *) 0;

	ip_offset = ip_h->ip_off & (~IP_DF);	/* ignore Don't Frag flag */
        if (ip_offset & 0x7fff) {	/* also ignore "reserved" flag */
	    /*
	     * Either IP_MF is set or the fragment offset is non-zero;
	     * either way, we are dealing with a fragmented datagram
	     */
	    ip_offset = (ip_offset & 0x1fff) << 3;

	    /* record offset for all cases */
#ifdef  LITTLEEND
	    ip_offset = htons(ip_offset);
#endif  LITTLEEND
	    INTERPRETV(FI_IP_OFFSET, &ip_offset);

	    packetlen = ntohs(ip_h->ip_len);
	    fraglen = ip_h->ip_len;	/* length of this fragment */

	    if (!ip_offset) {                  /* first frag ? */
		/* find transport header address */
		ipdp = (union ip_data *) ((u_char *) ip_h +
		    (ip_h->ip_hl<<2 ));
		/* store frag in frag cache */
		RecordFragLeader(ip_h, ipdp);
		/* indicate need to skip transport layer */
		l_iphdr = ip_h;

		/*
		 * Must record fragment ip_len in case it is referenced
		 * by objects using IP header fields in if clauses.
		 * When fragments are being handled, it is essentially
		 * impossible to get this value right (design error in
		 * the IP protocol).
		 */
		INTERPRET(FI_IP_PKTLEN, &ip_h->ip_len) ;
		INTERPRET(FI_IP_FRAGLEN, &ip_h->ip_len) ;     
	    } else if (ip_h->ip_off&IP_MF) {   /* 1<frag<n ? */
		/* look for first frag in frag cache */
		if (FindFragLeader(ip_h, &l_iphdr, &l_ipdata)) {
		    /* found: keep track of length */
		    l_iphdr->ip_len = htons(packetlen+ntohs(l_iphdr->ip_len));
		    /* indicate no need to restore transport hdr yet */
		    l_ipdata = (union ip_data *) 0;

		    /* Must record fragment ip_len in case it is referenced */
		    INTERPRET(FI_IP_PKTLEN, &ip_h->ip_len) ;
		    INTERPRET(FI_IP_FRAGLEN, &ip_h->ip_len) ;     
		} else
		    return;  /* anything better to do? */
	    } else {                           /* last frag */
		/* look for first frag in frag cache */
		if (!FindFragLeader(ip_h, &l_iphdr, &l_ipdata)) {
		    /* not found */
		    return;  /* anything better to do? */
		}
		/*
		 * set packetlen so upper layers see 1 whole datagram;
		 * note that this calculation is bogus if we don't
		 * get all the fragments, or we get them in the wrong
		 * order.
		 */
		l_iphdr->ip_len = htons(packetlen + ntohs(l_iphdr->ip_len));
		packetlen = ntohs(l_iphdr->ip_len);
		/* at this point, l_iphdr and l_ipdata are set */
		/* we will do ip hdr then restore transport hdr */

		/* INTERPRET requires total length in network order */
		ip_h->ip_len = l_iphdr->ip_len;
		INTERPRET(FI_IP_PKTLEN, &ip_h->ip_len) ;
		INTERPRET(FI_IP_FRAGLEN, &fraglen) ;     
	    }
	} /* done handling fragmented datagram */
	else {
	    /* This datagram is not fragmented, handle length normally */
	    packetlen = ntohs(ip_h->ip_len);
	    INTERPRET(FI_IP_PKTLEN, &ip_h->ip_len) ;
	    /* for unfragmented datagram, FRAGLEN == PKTLEN */
	    INTERPRET(FI_IP_FRAGLEN, &ip_h->ip_len) ;     
	}

            /* Compute ptr to IP data */
        if (!ipdp)
            ipdp = (union ip_data *) ((u_char *) ip_h + (ip_h->ip_hl<<2 ));
#endif

            /* Compute IP option pointer */
        optp = (u_char *) ip_h + sizeof(struct ip);
         
        if (optp < (u_char *) ipdp) {
            while (optp < (u_char *) ipdp) {
                if (*optp == IPOPT_EOL) break;
                else if (*optp == IPOPT_NOP)
                    optp++;  /* just ignore NOP */
                else {
                    INTERPRET(FI_IP_OPTION, optp);
                    optp += *(optp+1);   
                }
            }  /* loop through options */
        }
        else  /* If no options, value is ZERO */
            INTERPRET(FI_IP_OPTION, &zero);
        
        INTERPRET(FI_IP_TOS, &ip_h->ip_tos) ;        
        INTERPRET(FI_IP_PROT, &ip_h->ip_p) ;
        
            /*** Source and Destination IP addresses ***/
            
#ifdef FALIGN
           /* For RISC processors requiring fullword alignment:
            * copy and full-word align src and dest IP addrs */
        bcopy((u_char *)&ip_h->ip_src, ip_addrs.ip_addrs_c, 
                                         sizeof(ip_addrs.ip_addrs_s));
        snet = SADDR & netmask[((ip_addrs.ip_addrs_c[0])>>4)&0xF];
        dnet = DADDR & netmask[((ip_addrs.ip_addrs_c[4])>>4)&0xF];
#ifdef SUBNETS
        ssbnet = snet;  /* Defaults: subnets same as network numbers. */
        dsbnet = dnet;
        if (SubHdr.Link) {
            register Subnet *sp;
                /*  Subnet table is not empty; look up src and dest networks. 
                 *  If they appear as subnetted, extract corresponding 
                 *  subnet address using specified mask.
                 */
            if (sp = FindSubnet(snet)) 
                ssbnet = SADDR & sp->Mask;
            if (sp = FindSubnet(dnet))
                dsbnet = DADDR & sp->Mask;
        }
        INTERPRETV(FI_IP_SSBNET, &ssbnet);
        INTERPRETV(FI_IP_DSBNET, &dsbnet);
#endif SUBNETS
        INTERPRETV(FI_IP_SRCNET, &snet) ;            
        INTERPRETV(FI_IP_DSTNET, &dnet) ;                 
        INTERPRET(FI_IP_SRCHST, &SADDR) ;  
        INTERPRET(FI_IP_DSTHST, &DADDR) ;  
#else FALIGN       
            /* UNALIGNED CASE... */    
        snet = ip_h->ip_src.s_addr & 
                           netmask[((ip_h->ip_src.s_net)>>4)&0xF] ;
        dnet = ip_h->ip_dst.s_addr & 
                           netmask[((ip_h->ip_dst.s_net)>>4)&0xF] ;
#ifdef SUBNETS
        ssbnet = snet;  /* Defaults: subnets same as network numbers. */
        dsbnet = dnet;
        if (SubHdr.Link) {
            register Subnet *sp;
                /*  Subnet table is not empty; look up src and dest networks. 
                 *  If they appear as subnetted, extract corresponding 
                 *  subnet address using specified mask.
                 */
            if (sp = FindSubnet(snet)) 
                ssbnet = ip_h->ip_src.s_addr & sp->Mask;
            if (sp = FindSubnet(dnet))
                dsbnet = ip_h->ip_dst.s_addr & sp->Mask;
        }
        INTERPRETV(FI_IP_SSBNET, &ssbnet);
        INTERPRETV(FI_IP_DSBNET, &dsbnet);
#endif SUBNETS
        INTERPRETV(FI_IP_SRCNET, &snet) ;            
        INTERPRETV(FI_IP_DSTNET, &dnet) ;                 
        INTERPRET(FI_IP_SRCHST, &ip_h->ip_src) ;  
        INTERPRET(FI_IP_DSTHST, &ip_h->ip_dst) ;  
#endif

#ifdef PARSE_FRAG
        if (l_iphdr)       /* frags? */
            if(l_ipdata) { /* last frag? */
                /* restore transport header */
                /* copying is expensive but makes mem mgmt easy */
                /* and preserves the structure of this module */
                bcopy(l_ipdata, &c_ipdata, sizeof(union ip_data));
                ipdp = &c_ipdata;
            } else         /* frag 1..n-1 */
                return;
#endif

            /**** Process higher-level protocol header... *****/
        switch (ip_h->ip_p) {
        
        case IPPROTO_TCP:
            INTERPRET(FI_TCP_SPORT, &ipdp->pk_tcphdr.th_sport) ;
            INTERPRET(FI_TCP_DPORT, &ipdp->pk_tcphdr.th_dport) ;
            
#ifdef XPARSE   /* Define (host,port) virtual fields */
#ifdef FALIGN
            Shostport.SHPhost =  ip_addrs.ip_addrs_s.saddr.s_addr;
            Dhostport.DHPhost =  ip_addrs.ip_addrs_s.daddr.s_addr;
#else
            Shostport.SHPhost =  ip_h->ip_src.s_addr;
            Dhostport.DHPhost =  ip_h->ip_dst.s_addr;
#endif
            Shostport.SHPport =  ipdp->pk_tcphdr.th_sport; 
            Dhostport.DHPport =  ipdp->pk_tcphdr.th_dport; 
            INTERPRET(FI_TCP_SHOSTPORT, &Shostport);
            INTERPRET(FI_TCP_DHOSTPORT, &Dhostport);
#endif
            return ;
        
        case IPPROTO_UDP:
            INTERPRET(FI_UDP_SPORT, &ipdp->pk_udphdr.uh_sport) ;
            INTERPRET(FI_UDP_DPORT, &ipdp->pk_udphdr.uh_dport) ;
#ifdef XPARSE   /* Define (host,port) virtual fields */

#ifdef FALIGN
            Shostport.SHPhost =  ip_addrs.ip_addrs_s.saddr.s_addr;
            Dhostport.DHPhost =  ip_addrs.ip_addrs_s.daddr.s_addr;
#else
            Shostport.SHPhost =  ip_h->ip_src.s_addr;
            Dhostport.DHPhost =  ip_h->ip_dst.s_addr;
#endif
            Shostport.SHPport =  ipdp->pk_udphdr.uh_sport; 
            Dhostport.DHPport =  ipdp->pk_udphdr.uh_dport; 
            INTERPRET(FI_UDP_SHOSTPORT, &Shostport);
            INTERPRET(FI_UDP_DHOSTPORT, &Dhostport);
#endif
            return ;
        
        case IPPROTO_ICMP:
            INTERPRET(FI_ICMP_TYPE, &ipdp->pk_icmphdr.icmp_type) ;
            INTERPRET(FI_ICMP_CODE, &ipdp->pk_icmphdr.icmp_code) ;
            return ;        
        
        default:
            return ;        
        } ;
        
        break ;
        
    default:
        return ;
    } ;
    
} /* parse() */


/*
 *    Initialize Fields array (FPT data structure)
 *      ...Called once when statspy starts up
 *
 */
InitFields()
    {
    register FIELD *FIp;
    int i, Eq;
    int EqNo = 0;  /* Unique equiv index */
    extern u_long snaplen;
    
    Fields[FI_PACKET].field_maxlen = snaplen; 
    
        /* clear subordination count fields, and set symother fields to
         *   self if they are zero.
         */
    for (FIp = &Fields[1]; FIp->field_name; FIp++) {
        FIp->field_subordno = 0;
        if (FIp->field_symother == 0) 
            FIp->field_symother = FIp->field_fino;
    }
        /* Now count up subordno fields -- the count of descendants from
         *    each node.
         */
    for (FIp = &Fields[1]; FIp->field_name; FIp++) {
        if (i = FIp->field_parent)
            Fields[i].field_subordno++;
    }
        /* Finally, partition nodes into equivalence sets, based on
         *  predicate "one descendant", ie, subordno == 1.  Assign a
         *  unique index to each set.
         */
    for (FIp = &Fields[1]; FIp->field_name; FIp++) {
        FIp->field_equivno = ((i = FIp->field_parent) && 
                              (Fields[i].field_subordno == 1) &&
                              (Eq = Fields[i].field_equivno))? Eq: ++EqNo;
    }    
}  /* InitFields() */


SetEtherType()
    {
    /* 
     *  ...Called by each attach operation.
     *  Scan through all FPT subtrees that correspond to parsed Ethernet
     *   types, and check if exactly one of them has any invocations,
     *   and if there are no invocations for the Ethernet fields themselves.
     *   If so, can use the Ethernet Type filter of NIT 3.x.  If there are
     *   no invocations at all, do nothing.
     */
     struct FilterTE *ftep;
     
     if (Fields[FI_ROOT].field_cuminv == 0 && 
         Fields[FI_PERSEC].field_cuminv == 0) 
           return;
     
     for (ftep = EtherFTable; ftep->FTE_fino; ftep++) {
          if (Fields[ftep->FTE_fino].field_cuminv) {
              break;
          }
     }
     if ((ftep->FTE_fino) && (Fields[ftep->FTE_fino].field_cuminv == 
               Fields[FI_ROOT].field_cuminv ))  {
             initdevice(ftep->FTE_value);
             return;
         }
         
     /* Else set promiscuous */
     initdevice(-1);
}

#ifdef SUBNETS  
Subnet *FindSubnet(NAddr)
    register u_int32 NAddr;
    {
    register Subnet *sp;
    for (sp = SubHdr.Link; sp&&(sp->Addr != NAddr); sp= sp->Link);
    return(sp);
}
#endif
