
/****************************************************************************/
/*                                                                          */
/*      NNstat -- Internet Statistics Collection Package                    */
/*                                                                          */
/*            Written by: Bob Braden & Annette DeSchon                      */
/*            USC Information Sciences Institute                            */
/*            Marina del Rey, California                                    */
/*                                                                          */
/*      Copyright (c) 1991 University of Southern California.               */
/*      All rights reserved.                                                */
/*                                                                          */
/*      Redistribution and use in source and binary forms are permitted     */
/*      provided that the above copyright notice and this paragraph are     */
/*      duplicated in all such forms and that any documentation,            */
/*      advertising materials, and other materials related to such          */
/*      distribution and use acknowledge that the software was              */
/*      developed by the University of Southern California, Information     */
/*      Sciences Institute.  The name of the University may not be used     */
/*      to endorse or promote products derived from this software           */
/*      without specific prior written permission.                          */
/*      THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR        */
/*      IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED      */
/*      WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR          */
/*      PURPOSE.                                                            */
/*                                                                          */
/****************************************************************************/

static char rcsid[]=
  "$Header: /tmp_mnt/r/jove_staff3/mogul/alpha/code/NNstat/RCS/sobjfob.c,v 1.4 1993/08/31 22:48:06 mogul Exp $";

/*
 *  CHANGES:
 *     Rel 2.4: This object added.
 *     Rel 3.0:
 *   ISI:  Fix bug -- handle nparms==0 without error.
 *   ISI:  Add List routine.  
 *   Aug93/DECWRL: Alpha/OSF port.  XDR is a pointer type.
 */
  
/*
 *   Internet Statistics Facility -- Generic Operation Routines
 *
 *    This module defines a set of operations on a particular class
 *    of Statistical Objects.  In the spirit of object-oriented
 *    programming and information-hiding, these routines are
 *    the only ones which know about the internal structre of the
 *    Statistical Object (SOBJ) data structures.
 *
 */
 
/*
 *
 *          FOB Object Class ==  freq-only-bytes 
 *
 *    Objects of the FOB class are frequency distributions using preset
 *    mappings of values into bins; there is also a default bin for
 *    those values which do not map into any of the defined bins.
 *    The total byte counts are also accumulated.
 *
 *    FOB objects are implemented using a precomputed open hashing.  When
 *    the table is computed it tries to choose a hash table large enough
 *    to get unique hashes. 
 *
 *
 *    Operations on FOB class objects:
 *
 *       SOBJ *Make_FOB( objname, &parmlist[#bins], #bins, datalen)
 *
 *       Write_FOB( (SOBJ *) SOBJ-ptr, &value , length)
 *
 *       Read_FOB( fh, (SOBJ *) SOBJ-ptr )
 *
 *       Delete_FOB( (SOBJ *) SOBJ-ptr )
 *
 *       Clear_FOB( (SOBJ *) SOBJ-ptr )
 *
 *       List_FOB( fd, OBJp )
 *
 */
#include <stdio.h>
#include <sys/types.h>
#include "stat.h"
#include "sobj.h"
    
SOBJ *Make_FOB() ;
boolean Write_FOB(), Read_FOB();
void  Clear_FOB(), Delete_FOB(), List_FOB();
extern boolean ReadFO();

    /* Tranfer Vector 
     *   for external access to these routines
     */
GENERICOP FOBop ={ Make_FOB, Write_FOB, Read_FOB, Clear_FOB, Delete_FOB, List_FOB};

    /****************************************************************     
     *  Define FOB object data structures
     *
     ****************************************************************/

    /* Parameters controlling size of hash table */
#define HASHmaxmult 2  /* min = 2*nbins */
#define HASHmaxtries 8 /* Try no more than 8 hash table sizes */
#define HASHincr  7    /* Add to hash table size for each try */

    /* One bin element */

#define FOBE struct only_bin
FOBE {
    FOBDE  fobe_data;     /* (Use external format internally) */
} ;

#define fobe_value  fobe_data.fob_value
#define fobe_count  fobe_data.fob_count
#define fobe_Lbyte  fobe_data.fob_Lbyte
#define fobe_Hbyte  fobe_data.fob_Hbyte

    /* Object area */
    
#define FOBA struct stat_farea
FOBA {
    SOBJ   foba_sobj ;         /* SOBJ = standard root of area */
    
    short  foba_esize ;        /* Size of element */
    short  foba_hashs ;        /* Size of hash table (#shorts) */
    short  foba_nbins ;        /* Size of bin area (#elements) */    
    char   foba_flags ;        /* Flags */
#define FOBF_MULTI 0x80        /* Non-unique hash */
    char   foba_dum2 ;         /* (unused) */
    struct dblword foba_bytes; /* Default byte count */
    u_int32 foba_others;        /* Count of Writes matching no bin */
    FOBE  **foba_hashp ;       /* Ptr to hash table */
    FOBE   *foba_fobep ;       /* Ptr to bin vector */
    /*----------------------------------------------------
                     Hash Table... 
         (a vector of pointers to bins)
    /*----------------------------------------------------
                    Vector of bins...
         (each entry is an FOBE)
    /*----------------------------------------------------*/
} ;
    

struct FOB_parms {
        u_int32 fobp_value[LONGPERVAL];  /* Actually use (dleng+3)/4 words */
};
    
                
#define hashfunc( V, S)  V%S    /* Simple congruential hash */  


SOBJ *Make_FOB(objname, parmp, nparms, datalen)
    char *objname;
    struct FOB_parms *parmp ;
    int datalen, nparms;
    {
    register FOBA *fobap ;
    register FOBE **hp, *fobep ;
    u_char *pp, *cp;    
    int i, ix, nhash ;
    int tries_left = HASHmaxtries ;    
       /* Try up to 'tries_left' sizes, to get unique hashing... */
       
    int esize = sizeof (FOBE)  - MAX_DLENG + ALIGN(datalen);
    u_int32 HValue;      
    union v_union Value;
    int offset = PARMALIGN(datalen);
    int vsize = ALIGN(datalen);
    int nbins = nparms/(vsize>>2);
         
    ZERO_UNION(&Value);
        
    nhash = (nparms)? HASHmaxmult*nbins : 1;  
    while( tries_left-- ) {

        i = ALIGN(nhash*sizeof(FOBE *) + sizeof(FOBA)) + nbins*esize  ;
        if ((fobap = (FOBA *) malloc(i)) == NULL) {
            SOBJ_error = "Memory overflow!" ;
            return(NULL) ;
        }
        bzero(fobap, i) ;  /* Clear entire area... */
        
        hp = (FOBE **) ((char *) fobap + sizeof(FOBA) ) ;
            /* hp = start of hash table */          
        fobap->foba_fobep = fobep = (FOBE *) PTRALIGN( &hp[nhash] ) ;
            /* align start of bins to fullword boundary */
            
            /* Hash all the values now */
            
        pp = ( u_char *) parmp + offset;
        for (i = 0; i < nbins ; i++)  {
            bcopy(pp, Value.v_byte, datalen);               
            HValue = SUM_UNION(&Value);
            ix = hashfunc( HValue, nhash ) ;
            while (hp[ix]) {
                    /* Bin already in use... */
                    /* Could check for duplicates... */
                if (tries_left)  goto Tryagain ;
                ix++ ;
                ix %= nhash ;
                fobap->foba_flags |= FOBF_MULTI ;
            }
            hp[ix] = fobep ;
            pp += vsize;
            fobep = (FOBE *) ((char *) fobep + esize);
        }
            /* If reach here, have hashed all OK. */
        
        fobap->foba_esize = esize;
        fobap->foba_hashs = nhash;
        fobap->foba_nbins = nbins ;
        fobap->foba_hashp = hp;
                
            /* Now fill bin vector with values*/
        pp = ( u_char *) parmp + offset;
        cp = fobap->foba_fobep->fobe_value; 
        for (i=0; i<nbins; i++) {
            bcopy( pp, cp, datalen) ;
            pp += vsize;
            cp += esize;
        }           
        return(&fobap->foba_sobj) ;  /* SUCCESS!! */
        
Tryagain:   
    free(fobap) ;                                
    nhash += HASHincr;
    }  /* End of loop to find perfect hash */
    
    /* (Never get here... ) */
}    
     
     
boolean Write_FOB( fobap, Valuep, L)
    register FOBA *fobap ;
    char *Valuep;
    register int L;
    {
    register FOBE *fobep, **hp ;
    union v_union  Value;
    extern Bcopy();
    extern short packetlen;  /* THIS CHEAT IS WHAT MAKES IT POSSIBLE... */
             
    ZERO_UNION(&Value);
    Bcopy(Valuep, Value.v_byte, L);                
    hp = &fobap->foba_hashp[ hashfunc((SUM_UNION(&Value)), fobap->foba_hashs) ];
    
    while (fobep = *hp) {
        if (WORD1(&Value) == WORD1(fobep->fobe_value) &&
              (L <= sizeof(int32) || WORD2(&Value) == WORD2(fobep->fobe_value))){
                /* Found bin, so increment its count and byte count */
            fobep->fobe_count++;
            if ((fobep->fobe_Lbyte += packetlen) >= MAX_LBYTE) {
                fobep->fobe_Lbyte -= MAX_LBYTE;
                fobep->fobe_Hbyte++;
            }                
            return(WRRET_OK);
        }
                /* Wrong bin...  */
        if (fobap->foba_flags & FOBF_MULTI == 0)  break;
        if (++hp >= &fobap->foba_hashp[fobap->foba_hashs]) hp = fobap->foba_hashp;
    }
        /* No match.  Increment "other" and add bytes. */
    fobap->foba_others++;
    if ((fobap->foba_bytes.L += packetlen) >= MAX_LBYTE) {
        fobap->foba_bytes.L -= MAX_LBYTE;
        fobap->foba_bytes.H++;
    }
    return(WRRET_OK);
}    
        
void Delete_FOB( fobap )
    FOBA *fobap ;
    {
    free(fobap) ;
}

void Clear_FOB( fobap )
    register FOBA *fobap ;
    {
    register FOBE *fobep = fobap->foba_fobep;
    register int left = fobap->foba_nbins;
    
    while (left-- > 0) { 
        fobep->fobe_count = fobep->fobe_Lbyte = fobep->fobe_Hbyte = 0; 
        fobep = (FOBE *) ((char *) fobep + fobap->foba_esize);
    };
    fobap->foba_others = 0;
    fobap->foba_bytes.H = fobap->foba_bytes.L = 0;
}


boolean Read_FOB(fh, fobap)
    XDR *fh;
    register FOBA *fobap ;
    {
    FOBE *fobep = fobap->foba_fobep;
    int   nbins = fobap->foba_nbins;
    OBJ  xobj;
    struct FOparmB bytparm;  /* Two double-precision byte counts */
            
    if (!PutReadResp(fh, FOBr_SIZE(nbins, fobap->foba_sobj.sob_dleng),
                  fobap)) return(FALSE);
   
    Sum_FOBytes(fobap, &bytparm);
       
    SetupXRep(&xobj, fobap);
    if (!ReadFO(fh, &fobep, &nbins, &fobap->foba_others, &xobj, 
                               fobap->foba_esize, ISTTY(fh), &bytparm)){
        return(FALSE);
    }
    return(TRUE);       
}


void
List_FOB(fd, OBJp)
XDR *fd;
OBJ *OBJp;
{
   FOBDE  bins[MAX_FONO];  /* Place to decode data from bins */
   FOBDE *binp = bins;
   int    count = MAX_FONO; 
   int    defaultno;
   struct FOparmB totalbytes; 
   
   if (!ReadFO(fd, &binp, &count, &defaultno, OBJp, sizeof(FOBDE), 1,  
                                                                &totalbytes))
      printf(" BAD ReadFOB! ");
} /* List_FOB() */

/*
 *  When read, call the following routine to sum up all bytes.  
 *  This is expensive, but probably not prohibitively expensive, 
 *  and maybe useful.
 */
Sum_FOBytes(fobap, parmBp) 
    FOBA *fobap;
    struct FOparmB *parmBp;
    {
    register struct dblword *dblp = &parmBp->totalB;
    register FOBE *fobep = fobap->foba_fobep;
    register int left = fobap->foba_nbins;
    
    parmBp->defaultB.H = parmBp->totalB.H = fobap->foba_bytes.H;
    parmBp->defaultB.L = parmBp->totalB.L = fobap->foba_bytes.L;
    while (left-- > 0) { 
        dblp->H += fobep->fobe_Hbyte;
        if ((dblp->L += fobep->fobe_Lbyte) >= MAX_LBYTE) {
                /* Handle carry from lower part */
            dblp->L -= MAX_LBYTE;
            dblp->H++; 
        } 
        fobep = (FOBE *) ((char *) fobep + fobap->foba_esize);
    };
}  
