
/* ev_Script.c */

/* does message handlers and functions */

#include <stdio.h>
#include <ctype.h>
#include <X10/Xlib.h>
#include "ev_tokens.h"
#include "clink.h"

static int	pastGlobals = 0;	/* have passed the global declarations */
static int	inMsgHandler = 0;	/* are we inside a handler (parser) */
static int	inFunction = 0;		/* are we inside a function (parser) */
static int	inRepeat = 0;		/* are we inside a repeat loop (parser) */

ENV_PTR		curenv = NULL;
OBJ_PTR		curobj = NULL;
CODE_PTR	curcode = NULL;

extern char	theProgName[64];

int
ev_Script()
{
char *calloc();

	EV_DEBUG(2,("--> entering Script\n"));
	++indentLevel;

	ev_Scan();
	if( theToken.type != T_environment )
		ev_Error( "expected <environment> declaration" );
	ev_Scan();
	if( theToken.type != T_word )
		ev_Error( "expected name for environment" );

	envlist = curenv = newenv(theToken.value);

	ev_Scan();


	if( theToken.type != T_begblock )
		ev_Error( "expected <uses> or '{'" );

	/* now know we have a T_begblock */

	ev_Scan();
	do
	{
		if( theToken.type == T_object )
		{
			if( !pastGlobals )
				pastGlobals = 1;	/* no more globals after this */
			ev_Object();
			ev_Scan();
		}
		else if( theToken.type == T_function )
		{
			if( pastGlobals )
				ev_Error( "functions declaration in <objects> section" );
			ev_Function();
			ev_Scan();
		}
		else if( theToken.type == T_on )
		{
			if( pastGlobals )
				ev_Error( "message handler in <objects> section" );
			ev_MsgHandler();
			ev_Scan();
		}
		else if( theToken.type == T_var )
		{
			if( pastGlobals )
				ev_Error( "variable declaration in <objects> section" );
			ev_Var();
			ev_Scan();
		}
		else if( theToken.type >= T_Cfunc && theToken.type <= T_Cflt )
		{
			if( pastGlobals )
				ev_Error( "C declaration in <objects> section" );		
			ev_C_declaration();
			ev_Scan();
		}
		else if( theToken.type != T_endblock )
			ev_Error( "syntax error; may need a '}'" );
	} while( theToken.type != T_endblock );

	ev_Scan();
	if( theToken.type != T_eof )
		ev_Error( "expected end-of-file" );


	--indentLevel;
	EV_DEBUG(2,("--> leaving Script\n"));
	return 0;
}

int
ev_Var()		/* works on phrases of form 'var c;' */
{

	ev_Scan();
	if( theToken.type != T_word )
		ev_Error( "expected variable name" );

	makevar();

	ev_Scan();
	if( theToken.type != T_cmdend )
		ev_Error( "expected ';'" );
	return 0;
}

int
ev_Object()
{
int	pastCommons = 0;

	EV_DEBUG(2,("--> entering Object\n"));
	++indentLevel;

	ev_Scan();
	if( theToken.type != T_word && theToken.type != T_string )
		ev_Error( "<object> expects name" );

	if( theToken.type == T_string && !strlen(theToken.value) )
		ev_Error( "<object> must have non-NULL name" );

	if( findobj(curenv,theToken.value) )
		ev_Error( "duplicate object declaration" );	
	curobj = newobj(theToken.value);

	ev_Scan();
	if( theToken.type != T_begblock )
		ev_Error( "expected a '{' for object" );	

	ev_Scan();
	do
	{
		if( theToken.type == T_on )
		{
			if( !pastCommons )
				pastCommons = 1;
			ev_MsgHandler();
			ev_Scan();
		}
		else if( theToken.type == T_function )
		{
			if( pastCommons )
				ev_Error( "function declared in handler section" );
			ev_Function();
			ev_Scan();
		}
		else if( theToken.type == T_var )
		{
			if( pastCommons )
				ev_Error( "var declared in handler section" );
			ev_Var();
			ev_Scan();
		}
		else if( theToken.type != T_endblock )
			ev_Error("expected <on>, <function>, declaration, or '}' token");
	} while( theToken.type != T_endblock );

	addobj( curenv, curobj );
	curobj = NULL;
	--indentLevel;
	EV_DEBUG(2,("--> leaving Object\n"));
	return 0;
}

int
ev_C_declaration()
{
VAR_PTR	vp;
CODE_PTR cp;
C_DCL_PTR cdp;
int dcl_type = theToken.type;
HashNodePtr	hnp;
char vname[32];

	EV_DEBUG(2,("--> entering C_declaration\n"));
	++indentLevel;

	if( theToken.type != T_Cfunc )
	{
		/* C declaration goes in special C-list and in global varlist */
			
		ev_Scan();
		if( theToken.type != T_word )
			ev_Error( "syntax error; C-variable name missing" );

		if( !findCdcl(theToken.value) )
		{
			hnp = cprog_lookup( C_ht, theToken.value );
			if( !hnp )
				ev_Error( "declared C-variable does not exist" );
			cdp = newCdcl( theToken.value, dcl_type );
			addCdcl( cdp );
			if( findvar(curenv->global_vars,theToken.value) )
				ev_Error( "duplicate name for global var and C-item" );
			vp = makevar();
			vp->Cptr = hnp->ptr;	/* copy the C address */
		}
		else
			ev_Error( "multiple declaration of C-item" );

		ev_Scan();
		if( theToken.type != T_cmdend )
			ev_Error( "C-variable declaration is missing ';'" );
	}
	else
	{
		ev_Scan();
		if( theToken.type != T_word )
			ev_Error( "syntax error; C-function name missing" );

		if( !findCdcl(theToken.value) )
		{
			hnp = cprog_lookup( C_ht, theToken.value );
			if( !hnp )
				ev_Error( "declared C-function does not exist" );
			cdp = newCdcl( theToken.value, dcl_type );
			addCdcl( cdp );
			/* make new code instead of var */
			if( findcode(curenv->global_code,theToken.value) )
				ev_Error( "duplicate name for C-function and global codeseg" );
			cp = newcode( theToken.value, CFUNC_CODESEG );
		}
		else
			ev_Error( "multiple declaration of C-function" );

		ev_Scan();
		if( theToken.type != T_begexpr )
			ev_Error( "C-function declaration is missing parameter list" );
		ev_Scan();
		while( theToken.type != T_endexpr )
		{
			if( theToken.type < T_Cint || theToken.type > T_Cflt )
				ev_Error( "C-function parameter list expects C-types" );

			/* clearly, this is duplicate info */
			cdp->ptype[cdp->nparms++] = theToken.type;
			cp->ptype[cp->nparms] = theToken.type;
			sprintf(vname,"%d",cp->nparms);
			cp->parms[cp->nparms++] = newvar(vname);

			ev_Scan();
			if( theToken.type != T_comma && theToken.type != T_endexpr )
				ev_Error( "C-function parameter list expects comma" );
			if( theToken.type == T_comma )
				ev_Scan();
		}
		ev_Scan();
		if( theToken.type == T_returns )
		{
			ev_Scan();
			if( theToken.type < T_Cint || theToken.type > T_Cflt )
				ev_Error( "C-function 'returns' needs C-type" );

			/* clearly, this is duplicate info */
			cp->retval = theToken.type;
			cdp->retval = theToken.type;

			ev_Scan();
		}
		if( theToken.type != T_cmdend )
			ev_Error( "C-function declaration is missing ';'" );
		addcode( &curenv->global_code, cp );
	}
	
	--indentLevel;
	EV_DEBUG(2,("--> leaving C_declaration\n"));
	return 0;
}

int
ev_MsgHandler()
{
VAR_PTR	vp;

	EV_DEBUG(2,("--> entering MsgHandler\n"));
	++indentLevel;
	inMsgHandler = 1;

	ev_Scan();
	if( theToken.type != T_word )	/* MUST be a single word */
		ev_Error( "invalid syntax for message handler" );

	curcode = newcode( theToken.value, MSG_CODESEG );

	/* new version: msg handlers now take parameter lists */

	ev_Scan();
	while( theToken.type != T_begblock )
	{
		if( theToken.type != T_word )
			ev_Error( "handler expects parameter" );

		vp = makevar();		/* goes into curcode: this handler */
		vp->IsParm = 1;		/* remember this is a parameter */
		if( curcode->nparms < MAXPARMS )
			curcode->parms[curcode->nparms++] = vp;
		else
			ev_Error( "too many parms for handler" );

		ev_Scan();	/* step over the var */
		if( theToken.type != T_comma && theToken.type != T_begblock )
			ev_Error( "handler parameter list expects comma" );
		if( theToken.type == T_comma )
			ev_Scan();		
	} 

	ev_Block();

	if( curobj )
		addcode( &(curobj->codelist), curcode );
	else
		addcode( &(curenv->global_code), curcode );
	curcode->ninstr = curcode->pc;

	curcode = NULL;
	inMsgHandler = 0;
	--indentLevel;
	EV_DEBUG(2,("--> leaving MsgHandler\n"));
	return 0;
}

int
ev_Function()
{
VAR_PTR	vp;

	EV_DEBUG(2,("--> entering Function\n"));
	++indentLevel;
	inFunction = 1;

	ev_Scan();
	if( theToken.type != T_word )
		ev_Error( "expected function name" );

	curcode = newcode( theToken.value, FUNC_CODESEG );
	if( curobj )
		addcode( &(curobj->codelist), curcode );
	else
		addcode( &(curenv->global_code), curcode );

	ev_Scan();
	if( theToken.type != T_begexpr )
		ev_Error( "function is missing parameter list" );
	ev_Scan();
	while( theToken.type != T_endexpr )
	{
		if( theToken.type != T_word )
			ev_Error( "function parameter list expects variable name" );
		
		/* increase the # of parms; add parms to code's varlist; put 
			the var ptrs in the codeseg struct */

		vp = makevar();		/* goes into curcode: this function */
		vp->IsParm = 1;		/* remember this is a parm */
		if( curcode->nparms < MAXPARMS )
			curcode->parms[curcode->nparms++] = vp;
		else
			ev_Error( "too many parms for function" );

		ev_Scan();
		if( theToken.type != T_comma && theToken.type != T_endexpr )
			ev_Error( "function parameter list expects comma" );
		if( theToken.type == T_comma )
			ev_Scan();
	}
	ev_Scan();

	if( theToken.type != T_begblock )
		ev_Error( "function expected beginning of block" );
	ev_Block();

/*	-- moved this up to enable recursive function definition
	if( curobj )
		addcode( &(curobj->codelist), curcode );
	else
		addcode( &(curenv->global_code), curcode );
*/
	curcode->ninstr = curcode->pc;

	curcode = NULL;
	inFunction = 0;
	--indentLevel;
	EV_DEBUG(2,("--> leaving Function\n"));
	return 0;
}

FCALL_PTR
ev_Fcall()		/* call this any time it's an fcall */
{
VAR_PTR	vp = NULL;
FCALL_PTR fcp, tfcp;
EXPR_PTR ep;
CODE_PTR cp;

	EV_DEBUG(2,("--> entering Function Call\n"));
	++indentLevel;

	/* must match function name and check # of parms */
	if( curobj && (cp=findcode(curobj->codelist,theToken.value)) )
		;
	else if( curenv && (cp=findcode(curenv->global_code,theToken.value)) )
		;
	else
		ev_Error( "function does not exist" );

	fcp = newfcall( cp );
	
	ev_Scan();
	if( theToken.type != T_begexpr )
		ev_Error( "function call is missing parameter list" );

	ev_Scan();
	while( theToken.type != T_endexpr )
	{
		if( theToken.type != T_word && theToken.type != T_begexpr &&
			theToken.type != T_int && theToken.type != T_string && 
			theToken.type != T_float )
			ev_Error( "function parameter list needs var, const, expr, or fcall" );

		if( theToken.type == T_begexpr )
		{
			ep = ev_Expression();
			if( addparm( fcp, (char *) ep ) )
				ev_Error( "too many parms for fcall" );
		}
		else if( theToken.type == T_word && isFcall() )
		{
			tfcp = ev_Fcall();
			if( addparm( fcp, (char *) tfcp ) )
				ev_Error( "too many parms for fcall" );
		}
		else
		{
			vp = makevar();
			if( addparm( fcp, (char *) vp ) )
				ev_Error( "too many parms for fcall" );
		}

		ev_Scan();
		if( theToken.type != T_comma && theToken.type != T_endexpr )
			ev_Error( "function parameter list expects comma" );
		if( theToken.type == T_comma )
			ev_Scan();
	}

	if( fcp->nparms != fcp->func->nparms )
		ev_Error( "parameter count mismatch for fcall" );

	--indentLevel;
	EV_DEBUG(2,("--> leaving Function Call\n"));
	return fcp;
}

int
ev_If()
{
EXPR_PTR ep;

	EV_DEBUG(2,("--> entering If\n"));
	++indentLevel;
	
	append_instr( curcode, (long) T_if );
	ev_Scan();
	if( theToken.type == T_begexpr )
	{
		ep = ev_Expression();
		append_instr( curcode, (long) ep );
	}
	else
		ev_Error( "no expression to evaluate for <if>" );

	ev_Scan();
	if( theToken.type == T_begblock )
		ev_Block();
	else
		ev_Error( "no block for <if>" );

	--indentLevel;
	EV_DEBUG(2,("--> leaving If\n"));
	return 0;
}

int
ev_Repeat()
{
EXPR_PTR ep;

	EV_DEBUG(2,("--> entering Repeat\n"));
	++indentLevel;
	inRepeat = 1;
	
	append_instr( curcode, (long) T_repeat );
	ev_Scan();
	if( theToken.type == T_begblock )
	{
		ev_Block();
		ev_Scan();
		if( theToken.type != T_while && theToken.type != T_until )
			ev_Error( "<repeat> block lacking <while> or <until> at bottom" );
		append_instr( curcode, (long) theToken.type );
		ev_Scan();
		if( theToken.type != T_begexpr )
			ev_Error( "<repeat> block needs Expression at bottom" );
		ep = ev_Expression();
		append_instr( curcode, (long) ep );
		ev_Scan();
		if( theToken.type != T_cmdend )
			ev_Error( "<repeat> block expects ';' after Expression at bottom" );
		append_instr( curcode, (long) T_cmdend );
	}
	else if( theToken.type == T_while || theToken.type == T_until )
	{
		append_instr( curcode, (long) theToken.type );
		ev_Scan();
		if( theToken.type != T_begexpr )
			ev_Error( "<repeat> block needs Expression at top" );
		ep = ev_Expression();
		append_instr( curcode, (long) ep );
		ev_Scan();
		if( theToken.type != T_begblock )
			ev_Error( "<repeat> needs block" );
		ev_Block();
	}
	else
		ev_Error( "syntax error in <repeat>" );
	
	inRepeat = 0;
	--indentLevel;
	EV_DEBUG(2,("--> leaving Repeat\n"));
	return 0;
}

int
ev_Command()
{
VAR_PTR	vp;
EXPR_PTR ep;
FCALL_PTR fcp;
int	dontadd = 0;

	EV_DEBUG(2,("--> entering Command\n"));
	++indentLevel;

	if( theToken.type == T_return )
	{
		if( !inFunction )
			ev_Error( "<return> not in Function" );
		append_instr( curcode, (long) T_return );
		ev_Scan();
		if( theToken.type == T_begexpr )
		{
			ep = ev_Expression();
			append_instr( curcode, (long) ep );
		}
		else if( theToken.type == T_int || theToken.type == T_float 
			|| theToken.type == T_word || theToken.type == T_string )
		{
			if( theToken.type == T_word )
			{
				if( isFcall() )
				{
					fcp = ev_Fcall();
					append_instr( curcode, (long) fcp );
				}
				else
				{
					vp = makevar();
					append_instr( curcode, (long) vp );
				}
			}	
			else
			{
				vp = makevar();
				append_instr( curcode, (long) vp );
			}
		}
		else
			ev_Error( "<return> needs Expression, Int, Flt, String, Word, or Fcall" );
	}
	else if( theToken.type == T_put )
	{
		append_instr( curcode, (long) T_put );
		ev_Scan();
		if( theToken.type == T_begexpr )
		{
			ep = ev_Expression();
			append_instr( curcode, (long) ep );
		}
		else if( theToken.type == T_int || theToken.type == T_float
			|| theToken.type == T_word || theToken.type == T_string )
		{
			if( theToken.type == T_word )
			{
				if( isFcall() )
				{
					fcp = ev_Fcall();
					append_instr( curcode, (long) fcp );
				}
				else
				{
					vp = makevar();
					append_instr( curcode, (long) vp );
				}
			}	
			else
			{
				vp = makevar();
				append_instr( curcode, (long) vp );
			}
		}
		else
			ev_Error( "<put> needs Expression, var, const, or Fcall" );

		ev_Scan();
		if( theToken.type != T_into )
			ev_Error( "<put> command missing <into>" );
		append_instr( curcode, (long) T_into );
		ev_Scan();
		if( theToken.type != T_word )
			ev_Error( "<put> command needs var for destination" );
		vp = makevar();
		append_instr( curcode, (long) vp );
	}
	else if( theToken.type == T_next )
	{
		if( !inRepeat )
			ev_Error( "<next> not in <repeat>" );
		append_instr( curcode, (long) T_next );
		ev_Scan();
		if( theToken.type != T_repeat )
			ev_Error( "<next> not followed by <repeat>" );
		append_instr( curcode, (long) T_repeat );
	}
	else if( theToken.type == T_exit )
	{
		append_instr( curcode, (long) T_exit );
		ev_Scan();
		if( theToken.type == T_word )
		{
			if( !inMsgHandler )
				ev_Error( "<exit> not in MsgHandler" );
			if( strcmp(theToken.value,curcode->name) )
				ev_Error( "<exit> name doesn't match current MsgHandler" );
			vp = makevar();
			append_instr( curcode, (long) vp );			
		}
		else if( theToken.type == T_repeat )
		{
			if( !inRepeat )
				ev_Error( "<exit repeat> not in <repeat>" );
			append_instr( curcode, (long) T_repeat );
		}
		else		
			ev_Error( "<exit> not followed by <repeat> or MsgHandler" );
	}
	else if( theToken.type == T_send )
	{
		append_instr( curcode, (long) T_send );
		ev_Scan();
		if( theToken.type != T_word && theToken.type != T_string )
			ev_Error( "<send> command needs Var or String" );
		vp = makevar();
		append_instr( curcode, (long) vp );
		ev_Scan();
		if( theToken.type != T_to )
			ev_Error( "<send> command needs <to>" );
		append_instr( curcode, (long) T_to );
		ev_Scan();
		if( theToken.type != T_word && theToken.type != T_string )
			ev_Error( "<send> needs var or string destination" );
		vp = makevar();
		append_instr( curcode, (long) vp );
	}
	else if( theToken.type == T_static )
	{
		ev_Scan();
		if( theToken.type != T_word )
			ev_Error( "static command needs variable name" );
		vp = makevar();
		vp->IsStatic = 1;
		dontadd = 1;	/* don't put this in the code! */
	}
	else if( theToken.type == T_word )
	{
		if( isFcall() )
		{
			fcp = ev_Fcall();
			append_instr( curcode, (long) fcp );
		}
		else
			ev_Error( "found variable, expected fcall, in Command" );
	}
	else
		ev_Error( "unknown command" );
		
	ev_Scan();
	if( theToken.type != T_cmdend )
		ev_Error( "Command followed by illegal token; expected ';'" );
	if( !dontadd )
		append_instr( curcode, (long) T_cmdend );

	--indentLevel;
	EV_DEBUG(2,("--> leaving Command\n"));
	return 0;
}

EXPR_PTR
ev_Expression()
{
static int	exprDepth = 0;
EXPR_PTR	ep = NULL;
VAR_PTR		vp = NULL;
FCALL_PTR	fcp = NULL;
int		got_end = 0;

	EV_DEBUG(2,("--> entering Expression, depth %d\n",exprDepth));
	++exprDepth;
	if( exprDepth > 10 )
		ev_Error( "expression too deep" );
	++indentLevel;
	
	ep = newexpr();
	ep->depth = exprDepth;
	
	ev_Scan();

	if( theToken.type == T_begexpr )
	{
		ep->left = (char *) ev_Expression();
		ev_Scan();
		if( IsOperator(theToken.type) && theToken.type <= T_SEP_OPS )
		{
			ep->op = theToken.type;
			if( IsUnaryOp(ep->op) && ep->op != T_minus )
				ev_Error( "unary operator used in binary expression" );
			ev_Scan();
			if( theToken.type == T_begexpr )
				ep->right = (char *) ev_Expression();
			else if( theToken.type == T_int || theToken.type == T_float
				|| theToken.type == T_word || theToken.type == T_string )
			{
				if( theToken.type == T_word )
				{
					if( isFcall() )
					{
						fcp = ev_Fcall();
						ep->right = (char *) fcp;
					}
					else
					{
						vp = makevar();
						ep->right = (char *) vp;
					}
				}	
				else
				{
					vp = makevar();
					ep->right = (char *) vp;
				}
			}
			else	
				ev_Error( "bad expression syntax; not expr or value" );		
		}
	}
	else if( IsOperator(theToken.type) && theToken.type != T_endexpr )
	{
		if( IsUnaryOp( theToken.type ) )	/* T_minus or T_not */
		{
			ep->op = theToken.type;
			ev_Scan();
			if( theToken.type != T_int && theToken.type != T_float 
				&& theToken.type != T_word && theToken.type!=T_begexpr )
				ev_Error( "target of unary '-' operator is not value" );
			ep->left = (char *) NULL;
			if( theToken.type == T_word )
			{
				if( isFcall() )
				{
					fcp = ev_Fcall();
					ep->right = (char *) fcp;
				}
				else
				{
					vp = makevar();
					ep->right = (char *) vp;
				}
			}
			else if( theToken.type == T_begexpr )
			{
				ep->right = (char *) ev_Expression();
			}	
			else  
			{
				vp = makevar();
				ep->right = (char *) vp;
			}
		}
		else
			ev_Error( "invalid expression; bad unary operator" );
	}
	else if( theToken.type == T_int || theToken.type == T_float
		|| theToken.type == T_word || theToken.type == T_string )
	{
		if( theToken.type == T_word )
		{
			if( isFcall() )
			{
				fcp = ev_Fcall();
				ep->left = (char *) fcp;
			}
			else
			{
				vp = makevar();
				ep->left = (char *) vp;
			}
		}	
		else
		{
			vp = makevar();
			ep->left = (char *) vp;
		}
		ev_Scan();
		if( IsOperator(theToken.type) && theToken.type <= T_SEP_OPS )
		{
			ep->op = theToken.type;
			if( IsUnaryOp(ep->op) && ep->op != T_minus )
				ev_Error( "unary operator used in binary expression" );
			ev_Scan();
			if( theToken.type == T_begexpr )
				ep->right = (char *) ev_Expression();
			else if( theToken.type == T_int || theToken.type == T_float
				|| theToken.type == T_word || theToken.type == T_string )
			{
				if( theToken.type == T_word )
				{
					if( isFcall() )
					{
						fcp = ev_Fcall();
						ep->right = (char *) fcp;
					}
					else
					{
						vp = makevar();
						ep->right = (char *) vp;
					}
				}	
				else
				{
					vp = makevar();
					ep->right = (char *) vp;
				}
			}
			else
				ev_Error( "invalid numeric expression" );
		}
		else if( theToken.type == T_endexpr )
			got_end = 1;
		else
			ev_Error( "invalid expression syntax" );
	}
	else
		ev_Error( "invalid expression syntax" );

	if( !got_end )
		ev_Scan();
	if( theToken.type != T_endexpr )
		ev_Error( "invalid expression syntax" );
	
	--indentLevel;
	--exprDepth;
	EV_DEBUG(2,("--> leaving Expression\n"));
	return ep;
}

int
ev_Block()
{
int can_else;

	EV_DEBUG(2,("--> entering Block\n"));
	++indentLevel;

	append_instr( curcode, (long) T_begblock );	
	
	ev_Scan();
	do
	{
		if( theToken.type == T_if )
		{
			ev_If();
			can_else = 1;
		}
		else if( theToken.type == T_else )
		{
			if( !can_else )
				ev_Error( "<else> without <if>" );
			append_instr( curcode, (long) T_else );
			ev_Scan();
			if( theToken.type == T_if )
			{
				ev_If();
				can_else = 1;
			}
			else
			{
				if( theToken.type != T_begblock )
					ev_Error( "no block for <else>" );
				else
					ev_Block();
				can_else = 0;
			}
		}
		else if( theToken.type == T_repeat )
		{
			ev_Repeat();
			can_else = 0;
		}
		else if( theToken.type == T_endblock )
		{
			break;
		}
		else
		{
			ev_Command();
			can_else = 0;
		}
		ev_Scan();
	} while( theToken.type != T_endblock );
	
	append_instr( curcode, (long) T_endblock );

	--indentLevel;
	EV_DEBUG(2,("--> leaving Block\n"));
	return 0;
}

int
isFcall()	/* look ahead to see if the T_word just found is a function call */
{
int	ret = 0;
long 	pos;
TOKEN 	tok;
char	line[256];
char *	loc = theLoc;
int	linenum = theLineNo;
int	theeof = eof;
long	ftell();
int	save_dbug = ev_Debug;	/* save & restore current state */

	/* save the state of important global variables, and file pos */

	ev_Debug = 0;

	pos = ftell(theFile);
	strcpy(line,theLine);
	bcopy( &theToken, &tok, sizeof(TOKEN) );

	ev_Scan();
	if( theToken.type == T_begexpr )
		ret = 1;

	/* restore the state */	

	bcopy( &tok, &theToken, sizeof(TOKEN) );	
	fseek(theFile,pos,0);
	eof = theeof;
	theLineNo = linenum;
	theLoc = loc;
	strcpy(theLine,line);

	ev_Debug = save_dbug;

	return ret;
}	
