/* $Id: conditional.c,v 1.1 2003/01/11 06:27:25 lcs Exp $
 * Expression parser and primitives to handle conditionalized templates
 * in cgiemail/cgiecho.
 * Stolen from  mod_include.c in the Apache 1.3 source, with credits below:
 */

/* UNCOMMENT this line to enable debugging output to stderr: */
/* #define DEBUG_INCLUDE 1 */

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2002 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" must
 *    not be used to endorse or promote products derived from this
 *    software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    nor may "Apache" appear in their name, without prior written
 *    permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * Portions of this software are based upon public domain software
 * originally written at the National Center for Supercomputing Applications,
 * University of Illinois, Urbana-Champaign.
 */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <regex.h>
#include <ctype.h>

/* cgiemail data structures */
#include "cgi.h"

#define HUGE_STRING_LEN  8192
#define MAX_STRING_LEN HUGE_STRING_LEN


/* compares string to regular expression
 */
static int re_check(cgi_form *formp, char *string, char *rexp)
{
    int regex_error;
    regex_t compiled;

#ifdef DEBUG_INCLUDE
    fprintf(stderr, "cgilib.c: regexp trying \'%s\' = /%s/\n", string, rexp);
#endif
    if (regcomp(&compiled, rexp, REG_EXTENDED | REG_NOSUB)) {
        cgi_seterror(formp, 500, "unable to compile regular expression pattern", rexp);
        return -1;
    }
    regex_error = regexec(&compiled, string, 0, (regmatch_t *) NULL, 0);
    regfree(&compiled);
    return (regex_error == 0);
}

enum token_type {
    token_string, token_variable,
    token_and, token_or, token_not, token_eq, token_ne,
    token_rbrace, token_lbrace, token_group,
    token_ge, token_le, token_gt, token_lt
};
struct token {
    enum token_type type;
    char value[MAX_STRING_LEN];
};

/* there is an implicit assumption here that string is at most MAX_STRING_LEN-1
 * characters long...
 */
static const char *get_ptoken(cgi_form *formp, const char *string, struct token *token)
{
    char ch;
    int next = 0;
    int qs = 0;

    /* Skip leading white space */
    if (string == (char *) NULL) {
        return (char *) NULL;
    }
    while ((ch = *string++)) {
        if (!isspace(ch)) {
            break;
        }
    }
    if (ch == '\0') {
        return (char *) NULL;
    }

    token->type = token_string; /* the default type */
    switch (ch) {
    case '(':
        token->type = token_lbrace;
        return (string);
    case ')':
        token->type = token_rbrace;
        return (string);
     /* HACK: accept both a single "=" and "==" as the equals rel-op. */
    case '=':
        token->type = token_eq;
        if (*string == '=')
            ++string;
        return (string);
    case '!':
        if (*string == '=') {
            token->type = token_ne;
            return (string + 1);
        }
        else {
            token->type = token_not;
            return (string);
        }
    case '\'':
    case '\"':
        token->type = token_string;
        qs = (int)ch;
        break;
    case '|':
        if (*string == '|') {
            token->type = token_or;
            return (string + 1);
        }
        break;
    case '&':
        if (*string == '&') {
            token->type = token_and;
            return (string + 1);
        }
        break;
    case '>':
        if (*string == '=') {
            token->type = token_ge;
            return (string + 1);
        }
        else {
            token->type = token_gt;
            return (string);
        }
    case '<':
        if (*string == '=') {
            token->type = token_le;
            return (string + 1);
        }
        else {
            token->type = token_lt;
            return (string);
        }
    default:
        token->type = token_string;
        break;
    }
    /* We should only be here if we are in a string */

    /* If string did not start with a quote, it is a variable reference. */
    if (!qs) {
        token->value[next++] = ch;
        token->type = token_variable;
    }

    /*
     * Yes I know that goto's are BAD.  But, c doesn't allow me to
     * exit a loop from a switch statement.  Yes, I could use a flag,
     * but that is (IMHO) even less readable/maintainable than the goto.
     */
    /*
     * I used the ++string throughout this section so that string
     * ends up pointing to the next token and I can just return it
     */
    for (ch = *string; ch != '\0'; ch = *++string) {
        if (ch == '\\') {
            if ((ch = *++string) == '\0') {
                goto TOKEN_DONE;
            }
            token->value[next++] = ch;
            continue;
        }
        if (!qs) {
            if (isspace(ch)) {
                goto TOKEN_DONE;
            }
            switch (ch) {
            case '(':
                goto TOKEN_DONE;
            case ')':
                goto TOKEN_DONE;
            case '=':
                goto TOKEN_DONE;
            case '!':
                goto TOKEN_DONE;
            case '|':
                if (*(string + 1) == '|') {
                    goto TOKEN_DONE;
                }
                break;
            case '&':
                if (*(string + 1) == '&') {
                    goto TOKEN_DONE;
                }
                break;
            case '<':
                goto TOKEN_DONE;
            case '>':
                goto TOKEN_DONE;
            }
            token->value[next++] = ch;
        }
        else {
            if (ch == (char)qs) {
                qs = 0;
                ++string;
                goto TOKEN_DONE;
            }
            token->value[next++] = ch;
        }
    }
  TOKEN_DONE:
    /* If qs is still set, I have an unmatched ' */
    if (qs) {
        fprintf(stderr, "cgilib.c: Unmatched quote (%c) in string in #if\n", qs);
        cgi_seterror(formp, 403, "Invalid #if expression, unmatched quote in string.", NULL);
        next = 0;
    }
    token->value[next] = '\0';

    /* Look up form or environment variable if not a quoted string: */
    if (token->type == token_variable && token->value[0] != '/') {
        int len = sizeof(token->value);
        char *val;

#ifdef DEBUG_INCLUDE
        fprintf(stderr, "cgilib.c: get_ptoken: Looking up var reference \"%s\"\n", token->value);
#endif
        if (token->value[0] == '$') {
            val = getenv(&token->value[1]);
            if (val == NULL) val = "";
        } else {
            val = cgi_value(formp, token->value);
            if (cgi_required(token->value) && !cgi_nonblank(val)) {
                cgi_seterror(formp, 400, "Required field left blank", token->value);
                token->value[0] = '\0';
                return NULL;
            }
        }
        strncpy(token->value, val, len);
        token->value[len-1] = '\0';
    }
    if (token->type == token_variable)
        token->type = token_string;
    return (string);
}


/*
 * Hey I still know that goto's are BAD.  I don't think that I've ever
 * used two in the same project, let alone the same file before.  But,
 * I absolutely want to make sure that I clean up the memory in all
 * cases.  And, without rewriting this completely, the easiest way
 * is to just branch to the return code which cleans it up.
 */
/* there is an implicit assumption here that expr is at most MAX_STRING_LEN-1
 * characters long...
 */
int parse_expr(cgi_form *formp, const char *expr, const char *error)
{
    struct parse_node {
        struct parse_node *left, *right, *parent;
        struct token token;
        int value, done;
    }         *root, *current, *new;
    const char *parse;
    char buffer[MAX_STRING_LEN];
    /* pool *expr_pool; */
    int retval = 0;

    if ((parse = expr) == (char *) NULL) {
        return (0);
    }
    root = current = (struct parse_node *) NULL;

    /* Create Parse Tree */
    while (1) {
        new = (struct parse_node *) malloc(sizeof(struct parse_node));
        new->parent = new->left = new->right = (struct parse_node *) NULL;
        new->done = 0;
        if ((parse = get_ptoken(formp, parse, &new->token)) == (char *) NULL) {
            break;
        }
        switch (new->token.type) {

        case token_string:
#ifdef DEBUG_INCLUDE
            fprintf(stderr, "cgilib.c: parse_expr: Token: string (%s)\n", new->token.value);
#endif
            if (current == (struct parse_node *) NULL) {
                root = current = new;
                break;
            }
            switch (current->token.type) {
            case token_string:
                if (current->token.value[0] != '\0') {
                    strncat(current->token.value, " ",
                         sizeof(current->token.value)
                            - strlen(current->token.value) - 1);
                }
                strncat(current->token.value, new->token.value,
                         sizeof(current->token.value)
                            - strlen(current->token.value) - 1);
                current->token.value[sizeof(current->token.value) - 1] = '\0';
                break;
            case token_eq:
            case token_ne:
            case token_and:
            case token_or:
            case token_lbrace:
            case token_not:
            case token_ge:
            case token_gt:
            case token_le:
            case token_lt:
                new->parent = current;
                current = current->right = new;
                break;
            default:
                cgi_seterror(formp, 403, "Invalid #if expression",expr);
                goto RETURN;
            }
            break;

        case token_and:
        case token_or:
#ifdef DEBUG_INCLUDE
            fprintf(stderr, "cgilib.c: parse_expr: Token: and/or\n");
#endif
            if (current == (struct parse_node *) NULL) {
                cgi_seterror(formp, 403, "Invalid #if expression",expr);
                goto RETURN;
            }
            /* Percolate upwards */
            while (current != (struct parse_node *) NULL) {
                switch (current->token.type) {
                case token_string:
                case token_group:
                case token_not:
                case token_eq:
                case token_ne:
                case token_and:
                case token_or:
                case token_ge:
                case token_gt:
                case token_le:
                case token_lt:
                    current = current->parent;
                    continue;
                case token_lbrace:
                    break;
                default:
                    cgi_seterror(formp, 403, "Invalid #if expression",expr);
                    goto RETURN;
                }
                break;
            }
            if (current == (struct parse_node *) NULL) {
                new->left = root;
                new->left->parent = new;
                new->parent = (struct parse_node *) NULL;
                root = new;
            }
            else {
                new->left = current->right;
                current->right = new;
                new->parent = current;
            }
            current = new;
            break;

        case token_not:
#ifdef DEBUG_INCLUDE
            fprintf(stderr, "cgilib.c: parse_expr: Token: not\n");
#endif
            if (current == (struct parse_node *) NULL) {
                root = current = new;
                break;
            }
            /* Percolate upwards */
            while (current != (struct parse_node *) NULL) {
                switch (current->token.type) {
                case token_not:
                case token_eq:
                case token_ne:
                case token_and:
                case token_or:
                case token_lbrace:
                case token_ge:
                case token_gt:
                case token_le:
                case token_lt:
                    break;
                default:
                    cgi_seterror(formp, 403, "Invalid #if expression",expr);
                    goto RETURN;
                }
                break;
            /* NOTREACHED */
            }
            if (current == (struct parse_node *) NULL) {
                new->left = root;
                new->left->parent = new;
                new->parent = (struct parse_node *) NULL;
                root = new;
            }
            else {
                new->left = current->right;
                current->right = new;
                new->parent = current;
            }
            current = new;
            break;

        case token_eq:
        case token_ne:
        case token_ge:
        case token_gt:
        case token_le:
        case token_lt:
#ifdef DEBUG_INCLUDE
            fprintf(stderr, "cgilib.c: parse_expr: Token: eq/ne/ge/gt/le/lt\n");
#endif
            if (current == (struct parse_node *) NULL) {
                cgi_seterror(formp, 403, "Invalid #if expression",expr);
                goto RETURN;
            }
            /* Percolate upwards */
            while (current != (struct parse_node *) NULL) {
                switch (current->token.type) {
                case token_string:
                case token_group:
                    current = current->parent;
                    continue;
                case token_lbrace:
                case token_and:
                case token_or:
                    break;
                case token_not:
                case token_eq:
                case token_ne:
                case token_ge:
                case token_gt:
                case token_le:
                case token_lt:
                default:
                    cgi_seterror(formp, 403, "Invalid #if expression",expr);
                    goto RETURN;
                }
                break;
            }
            if (current == (struct parse_node *) NULL) {
                new->left = root;
                new->left->parent = new;
                new->parent = (struct parse_node *) NULL;
                root = new;
            }
            else {
                new->left = current->right;
                current->right = new;
                new->parent = current;
            }
            current = new;
            break;

        case token_rbrace:
#ifdef DEBUG_INCLUDE
            fprintf(stderr, "cgilib.c: parse_expr: Token: rbrace\n");
#endif
            while (current != (struct parse_node *) NULL) {
                if (current->token.type == token_lbrace) {
                    current->token.type = token_group;
                    break;
                }
                current = current->parent;
            }
            if (current == (struct parse_node *) NULL) {
                cgi_seterror(formp, 403, "Unmatched ')' in expression",expr);
                goto RETURN;
            }
            break;

        case token_lbrace:
#ifdef DEBUG_INCLUDE
            fprintf(stderr, "cgilib.c: parse_expr: Token: lbrace\n");
#endif
            if (current == (struct parse_node *) NULL) {
                root = current = new;
                break;
            }
            /* Percolate upwards */
            while (current != (struct parse_node *) NULL) {
                switch (current->token.type) {
                case token_not:
                case token_eq:
                case token_ne:
                case token_and:
                case token_or:
                case token_lbrace:
                case token_ge:
                case token_gt:
                case token_le:
                case token_lt:
                    break;
                case token_string:
                case token_group:
                default:
                    cgi_seterror(formp, 403, "Invalid #if expression",expr);
                    goto RETURN;
                }
                break;
            /* NOTREACHED */
            }
            if (current == (struct parse_node *) NULL) {
                new->left = root;
                new->left->parent = new;
                new->parent = (struct parse_node *) NULL;
                root = new;
            }
            else {
                new->left = current->right;
                current->right = new;
                new->parent = current;
            }
            current = new;
            break;
        default:
            break;
        }
    }

    /* if lexing generated fatal error, get out now. */
    if (formp->errcond)
        goto RETURN;

    /* Evaluate Parse Tree */
    current = root;
    while (current != (struct parse_node *) NULL) {
        switch (current->token.type) {
        case token_string:
#ifdef DEBUG_INCLUDE
            fprintf(stderr, "cgilib.c: parse_expr: Evaluate string\n");
#endif
            /* XXX no embedded vars
            parse_string(r, current->token.value, buffer, sizeof(buffer), 0);
            ap_cpystrn(current->token.value, buffer, sizeof(current->token.value));
            */
            current->value = (current->token.value[0] != '\0');
            current->done = 1;
            current = current->parent;
            break;

        case token_and:
        case token_or:
#ifdef DEBUG_INCLUDE
            fprintf(stderr, "cgilib.c: parse_expr: Evaluate and/or\n");
#endif
            if (current->left == (struct parse_node *) NULL ||
                current->right == (struct parse_node *) NULL) {
                cgi_seterror(formp, 403, "Invalid #if expression",expr);
                goto RETURN;
            }
            if (!current->left->done) {
                switch (current->left->token.type) {
                case token_string:
                    /* XXX parse_string(r, current->left->token.value,
                                 buffer, sizeof(buffer), 0);
                    ap_cpystrn(current->left->token.value, buffer,
                            sizeof(current->left->token.value));
                    */
                    current->left->value = (current->left->token.value[0] != '\0');
                    current->left->done = 1;
                    break;
                default:
                    current = current->left;
                    continue;
                }
            }
            if (!current->right->done) {
                switch (current->right->token.type) {
                case token_string:
                    /* XXX parse_string(r, current->right->token.value,
                                 buffer, sizeof(buffer), 0);
                    ap_cpystrn(current->right->token.value, buffer,
                            sizeof(current->right->token.value));
                    */
                    current->right->value = (current->right->token.value[0] != '\0');
                    current->right->done = 1;
                    break;
                default:
                    current = current->right;
                    continue;
                }
            }
#ifdef DEBUG_INCLUDE
            fprintf(stderr, "cgilib.c: parse_expr:  Left: %d; Right: %d.\n",
                 (current->left->value ? 1 : 0),
                 (current->right->value ? 1 : 0));
#endif
            if (current->token.type == token_and) {
                current->value = current->left->value && current->right->value;
            }
            else {
                current->value = current->left->value || current->right->value;
            }
#ifdef DEBUG_INCLUDE
            fprintf(stderr, "cgilib.c: parse_expr:  Returning %d\n", (current->value != 0));
#endif
            current->done = 1;
            current = current->parent;
            break;

        case token_eq:
        case token_ne:
#ifdef DEBUG_INCLUDE
            fprintf(stderr, "cgilib.c: parse_expr: Evaluate eq/ne\n");
#endif
            if ((current->left == (struct parse_node *) NULL) ||
                (current->right == (struct parse_node *) NULL) ||
                (current->left->token.type != token_string) ||
                (current->right->token.type != token_string)) {
                cgi_seterror(formp, 403, "Invalid #if expression",expr);
                goto RETURN;
            }
            /* XXX parse_string(r, current->left->token.value,
                         buffer, sizeof(buffer), 0);
            ap_cpystrn(current->left->token.value, buffer,
                        sizeof(current->left->token.value));
            */
            /* XXX parse_string(r, current->right->token.value,
                         buffer, sizeof(buffer), 0);
            ap_cpystrn(current->right->token.value, buffer,
                        sizeof(current->right->token.value));
            */
            if (current->right->token.value[0] == '/') {
                int len;
                len = strlen(current->right->token.value);
                if (current->right->token.value[len - 1] == '/') {
                    current->right->token.value[len - 1] = '\0';
                }
                else {
                cgi_seterror(formp, 403, "Invalid regular expression",current->right->token.value);
                    goto RETURN;
                }
#ifdef DEBUG_INCLUDE
            fprintf(stderr, "cgilib.c: parse_expr:  RE Compare '%s' with /%s/\n",
                current->left->token.value, &current->right->token.value[1]);
#endif
                current->value =
                    re_check(formp, current->left->token.value,
                             &current->right->token.value[1]);
            }
            else {
#ifdef DEBUG_INCLUDE
            fprintf(stderr, "cgilib.c: parse_expr:  Compare (%s) with (%s)\n",
                current->left->token.value, current->right->token.value);
#endif
                current->value =
                    (strcmp(current->left->token.value,
                            current->right->token.value) == 0);
            }
            if (current->token.type == token_ne) {
                current->value = !current->value;
            }
#ifdef DEBUG_INCLUDE
            fprintf(stderr, "cgilib.c: parse_expr:  Returning %d\n", (current->value != 0));
#endif
            current->done = 1;
            current = current->parent;
            break;
        case token_ge:
        case token_gt:
        case token_le:
        case token_lt:
#ifdef DEBUG_INCLUDE
            fprintf(stderr, "cgilib.c: parse_expr: Evaluate ge/gt/le/lt\n");
#endif
            if ((current->left == (struct parse_node *) NULL) ||
                (current->right == (struct parse_node *) NULL) ||
                (current->left->token.type != token_string) ||
                (current->right->token.type != token_string)) {
                cgi_seterror(formp, 403, "Invalid #if expression",expr);
                goto RETURN;
            }
            /* XXX parse_string(r, current->left->token.value,
                         buffer, sizeof(buffer), 0);
            ap_cpystrn(current->left->token.value, buffer,
                        sizeof(current->left->token.value));
            */
            /* XXX parse_string(r, current->right->token.value,
                         buffer, sizeof(buffer), 0);
            ap_cpystrn(current->right->token.value, buffer,
                        sizeof(current->right->token.value));
            */
#ifdef DEBUG_INCLUDE
            fprintf(stderr, "cgilib.c: parse_expr:  Compare (%s) with (%s)\n",
                current->left->token.value, current->right->token.value);
#endif
            current->value =
                strcmp(current->left->token.value,
                       current->right->token.value);
            if (current->token.type == token_ge) {
                current->value = current->value >= 0;
            }
            else if (current->token.type == token_gt) {
                current->value = current->value > 0;
            }
            else if (current->token.type == token_le) {
                current->value = current->value <= 0;
            }
            else if (current->token.type == token_lt) {
                current->value = current->value < 0;
            }
            else {
                current->value = 0;     /* Don't return -1 if unknown token */
            }
#ifdef DEBUG_INCLUDE
            fprintf(stderr, "cgilib.c: parse_expr:  Returning %d\n", (current->value != 0));
#endif
            current->done = 1;
            current = current->parent;
            break;

        case token_not:
            if (current->right != (struct parse_node *) NULL) {
                if (!current->right->done) {
                    current = current->right;
                    continue;
                }
                current->value = !current->right->value;
            }
            else {
                current->value = 0;
            }
#ifdef DEBUG_INCLUDE
            fprintf(stderr, "cgilib.c: parse_expr:  Evaluate !: => %d\n", current->value != 0);
#endif
            current->done = 1;
            current = current->parent;
            break;

        case token_group:
            if (current->right != (struct parse_node *) NULL) {
                if (!current->right->done) {
                    current = current->right;
                    continue;
                }
                current->value = current->right->value;
            }
            else {
                current->value = 1;
            }
#ifdef DEBUG_INCLUDE
            fprintf(stderr, "cgilib.c: parse_expr:  Evaluate (): => %d\n", current->value != 0);
#endif
            current->done = 1;
            current = current->parent;
            break;

        case token_lbrace:
            cgi_seterror(formp, 403, "Unmatched '(' in expression", expr);
            goto RETURN;

        case token_rbrace:
            cgi_seterror(formp, 403, "Unmatched ')' in expression", expr);
            goto RETURN;

        default:
            cgi_seterror(formp, 403, "bad token type in expression", NULL);
            goto RETURN;
        }
    }

    retval = (root == (struct parse_node *) NULL) ? 0 : root->value;
  RETURN:
    return (retval);
}
