#if (!defined(lint) && !defined(SABER))
static char rcsid[] = "$Header: /afs/athena.mit.edu/astaff/project/olhdev/src/curses/RCS/ui_curses.c,v 1.6 90/08/28 09:22:31 lwvanels Exp $";
#endif


#include <curses.h>
#include <signal.h>
#include <strings.h>
#include "menu.h"
#include "menupath.h"
#include "olh_curses.h"

#if defined(ultrix) && defined(mips)
#define cbreak() crmode()
#define nocbreak() nocrmode()
#endif


#define max(a,b)	((a) > (b)) ? (a) : (b)
#define min(a,b)	((a) < (b)) ? (a) : (b)


/*
 *  Global data
 */

Menu *menu[512];
MenuEntry *menu_entry[512];

short current[512];
short viewed[512];
short current_menu = 0;

MenuEntry *top_entry;
char *top_pointer;

void show_message(), OLH_ui_message(), select();

extern char *malloc(), *realloc();
extern void (*olh_attach_hook)();
extern void (*olh_attach_done_hook)();

char *program;
int hist_on;

void main(), MainLoop(), OLH_ui_help_primary(), OLH_ui_help_keyword(),
  attach_hook(), attach_done_hook();

/*
 *  Static data
 */

/*  Screens  */

#define SCR_MAX		3
#define SCR_PRIM	0
#define SCR_VIEW	1
#define SCR_SUGG	2

static int scount;
static WINDOW* scr[SCR_MAX];

static WINDOW* scr_prim;
static WINDOW* scr_view;
static WINDOW* scr_sugg;

static int current_screen;


/*  Windows  */

#define WIN_MAX		20

static int wcount;
static WINDOW* win[WIN_MAX];

static WINDOW* win_heading;
static WINDOW* win_instr;
static WINDOW* win_label;
static WINDOW* win_entries;
static WINDOW* win_above;
static WINDOW* win_below;
static WINDOW* win_prompt;
static WINDOW* win_message;
static WINDOW* win_view_wait;
static WINDOW* win_sugg_wait;


/*  Menu information  */

static int current_offset = 0;
static int selection = 0;
static int menu_lines;
static int prev_above;
static int prev_below;


/*  Prompt information  */

static char* current_prompt = NULL;
static char current_input[SMALL_BUFFER];


/*  Message information  */

static char* current_message = NULL;


/*  Screen size information  */

static int max_lines = 0;
static int max_cols = 0;


/*  Help information  */

#define HELP_PRIM	0
#define HELP_KWD	1

static	char* help_filename[] =
{
    "primary",
    "keyword",
};

static	char* help_text[] =
{
    NULL,
    NULL,
};

static	int num_help_topics = 2;


/*  Instructions  */

#define NUM_INSTR_LINES		5

#define INSTR_NORMAL		0
#define INSTR_NORMAL_TOP	1
#define INSTR_KEYWORD_NUM	2
#define INSTR_KEYWORD_WORD	3


static	char* instr_text[][NUM_INSTR_LINES] =
{
    {
	"",
	"To choose a topic, type its number and press the Return key.",
	"Press the '?' key for help.  Press the 'q' key to quit.",
	"To go up to the previous menu, press the 'u' key.",
	"",
    },
    {
	"",
	"To choose a topic, type its number and press the Return key.",
	"Press the '?' key for help.  Press the 'q' key to quit.",
	"",
	"",
    },
    {
	"To choose a keyword, type its number and press the Return key,",
	"or press 'k', then type the keyword itself and press the Return key.",
	"",
	"Press the '?' key for help.  Press the 'q' key to quit.",
	"To go up to the previous menu, press the 'u' key.",
    },
    {
	"",
	"Type a keyword and press the Return key.",
	"To get out of keyword-entry mode, press Ctrl-G.",
	"Press the '?' key for help.",
	"",
    },
};

static	int current_instr = INSTR_NORMAL_TOP;



/*----------------------------------------------------------------------------

  Internal operations

----------------------------------------------------------------------------*/

static	int curses_init( );
static	void curses_shutdown( );
static	int get_token( );
static	int get_number( );
static	int get_keyword( );
static	void delete_char_from_buffer( );
static	void sound_bell( );
static	void up_to_prev_menu( );
static	void scroll_above( );
static	void scroll_below( );
static	void erase_message( );
static	void view( );
static	void invoke_item( );
static	void update_instructions( );
static	void wait_for_key( );
static	void create_windows( );
static	void delete_windows( );
static	void paint_windows( );
static	void paint_instructions( );
static	void paint_menu( );
static	void paint_prompt( );
static	void paint_message( );
static	void paint_sugg( );
static	void set_current_screen( );
static	int display_file( );
static	int display_buffer( );
static	void take_suggestion( );
static	void show_help_message( );
/* static	void read_help_text( ); */
static	void invoke_olc( );
static	int handle_resize_event( );
static	int handle_interrupt_event( );
 


/****************************************************************
 * MAIN -- make interface and start loop			*
 ****************************************************************/
void
main(argc, argv)
     int argc;
     char **argv;
{
  long code;	/* Error code returned from libmenu routines */
  Menu *m;	/* Pointer to main menu */
  MenuEntry *e;	/* */

/*
 *  Find out what my name is today...  useful for printing in error
 *  messages, etc.
 */

  program = rindex(*argv,'/');
  if(program == (char *) NULL)
    program = *argv;
  if(*program == '/')
    ++program;

/*
 *  First, try creating display.  If this fails, print a 'nice' error
 *  message and exit.
 */

  if (! curses_init())
    exit(-2);

/*
 *  If creating display was successful, then initialize interface, etc.
 */
  current_input[0] = '\0';

  create_windows();

  menu_init(NULL);
  e = find_group("main");
  code = menu_load(e, &m);
  if (code) {
    com_err(program, code, "loading main menu", "");
    exit(1);
  }
  current_menu = 0;
  menu[current_menu] = m;
  menu_entry[current_menu] = e;
  current[current_menu] = 0;
  viewed[current_menu] = FALSE;
  add_to_menupath(e);

  paint_windows();
  show_message(COPYRIGHT);
  hist_on = FALSE;

/*
 * Set up signal handlers.
 */
  signal(SIGINT, handle_interrupt_event);
  signal(SIGWINCH, handle_resize_event);

/*
 * Go into MainLoop.  This does not exit.  Rather, the user exits the
 *  program by pressing "q" or through some other action.
 */
  olh_attach_hook = attach_hook;
  olh_attach_done_hook = attach_done_hook;
  MainLoop();
  curses_shutdown();
}



/*----------------------------------------------------------------------------

  MainLoop

----------------------------------------------------------------------------*/

void
MainLoop()
{
  int done = FALSE;
  int token, high, number;
  char buffer[SMALL_BUFFER];

  /*  Display the primary screen.  */

  set_current_screen(SCR_PRIM);

  while (! done)
    {
      token = get_token("Number: ");

      switch (token)
	{
	case TOKEN_NONE:
	  break;

	case TOKEN_EXIT:
	  done = TRUE;
	  break;

	case TOKEN_UP:
	  up_to_prev_menu();
	  break;

	case TOKEN_ABOVE:
	  scroll_above();
	  break;

	case TOKEN_BELOW:
	  scroll_below();
	  break;

	case TOKEN_HELP:
	  OLH_ui_help_primary();
	  break;

	case TOKEN_SUGGEST:
	  take_suggestion();
	  break;

	case TOKEN_OLC:
	  invoke_olc();
	  break;

	case TOKEN_NEXT:
	  view(1);
	  break;
		
	case TOKEN_CURR:
	  view(0);
	  break;
		
	case TOKEN_PREV:
	  view(-1);
	  break;
		
	case TOKEN_SEL_PREV:
	  select(-1);
	  break;
		
	case TOKEN_SEL_NEXT:
	  select(1);
	  break;
		
	default:
	  number = token - TOKEN_NUMBER;
	  high = size_menu(menu[current_menu]) - 1;
	  if ((number < 1) || (number > high))
	    {
	      sprintf(buffer, "Please choose a number from 1 to %d.", high);
	      OLH_ui_message(buffer);
	    }
	  else
	    {
	      current[current_menu] = number;
	      invoke_item(number);
	    }
	}
    }
}



/*----------------------------------------------------------------------------

  OLH_ui_update_menus

----------------------------------------------------------------------------*/

void
OLH_ui_update_menus( )
{
    paint_menu();
    wrefresh(win_label);
    wrefresh(win_above);
    wrefresh(win_entries);
    wrefresh(win_below);
}



/*----------------------------------------------------------------------------

  OLH_ui_message

----------------------------------------------------------------------------*/

void
OLH_ui_message( str )
char* str;
{
  if ((current_message == (char *) NULL) || strcmp(str, current_message))
    {
      current_message = str;
      paint_message( );
      wrefresh(win_message);
    }
}



/*----------------------------------------------------------------------------

  curses_init

----------------------------------------------------------------------------*/

static int 
curses_init( )
{
    if (initscr() == ERR)
      {
	  fprintf(stderr, "Unable to initialize the curses library.\n");
	  return FALSE;
      }

    if ((LINES < MIN_LINES) || (COLS < MIN_COLS))
      {
	  endwin();
	  printf("\nTo use the On-Line Help service, you need to have a\n");
	  printf("screen with at least %d columns by %d rows.\n\n",
		 MIN_COLS, MIN_LINES);
	  return FALSE;
      }

    max_lines = max(LINES, max_lines);
    max_cols = max(COLS, max_cols);

    noecho();
    cbreak();

    return TRUE;
}



/*----------------------------------------------------------------------------

  curses_shutdown

----------------------------------------------------------------------------*/

static
void curses_shutdown( )
{
    echo();
    nocbreak();
    endwin();
}



/*----------------------------------------------------------------------------

  get_token

----------------------------------------------------------------------------*/

static int
get_token( prompt_string )
     char* prompt_string;
{
    int token = TOKEN_NONE;
    int number_token;
    char ch;
    int arrow = 0;

    current_prompt = prompt_string;
    paint_prompt( );

    while (token == TOKEN_NONE)
      {
	wrefresh(win_prompt);
	ch = wgetch(win_prompt);
	erase_message();

	switch (ch)
	  {
	  case ESC:
	    arrow++;
	    break;

	  case '[':
	    arrow++;
	    break;

	  case 'A':
	    if (arrow==2)
	      {
		arrow=0;
		token = TOKEN_SEL_PREV;
	      }
	    break;

	  case 'B':
	    if (arrow==2)
	      {
		arrow=0;
		token = TOKEN_SEL_NEXT;
	      }
	    else
	      token = TOKEN_ABOVE;
	    break;

	  case 'C':
	    if (arrow==2)
	      {
		arrow=0;
		if (selection)
		  token = TOKEN_SELECT;
	      }
	    else
	      token = TOKEN_OLC;
	    break;

	  case 'D':
	    if (arrow==2)
	      {
		arrow=0;
		token = TOKEN_UP;
	      }
	    break;

	  case 'n':
	  case 'N':
	    token = TOKEN_NEXT;
	    break;

	  case 'v':
	  case '.':
	    token = TOKEN_CURR;
	    break;

	  case 'p':
	  case 'P':
	    token = TOKEN_PREV;
	    break;

	  case CTRL_L:
	    wrefresh(curscr);
	    break;

	  case 'u':
	  case 'U':
	  case CTRL_B:
	    token = TOKEN_UP;
	    break;

	  case '-':
	  case DELETE:
	  case 'b':
	    token = TOKEN_ABOVE;
	    break;

	  case '+':
	  case ' ':
	    token = TOKEN_BELOW;
	    break;

	  case '?':
	  case 'h':
	    token = TOKEN_HELP;
	    break;

	  case 's':
	  case 'S':
	    token = TOKEN_SUGGEST;
	    break;

	  case 'c':
	    token = TOKEN_OLC;
	    break;

	  case 'q':
	  case CTRL_C:
	    token = TOKEN_EXIT;
	    break;

	  case 'k':
	  case 'K':
	    if ((number_token = get_keyword()) != TOKEN_NONE)
	      token = number_token;
	    else
	      {
		current_prompt = prompt_string;
		paint_prompt( );
	      }
	    break;

	  case '1':
	  case '2':
	  case '3':
	  case '4':
	  case '5':
	  case '6':
	  case '7':
	  case '8':
	  case '9':
	    if ((number_token = get_number(ch)) != TOKEN_NONE)
	      token = number_token;
	    break;

	  case CTRL_U:
	    break;

	  case RETURN:
	  case NEWLINE:
	    if (selection)
	      token = TOKEN_SELECT;

	  default:
	    sound_bell();
	  }
      }

    wclear(win_prompt);
    wrefresh(win_prompt);
    return token;
}



/*----------------------------------------------------------------------------

  get_number

----------------------------------------------------------------------------*/

static int
get_number( ch )
     char ch;
{
  int done = FALSE;
  char* ptr = current_input;
  int token;

  *ptr++ = ch;
  waddch(win_prompt, ch);

  while (! done)
    {
      wrefresh(win_prompt);
      ch = wgetch(win_prompt);
      erase_message();

      switch (ch)
	{
	case CTRL_L:
	  wrefresh(curscr);
	  break;

	case '0':
	case '1':
	case '2':
	case '3':
	case '4':
	case '5':
	case '6':
	case '7':
	case '8':
	case '9':
	  if (ptr-current_input < MAX_DIGITS)
	    {
	      *ptr++ = ch;
	      waddch(win_prompt, ch);
	    }
	  else
	    sound_bell();
	  break;

	case DELETE:
	  delete_char_from_buffer(&ptr);
	  if (ptr == current_input)
	    {
	      strcpy(current_input, "");
	      return TOKEN_NONE;
	    }
	  break;

	case CTRL_U:
	  ptr = current_input;
	  *ptr = END_OF_STRING;
	  paint_prompt( );
	  return TOKEN_NONE;

	case NEWLINE:
	case RETURN:
	  *ptr = END_OF_STRING;
	  done = TRUE;
	  break;

	default:
	  sound_bell();
	}
      
    }

  token = TOKEN_NUMBER + atoi(current_input);
  strcpy(current_input, "");
  return token;
}



/*----------------------------------------------------------------------------

  get_keyword

----------------------------------------------------------------------------*/

static int
get_keyword( )
{
  int done = FALSE;
  char* ptr = current_input;
  char ch;
  int pos = -1;
  int token;
/*   char buffer[SMALL_BUFFER]; */

/*
  if (current_menu != keyword_menu)
    return TOKEN_NONE;
*/
  
  current_prompt = "Keyword: ";
  paint_prompt();
  current_instr = INSTR_KEYWORD_WORD;
  paint_instructions();
  wrefresh(win_instr);

  while (! done)
    {
      wrefresh(win_prompt);
      ch = wgetch(win_prompt);
      erase_message();
      
      switch (ch)
	{
	case CTRL_L:
	  wrefresh(curscr);
	  break;

	case CTRL_G:
	  strcpy(current_input, "");
	  paint_prompt( );
	  pos = -1;
	  done = TRUE;
	  break;
		
	case DELETE:
	  delete_char_from_buffer(&ptr);
	  break;

	case CTRL_U:
	  ptr = current_input;
	  *ptr = END_OF_STRING;
	  paint_prompt( );
	  break;

	case RETURN:
	case NEWLINE:
	  *ptr = END_OF_STRING;
	  if (!strncmp(current_input, ""))
	    {
	      sound_bell();
	      OLH_ui_message("Please type in a keyword to search for.");
	      break;
	    }
/*	  pos = OLH_find_keyword_index(current_input, FALSE);
	  if (pos > -1)
	    done = TRUE;
	  else
	    {
	      sprintf(buffer,
		      "More than one keyword begins with \"%s\".  ",
		      current_input);
	      strcat(buffer, "Please type more characters.");
	      sound_bell();
	      OLH_ui_message(buffer);
	    }
*/
	  break;

	default:
	  if ((ch >= ' ') && (ch <= '~'))
	    {
	      if (ptr-current_input < MAX_CHARS)
		{
		  *ptr++ = ch;
		  *ptr = END_OF_STRING;
/*		  pos = OLH_find_keyword_index(current_input, TRUE);
		  if (pos > -1)
		    {
		      waddch(win_prompt, ch);
		      current_offset = pos;
		      paint_menu( );
		      wrefresh(win_above);
		      wrefresh(win_entries);
		      wrefresh(win_below);
		    }
		  else
		    {
		      --ptr;
		      sprintf(buffer,
			      "No keywords begin with \"%s\".",
			      current_input);
		      sound_bell();
		      OLH_ui_message(buffer);
		    }
*/
		}
	      else
		sound_bell();
	    }
	  else
	    sound_bell();
	  break;
	}
    }
  
  token = (pos == -1) ? TOKEN_NONE : (TOKEN_NUMBER + pos + 1);
  strcpy(current_input, "");
  if (token == TOKEN_NONE)
    current_instr = INSTR_KEYWORD_NUM;
  paint_instructions();
  wrefresh(win_instr);
  return token;
}


/*----------------------------------------------------------------------------

  delete_char_from_buffer

----------------------------------------------------------------------------*/

static void
delete_char_from_buffer( ptr )
char** ptr;
{
    int x, y;

    if (*ptr <= current_input)
      {
	  sound_bell();
	  return;
      }

    *ptr = *ptr - 1;
    getyx(win_prompt, y, x);
    wmove(win_prompt, y, --x);
    wclrtoeol(win_prompt);
}



/*----------------------------------------------------------------------------

  sound_bell

----------------------------------------------------------------------------*/

static void
sound_bell( )
{
    printf("\007");
}



/*----------------------------------------------------------------------------

  up_to_prev_menu

----------------------------------------------------------------------------*/

static void
up_to_prev_menu( )
{
  if (current_menu == 0)
    {
      show_message(NO_MORE_UP);
      return;
    }
  current_menu--;
  del_last_menupath;
  current_offset = selection = 0;
  OLH_ui_update_menus();
  update_instructions();
}



/*----------------------------------------------------------------------------

  scroll_above

----------------------------------------------------------------------------*/

static void
scroll_above( )
{
    if (current_offset == 0)
      {
	  OLH_ui_message("No topics above.");
	  sound_bell();
	  return;
      }

    current_offset = max(0, current_offset - (menu_lines - 1));
    OLH_ui_update_menus();
}



/*----------------------------------------------------------------------------

  scroll_below

----------------------------------------------------------------------------*/

static void
scroll_below( )
{
    int high = size_menu(menu[current_menu]) - menu_lines - 1;

    if (current_offset >= high)
      {
	  OLH_ui_message("No topics below.");
	  sound_bell();
	  return;
      }

    current_offset = min(high, current_offset + (menu_lines - 1));
    OLH_ui_update_menus();
}



/*----------------------------------------------------------------------------

  erase_message

----------------------------------------------------------------------------*/

static void
erase_message( )
{
  if (current_message != NULL)
    {
      current_message = NULL;
      paint_message();
      wrefresh(win_message);
    }
}



/*----------------------------------

  select

----------------------------------*/
void
select(num)
     int num;
{
  int num_entries = size_menu(menu[current_menu]) - 1;

  if (!selection)
    {
      if (num == 1)
	selection = current[current_menu] + 1;
      if (num == -1)
	selection = current[current_menu] - 1;
    }
  else
    selection += num;

  if (selection < 1)
    selection = num_entries;
  if (selection > num_entries)
    selection = 1;

  wmove(win_entries, selection - 1, 0);
  wprintw(win_entries, ">");
  wrefresh(win_entries);
}


/*----------------------------------

  view

----------------------------------*/

static void
view(num)
     int num;
{
  static int go_up = FALSE;
  static int select_first = FALSE;
  int num_entries = size_menu(menu[current_menu]) - 1;

  if (!current[current_menu])
    {
      if (num == 1)
	current[current_menu] = 1;
      else if (num == -1)
	current[current_menu] = num_entries;
      else if (num == 0)
	{
	  if (select_first)
	    {
	      select_first = FALSE;
	      current[current_menu] = 1;
	    }
	  else
	    {
	      select_first = TRUE;
	      show_message(NONE_SELECTED);
	      return;
	    }
	}
    }

  if (viewed[current_menu])
    {
      if ((num == 1) || (num == -1))
	{
	  current[current_menu] += num;
	  if ((current[current_menu] > num_entries) ||
	      (current[current_menu] < 1))
	    {
	      if (go_up)
		{
		  go_up = FALSE;
		  up_to_prev_menu();
		  return;
		}
	      if (current[current_menu] > num_entries)
		{
		  if (current_menu == 0)
		    {
		      show_message(TOP_AND_NO_MORE);
		    }
		  else
		    {
		      go_up = TRUE;
		      show_message(NO_MORE);
		    }
		  current[current_menu] = num_entries;
		}
	      if (current[current_menu] < 1)
		{
		  if (current_menu == 0)
		    {
		      show_message(TOP_AND_NO_PREV);
		    }
		  else
		    {
		      go_up = TRUE;
		      show_message(NO_PREV);
		    }
		  current[current_menu] = 1;
		}
	      return;
	    }
/*	  if (current[current_menu] > XmListGetBottomPos(w[MENU_LIST]))
	    XmListSetBottomPos(w[MENU_LIST], current[current_menu]);
	  else if (current[current_menu] < XmListGetTopPos(w[MENU_LIST]))
	    XmListSetPos(w[MENU_LIST], current[current_menu]);  */
	}
    }

  viewed[current_menu] = TRUE;
  go_up = FALSE;
  invoke_item(current[current_menu]);
}


/*----------------------------------------------------------------------------

  invoke_item

----------------------------------------------------------------------------*/

static void
invoke_item( number )
int number;
{
  Menu *m;
  MenuEntry *e;
  long code;

  m = menu[current_menu];

  e = nth_entry(m, number);
  if (is_menu(e))
    {
      code = menu_load(e, &m);
      if (code)
	{
	  com_err(program, code, "loading %s", field_value(e, FILE_LOCATION));
	}
      else
	{
	  add_to_menupath(e);
	  current_menu++;
	  menu[current_menu] = m;
	  menu_entry[current_menu] = e;
	  current[current_menu] = 0;
	  viewed[current_menu] = FALSE;
	  current_offset = selection = 0;
	  OLH_ui_update_menus();
	  update_instructions();
	}
    }
  else
    {
      set_current_screen(SCR_VIEW);
      nocbreak();
      echo();
      viewdoc(e, "tty");
/* Previewer handles this now */
/*      wait_for_key(win_view_wait); */
      cbreak();
      noecho();
      paint_menu();
      set_current_screen(SCR_PRIM);
    }
}



/*----------------------------------------------------------------------------

  update_instructions

----------------------------------------------------------------------------*/

static void
update_instructions( )
{
  current_instr = /*((current_menu == keyword_menu) ? 
		   INSTR_KEYWORD_NUM : */
    (current_menu == 0) ? INSTR_NORMAL_TOP : INSTR_NORMAL;
  paint_instructions();
  wrefresh(win_instr);
}



/*----------------------------------------------------------------------------

  OLH_ui_viewer_wait

----------------------------------------------------------------------------*/

void
OLH_ui_viewer_wait( )
{
    wait_for_key(win_view_wait);
}



/*----------------------------------------------------------------------------

  wait_for_key

----------------------------------------------------------------------------*/

static void
wait_for_key( w )
WINDOW* w;
{
    wmove(w, 0, 0);
    wstandout(w);
    wprintw(w, "Press any key to continue... ");
    wstandend(w);
    wclrtoeol(w);
    wrefresh(w);
    wgetch(w);
    wclear(w);
    wrefresh(w);
}



/*----------------------------------------------------------------------------

  create_windows

----------------------------------------------------------------------------*/

static void
create_windows( )
{
    menu_lines = LINES - 12;

    scr[SCR_PRIM] = (scr_prim = newwin(0, 0, 0, 0));
    scr[SCR_VIEW] = (scr_view = newwin(0, 0, 0, 0));
    scr[SCR_SUGG] = (scr_sugg = newwin(0, 0, 0, 0));
    scount = SCR_MAX;

    wcount = 0;
    win[wcount++] = (win_heading = subwin(scr_prim, 1, 0, 0, 0));
    win[wcount++] = (win_instr = subwin(scr_prim, 5, 0, 1, 1));
    win[wcount++] = (win_label = subwin(scr_prim, 1, 0, 7, 3));
    win[wcount++] = (win_entries = subwin(scr_prim, menu_lines, 0, 9, 3));
    win[wcount++] = (win_above = subwin(scr_prim, 1, 0, 8, 40));
    win[wcount++] = (win_below = subwin(scr_prim, 1, 0, 9+menu_lines, 40));
    win[wcount++] = (win_prompt = subwin(scr_prim, 1, 0, LINES-2, 3));
    win[wcount++] = (win_message = subwin(scr_prim, 1, 0, LINES-1, 0));

    win[wcount++] = (win_view_wait = subwin(scr_view, 1, 0, LINES-1, 0));
    win[wcount++] = (win_sugg_wait = subwin(scr_sugg, 1, 0, LINES-1, 0));
}



/*----------------------------------------------------------------------------

  delete_windows

----------------------------------------------------------------------------*/

static void
delete_windows( )
{
    register int i;

    for (i = wcount-1; i >= 0; i--)
      delwin(win[i]);

    for (i = scount-1; i >= 0; i--)
      delwin(scr[i]);
}



/*----------------------------------------------------------------------------

  paint_windows

----------------------------------------------------------------------------*/

static void
paint_windows( )
{
  int start_col = (COLS/2) - (strlen(HEADER_TEXT)/2) - 5;

  wclear(scr_prim);
  wmove(win_heading, 0, (start_col > 0) ? start_col : 0);
  waddstr(win_heading, HEADER_TEXT);

  paint_instructions();

  prev_above = FALSE;
  prev_below = FALSE;

  paint_menu();
  paint_prompt();
  paint_message();
  paint_sugg();
}


/*----------------------------------------------------------------------------

  paint_instructions

----------------------------------------------------------------------------*/

static void
paint_instructions( )
{
    int i;

    wclear(win_instr);
    for (i = 0; i < NUM_INSTR_LINES; i++)
      {
	  wmove(win_instr, i, 0);
	  waddstr(win_instr, instr_text[current_instr][i]);
      }
}



/*----------------------------------------------------------------------------

  paint_menu

----------------------------------------------------------------------------*/

static void
paint_menu( )
{
  int num;
  int i;
  int above;
  int below;
  Menu *m;
  int line;

  m = menu[current_menu];

  num = size_menu(m) - 1;

  /*  Display the label  */

  wmove(win_label, 0, 0);
  waddstr(win_label, field_value(nth_entry(m, 0), NODE_LABEL));
  wclrtoeol(win_label);

  /*  Display each item  */

  for (i = 0; i < menu_lines; i++)
    {
      line = current_offset+i+1;
      wmove(win_entries, i, 0);
      if (line <= num)
	wprintw(win_entries, "%c%4d%c%c %s",
		(line == selection) ? '>' : ' ',
		line,
		is_menu(nth_entry(m, line)) ? '*' : ' ',
		(line == current[current_menu])
		? '.'
		: (((line-1 == current[current_menu])
		    || ((line == 1) && (!current[current_menu])))
		   ? 'n'
		   : (((line+1 == current[current_menu])
		       || ((line == num) && (!current[current_menu])))
		      ? 'p'
		      : ' ')),
		field_value(nth_entry(m, line), NODE_LABEL));
      wclrtoeol(win_entries);
    }


  /*  Display "above" message, if necessary  */

  above = (current_offset > 0);
  wmove(win_above, 0, 0);

  if (above && ! prev_above)
    waddstr(win_above, "More topics above (press 'b')");

  if (! above && prev_above)
    wclrtoeol(win_above);

  prev_above = above;


  /*  Display "below" message, if necessary  */

  below = (current_offset + menu_lines < num);
  wmove(win_below, 0, 0);

  if (below && ! prev_below)
    waddstr(win_below, "More topics below (press SPC)");

  if (! below && prev_below)
    wclrtoeol(win_below);

  prev_below = below;
}



/*----------------------------------------------------------------------------

  paint_prompt

----------------------------------------------------------------------------*/

static void
paint_prompt( )
{
    wmove(win_prompt, 0, 0);
    wclear(win_prompt);
    if (current_prompt != NULL)
      waddstr(win_prompt, current_prompt);
    if (current_input != NULL)
      waddstr(win_prompt, current_input);
}



/*----------------------------------------------------------------------------

  paint_message

----------------------------------------------------------------------------*/

static void
paint_message( )
{
    wmove(win_message, 0, 0);
    wclear(win_message);
    if (current_message != NULL)
      waddstr(win_message, current_message);
}


/*----------------------------------------------------------------------------

  paint_sugg

----------------------------------------------------------------------------*/

static void
paint_sugg( )
{
    wmove(scr_sugg, 0, 0);
    waddstr(scr_sugg,
	    "Please type your suggestions for improving the On-Line Help service.");
    wmove(scr_sugg, 1, 0);
    waddstr(scr_sugg,
	    "Your suggestions will be forwarded to the person responsible for");
    wmove(scr_sugg, 2, 0);
    waddstr(scr_sugg,
	    "maintaining On-Line Help.  Your comments are appreciated.  Thanks!");
    wmove(scr_sugg, 4, 0);
    waddstr(scr_sugg,
	    "When you are done typing your suggestion, type Ctrl-D or '.' on a line");
    wmove(scr_sugg, 5, 0);
    waddstr(scr_sugg,
	    "by itself.");
}



/*----------------------------------------------------------------------------

  set_current_screen

----------------------------------------------------------------------------*/

static void
set_current_screen( snumber )
int snumber;
{
    /*  Make sure previous screen is fully erased  */
    touchwin(stdscr);
    wrefresh(stdscr);
    wrefresh(curscr);

    /*  Bring up new screen  */

    current_screen = snumber;
    touchwin(scr[snumber]);
    wrefresh(scr[snumber]);

    /*  Special case:  Get cursor in right place  */

    switch (snumber)
      {
      case SCR_PRIM:
	  touchwin(win_prompt);
	  wrefresh(win_prompt);
	  break;
      }
}



/*----------------------------------------------------------------------------

  display_file

----------------------------------------------------------------------------*/

static int
display_file( path )
char* path;
{
  char buf[1024];

  sprintf(buf, "/usr/ucb/more -d %s", path);
  return(system(buf));
}



/*----------------------------------------------------------------------------

  display_buffer

----------------------------------------------------------------------------*/

static int
display_buffer( buffer )
char* buffer;
{
    char name[16];
    int fd, len, count;
    int status;

    strncpy(name,"/tmp/olh_XXXXXX",16);
    fd = mkstemp(name);
    len = strlen(buffer);
    count = 0;
    while ((count+=write(fd, &buffer[count], len-count) < len) &&
	   (count != -1));

    close(fd);

    if (count == -1)
      {
	  status = FALSE;
      }
    else
      {
	  status = display_file(name);
	  unlink(name);
      }
    return status;
}



/*----------------------------------------------------------------------------

  take_suggestion

----------------------------------------------------------------------------*/

static void
take_suggestion( )
{
  char *buffer;
  int allocated;
  int used; 
  char line[LARGE_BUFFER];
  int done = FALSE;
  char ch;

  allocated=LARGE_BUFFER;
  buffer = (char *) malloc(allocated * sizeof(char));

  wmove(scr_sugg, 7, 0);
  set_current_screen(SCR_SUGG);

  nocbreak();
  echo();

  strcpy(buffer, "");
  used = 0;
  while ((fgets(line, LARGE_BUFFER, stdin) != NULL) &&
	 (strcmp(line, ".\n")))
    {
      if (used+=strlen(line) >= allocated)
	{
	  allocated+=LARGE_BUFFER;
	  buffer = (char *) realloc(buffer, allocated * sizeof(char));
	}
      strcat(buffer, line);
    }

  freopen("/dev/tty", "r+", stdin);

  noecho();
  cbreak();

  fprintf(stdout, "\n\n");

  if (strlen(buffer) == 0)
    {
      fprintf(stdout,
	      "No suggestion to send.\n");
    }
  else
    while (! done)
	{
	    fprintf(stdout,
		    "Press 's' to send your suggestion, press '?' for other options: ");
	    ch = fgetc(stdin);
	    if (index("SsCcLl", ch) == NULL)
	      {
		  fprintf(stdout, "\n\nOptions:\n");
		  fprintf(stdout, "  s   Send the suggestion\n");
		  fprintf(stdout, "  c   Cancel the suggestion\n");
		  fprintf(stdout, "  l   List the text of the suggestion\n\n");
	      }
	    else
	      switch (ch)
		{
		case 's':
		case 'S':
		    fprintf(stdout, "\n\nSending...\n");
/*		    OLH_send_suggestion(buffer);*/
		    fprintf(stdout, "Your suggestion has been sent.\n\n");
		    done = TRUE;
		    break;
		case 'C':
		case 'c':
		    fprintf(stdout,
			    "\n\nYour suggestion has been cancelled.\n\n");
		    done = TRUE;
		    break;
		case 'L':
		case 'l':
		    display_buffer(buffer);
		    wait_for_key(win_sugg_wait);
		    break;
		}
	}

    wait_for_key(win_sugg_wait);
    set_current_screen(SCR_PRIM);
}



/*----------------------------------------------------------------------------

  OLH_ui_help_primary

----------------------------------------------------------------------------*/

void
OLH_ui_help_primary( )
{
    show_help_message(help_text[HELP_PRIM]);
}


/*----------------------------------------------------------------------------

  OLH_ui_help_keyword

----------------------------------------------------------------------------*/

void
OLH_ui_help_keyword( )
{
    show_help_message(help_text[HELP_KWD]);
}



/*----------------------------------------------------------------------------

  show_help_screen

----------------------------------------------------------------------------*/

static void
show_help_message( buffer )
char* buffer;
{
    if (buffer == NULL)
      {
	  OLH_ui_message("Sorry, help is currently unavailable...");
	  return;
      }

    set_current_screen(SCR_VIEW);
    display_buffer(buffer);
    wait_for_key(win_view_wait);
    set_current_screen(SCR_PRIM);
}



/*----------------------------------------------------------------------------

  read_help_text

static void
read_help_text( )
{
    int i;
    Loc* loc;
    char path[SMALL_BUFFER];

    loc = Loc_create(help_text_path, help_text_filsys);

    if (! Loc_verify(loc, AM_ReadOnly))
      OLH_ui_message("Warning:  Help on help not available");
    else
      for (i = 0; i < num_help_topics; i++)
	{
	    sprintf(path, "%s/%s", help_text_path, help_filename[i]);
	    help_text[i] = get_file(path);
	}

    Loc_delete(loc);
}

----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------

  invoke_olc

----------------------------------------------------------------------------*/

static void
invoke_olc()
{
  set_current_screen(SCR_VIEW);

  nocbreak();
  echo();

  fprintf(stdout, "Calling the On-Line Consulting system...\n\n");
  system("/usr/athena/olc");

  cbreak();
  noecho();

  wait_for_key(win_view_wait);
  set_current_screen(SCR_PRIM);
}



/*----------------------------------------------------------------------------

  handle_resize_event

----------------------------------------------------------------------------*/

static int
handle_resize_event( )
{
  struct winsize ws;
  int lines;
  int cols;
  int cont = FALSE;

  signal(SIGWINCH, SIG_IGN);

  delete_windows();

  /*  Find out the new size.  */

  if (ioctl(fileno(stdout), TIOCGWINSZ, &ws) != -1)
    {
      if (ws.ws_row != 0)
	lines = ws.ws_row;
      if (ws.ws_col != 0)
	cols = ws.ws_col;
    }


  /*  Check that it is large enough  */

  if ((lines < MIN_LINES) || (cols < MIN_COLS))
    {
/********
  if ((lines < 1)
      || ((lines == 1) && (cols < 8))
      || ((lines == 2) && (cols < 4)))
    {
**********/
      curses_shutdown();
      printf("\n\nYou have made the window too small.\n\n");
      printf("Please enlarge it to %d columns by %d lines (or larger)\n",
	     MIN_COLS, MIN_LINES);
      printf("and start the On-Line Help program again.\n\n");
      exit(-2);
    }
/*********
  if ((lines == 1) && (cols >= 8))
    {
      curses_shutdown();
      printf("\ntoo small");
    }

  if ((lines >= 2) && (cols >= 4))
    {
      curses_shutdown();
      printf("\ntoo\nsmall");
    }

  if ((lines >= MIN_LINES) && (cols >= MIN_COLS))
    cont = TRUE;

  if (cont)
    {
***********/

  if ((lines > max_lines) || (cols > max_cols))
    {
      curses_shutdown();
      LINES = lines;
      COLS = cols;
      if (! curses_init())
	exit(-2);
    }
  else
    {
      LINES = lines;
      COLS = cols;
    }

  /*  Refresh everything  */

  create_windows();
  paint_windows();
  set_current_screen(current_screen);
/****    }  ****/

  signal(SIGWINCH, handle_resize_event);
}



/*----------------------------------------------------------------------------

  handle_interrupt_event

----------------------------------------------------------------------------*/

static int
handle_interrupt_event( )
{
  signal(SIGINT, SIG_IGN);

  curses_shutdown();
  exit(0);
}


void
attach_hook(filsys)
     char *filsys;
{
  char buf[1024];

  sprintf(buf, ATTACHWAIT, filsys);
  show_message(buf);
}

void
attach_done_hook(filsys, status)
     char *filsys;
     int status;
{
  char buf[1024];

  if (status == 0)
    show_message("");
  else
    {
      sprintf(buf, ATTACHWARNING, filsys);
      show_message(buf);
    }
}

void
show_message(string)
     char *string;
{
  static char *stored = (char *) NULL;

  if ((stored == (char *) NULL) || strcmp(string, stored))
    {
      stored = string;
      current_message = stored;
      paint_message( );
      wrefresh(win_message);
    }
}
