/* the Command Line User Interface */

#include "defs.h"

#include "clui.h"

#include <unistd.h>
#include <getopt.h>

static void usage();

/* program-wide global variables definitions set by command line with defaults */
bool verbmode=FALSE;
bool quietmode=FALSE;
bool info_and_quit=FALSE;
bool guess=TRUE;
double given_angle;
bool rotate=TRUE;
bool save_rotated=FALSE;
bool save_skeleton=FALSE;
bool save_xetresult=FALSE;
bool dump_paths=FALSE;
int cutlevel=-1;
double available_megabytes=10.0;
char *scanned_image_basename;

static struct option long_options[] =
{
  {"help",                no_argument,       NULL, 'h'},
  {"version",             no_argument,       NULL, 'V'},
  {"verbose",             no_argument,       NULL, 'v'},
  {"quiet",               no_argument,       NULL, 'q'},
  {"just-info",           no_argument,       NULL, 'j'},
  {"no-rotate",           no_argument,       NULL, 'n'},
  {"save-rotated",        no_argument,       NULL, 'r'},
  {"save-skeleton",       no_argument,       NULL, 'k'},
  {"save-xetresult",      no_argument,       NULL, 'x'},
  {"dump-paths-and-quit", no_argument,       NULL, 'd'},
  {"force-rotation",      required_argument, NULL, 'f'},
  {"cutlevel",            required_argument, NULL, 'c'},
  {"megabytes",           required_argument, NULL, 'm'},
  {0, 0, 0, 0}
};

static char *optstring = "hVvqjnrkxdf:c:m:";

void clui_understand(int argc,char **argv)
{  
  int c,s,long_option_index;
  
  while ((c=getopt_long(argc,argv,optstring,long_options,&long_option_index))!=EOF)
    {
      switch (c)
	{
	case 'h':
	  banner();
	  usage();
	  exit(0);
	  break;
	case 'V':
	  banner();
	  exit(0);
	  break;
	case 'v':
	  verbmode=TRUE;
	  break;
	case 'q':
	  quietmode=TRUE;
	  break;
	case 'j':
	  info_and_quit=TRUE;
	  break;
	case 'n':
	  rotate=FALSE;
	  break;
	case 'r':
	  save_rotated=TRUE;
	  break;
	case 'k':
	  save_skeleton=TRUE;
	  break;
	case 'x':
	  save_xetresult=TRUE;
	  break;
	case 'd':
	  dump_paths=TRUE;
	  break;
	case 'f':
	  guess=FALSE;
	  s=sscanf(optarg,"%lf",&given_angle);
	  if (s!=1) FATALMESSAGE(ERROR_WRONGOPT);
	  break;
	case 'c':
	  s=sscanf(optarg,"%d",&cutlevel);
	  if (s!=1) FATALMESSAGE(ERROR_WRONGOPT);
	  break;
	case 'm':
	  s=sscanf(optarg,"%lf",&available_megabytes);
	  if (s!=1) FATALMESSAGE(ERROR_WRONGOPT);
	  break;
	default:
	  banner();
	  FATALMESSAGE(ERROR_WRONGOPT);
	}
    }
    if (optind>=argc)
      {
	/* beware, we could pipe also from stdin. not implemented yet */
	FATALMESSAGE(ERROR_NOFILE);
      }
    else
	MYOWN_ASPRINTF(&scanned_image_basename,"%s",argv[optind]);
}

void banner()
{
MESSAGE(
"Qui-ne-faut "VERSION": a GNU optical character recognition filter.\n"
"Copyleft 1999 Manu <michon@enst.fr>\n\n"

"This is prerelease software! Report bugs to michon@enst.fr\n"
"http://www.enst.fr/~michon/quinefaut/index.html\n\n"
	);
}

void usage()
{
  MESSAGE(
	  "USAGE:\n"
	  "  quinefaut [options] scanned_text\n"
	  "   (scanned_text is a raw grayscale PNM without extension)\n"
	  "\n"
	  "OPTIONS:\n"
	  "  --help                   -h : display this message\n"
	  "  --version                -V : display version number\n"
	  "  --verbose                -v : verbose operation\n"
	  "  --quiet                  -q : quiet operation\n"
	  "  --just-info              -j : prints image grayscale repartition and quits\n"
	  "  --no-rotate              -n : disable skew detection and correction\n"
	  "  --save-rotated           -r : save skew-corrected image\n"
	  "  --save-skeleton          -k : save skeleton\n"
	  "  --save-xetresult         -x : save xet line layout\n"
	  "  --dump-paths-and-quit    -d : dump paths (often several MB) and quit\n"
	  "\n"
          "  --force-rotation=alpha   -f : skip skew detection\n"
	  "                                and rotate by alpha rad (double)\n"
	  "  --megabytes=x            -m : max patch is x MB (default x=10) (double)\n" 
	  "  --cutlevel=z             -c : altitude of cut z in [0,255] (int)\n"
	  "                                z=-1 means (z_min+z_max)/2 (default)\n"
	  "                                z=-2 means z_average\n"
	  "                                z=-3 means z_mediane\n"
	  "                                z too high makes character fusions\n"
	  );
}







