#include "defs.h"

#include <math.h>

#include "matrix.h"
#include "path.h"
#include "connexity.h"

#define FUZZINESS 0.000001
#define SAMEPOINT 0.0001

static double fuzzy_number();
static void supermatrix_malloc_and_fill(supermatrix *SM,matrix *M,int cutlevel,
					int i0,int j0,int h,int w);
static void supermatrix_free(supermatrix *SM);
static void compute_intersect(triangle t1,triangle t2,double *x,double *y);
static void follow_the_line(path *P,supermatrix *SM,int i,int j,bool alt,int cuttype,bool direct);
static void follow_the_line_bi(path *P,supermatrix *SM,int i,int j,bool alt,int cuttype);
static lop *connexity_find_paths_monolithic(lop *Lorg,supermatrix *SM);
static void connexity_reassemble_paths(lop *L,top *T);
static void connexity_smooth_paths(top *T);
static void connexity_integrate(top *T);
static void connexity_attribute_holes(top *T);
static void connexity_final_computations(top *T);

double fuzzy_number()
{
  return 2.0*(rand()%GRANY)/GRANY-1;
}

/* real cutlevel MUST NOT BE an integer, we automatically translate it 0.5 right 
   trx and try parameters allow to arbitrarily translate double coordinates
   written in the supermatrix. This is a hack for submatrices. If you do not
   understand, just put trx=0 and try=0 and that'll be ok */
void supermatrix_malloc_and_fill(supermatrix *SM,matrix *M,int cutlevel,int i0,int j0,int h,int w)
{ 
  int i,j;

  SM->h=h;
  SM->w=w;
  
  VERBMESSAGE(" allocating %dx%d supermatrix (%.1f Mb)\n",
	  SM->h,SM->w,1e-6*sizeof(triangle)*SM->h*SM->w);

  MYOWN_MALLOC(SM->lines,triangle_line,SM->h);
  for (i=0;i<SM->h;i++) MYOWN_MALLOC(SM->lines[i],triangle,SM->w);

  for (i=0;i<SM->h;i++)
    for (j=0;j<SM->w;j++)
      {
	SM->lines[i][j].topleft=FALSE;
	SM->lines[i][j].bottomright=FALSE;
	
	SM->lines[i][j].x=(double)i0+i+0.5+fuzzy_number()*FUZZINESS;
	SM->lines[i][j].y=(double)j0+j+0.5+fuzzy_number()*FUZZINESS;
	SM->lines[i][j].z=(double)M->lines[i0+i][j0+j]-cutlevel
                          +0.5+fuzzy_number()*FUZZINESS;
      }
}

void supermatrix_free(supermatrix *SM)
{
  int i;

  VERBMESSAGE(" freeing %dx%d supermatrix\n",SM->h,SM->w);

  for (i=0;i<SM->h;i++) MYOWN_FREE(SM->lines[i],triangle,SM->w);
  MYOWN_FREE(SM->lines,triangle_line,SM->h);
}

/* >= does not exist for us. 
   probability of equality cases ruined by fuzziness */
#define OPPSGN(a, b)  ( ((a>0)&&(b<0)) || ((b>0)&&(a<0)) )
#define CUTVERTICAL 0
#define CUTHORIZONTAL 1
#define CUTDIAGONAL 2

void compute_intersect(triangle t1,triangle t2,double *x,double *y)
{
  *x=(t2.z*t1.x-t1.z*t2.x)/(t2.z-t1.z);
  *y=(t2.z*t1.y-t1.z*t2.y)/(t2.z-t1.z);
}

void follow_the_line(path *P,supermatrix *SM,
		     int i,int j,bool alt,int cuttype,bool direct)
{
  double x,y;
  bool first=TRUE; /* first is used when direct=FALSE. see above for explanation */
 
  /* if this works, I eat my keyboard */

  /* alt=FALSE means topleft. alt=TRUE means bottomright. */

  /* the conditions above says: while we are still in the rectangle, and if we are,
     while (i,j,alt) triangle is not visited (this last case meaning: we have returned
     on the first visited = closed path). */
  while ((0<=i)&&(i<SM->h-1)&&(0<=j)&&(j<SM->w-1))
    {
      if  ((alt)
	   ?SM->lines[i][j].bottomright
	   :SM->lines[i][j].topleft
	   )
	{
	  P->closed=TRUE;
	  goto outclosed;
	}
      
      if (!alt) /* alt=FALSE means topleft. alt=TRUE means bottomright. */
	{
	  /* entering a topleft triangle */
	  SM->lines[i][j].topleft=1;
	  switch (cuttype)
	    {
	    case CUTVERTICAL:
	      compute_intersect(SM->lines[i][j],SM->lines[i+1][j],&x,&y);
	      if (OPPSGN(SM->lines[i][j].z,SM->lines[i][j+1].z)) 
		{
		  cuttype=CUTHORIZONTAL;
		  i--; 
		}
	      else
		{ 
		  cuttype=CUTDIAGONAL;
		}
	      break;
	    case CUTHORIZONTAL:
	      compute_intersect(SM->lines[i][j],SM->lines[i][j+1],&x,&y);
	      if (OPPSGN(SM->lines[i][j].z,SM->lines[i+1][j].z)) 
		{
		  cuttype=CUTVERTICAL;
		  j--; 
		}
	      else
		{
		  cuttype=CUTDIAGONAL;
		}
	      break;
	    case CUTDIAGONAL:
	      compute_intersect(SM->lines[i+1][j],SM->lines[i][j+1],&x,&y);
	      if (OPPSGN(SM->lines[i][j].z,SM->lines[i+1][j].z)) 
		{
		  cuttype=CUTVERTICAL;
		  j--;
		} 
	      else
		{ 
		  cuttype=CUTHORIZONTAL;
		  i--;
		} 
	      break;
	    }
	}
      else
	{
	  /* entering a bottomright triangle */
	  SM->lines[i][j].bottomright=1;
	  switch (cuttype)
	    {
	    case CUTVERTICAL:
	      compute_intersect(SM->lines[i][j+1],SM->lines[i+1][j+1],&x,&y);
	      if (OPPSGN(SM->lines[i+1][j].z,SM->lines[i+1][j+1].z)) 
		{
		  cuttype=CUTHORIZONTAL;
		  i++;
		}
	      else
		{
		  cuttype=CUTDIAGONAL;
		}
	      break;
	    case CUTHORIZONTAL:
	      compute_intersect(SM->lines[i+1][j],SM->lines[i+1][j+1],&x,&y);
	      if (OPPSGN(SM->lines[i][j+1].z,SM->lines[i+1][j+1].z))  
		{
		  cuttype=CUTVERTICAL;
		  j++;
		}
	      else
		{
		  cuttype=CUTDIAGONAL;
		}
	      break;
	    case CUTDIAGONAL:
	      compute_intersect(SM->lines[i+1][j],SM->lines[i][j+1],&x,&y);
	      if (OPPSGN(SM->lines[i][j+1].z,SM->lines[i+1][j+1].z)) 
		{
		  cuttype=CUTVERTICAL;
		  j++;
		}
	      else
		{
		  cuttype=CUTHORIZONTAL;
		  i++;
		}
	      break;
	    }
	}
      
      if (direct)
	path_add_after_last(P,x,y);
      else
	{
	  /* if we are there, this is the second call to follow_the_line,
	     to reverse walk with direct=FALSE. The first point has already been put in the
	     list by the direct walk, so we forget it. */
	  if (first) 
	    first=FALSE;
	  else 
	    path_add_before_first(P,x,y);
	}

      /* neighbour triangle is always the other shape */
      alt=(alt)?FALSE:TRUE;
    }
  /* if we come here, it's because we encoutered the border
     of the matrix. Let's put the last point on it. Notice that
     cuttype is necessarily CUTHORIZONTAL or CUTVERTICAL, we could build an 
     alternative ifthenelse*/
  if (cuttype==CUTHORIZONTAL)
    {
      if (i<0) 
	compute_intersect(SM->lines[i+1][j],SM->lines[i+1][j+1],&x,&y);
      if (i>=SM->h-1) 
	compute_intersect(SM->lines[i][j],SM->lines[i][j+1],&x,&y);
    }
  if (cuttype==CUTVERTICAL)
    {
      if (j<0) 
	compute_intersect(SM->lines[i][j+1],SM->lines[i+1][j+1],&x,&y);
      if (j>=SM->w-1) 
	compute_intersect(SM->lines[i][j],SM->lines[i+1][j],&x,&y);
    }
  if (direct)
    path_add_after_last(P,x,y);
  else
    if (!first) path_add_before_first(P,x,y);
  
 outclosed:
}

void follow_the_line_bi(path *P,supermatrix *SM,
			 int i,int j,bool alt,int cuttype)
{
  path_init(P);
  
  /* following the path in the direct direction */
  follow_the_line(P,SM,i,j,alt,cuttype,TRUE);

  /* turn pi on your feet and reversewalk! */
  if (cuttype==CUTHORIZONTAL)
    if (!alt) i--; else i++;
  if (cuttype==CUTVERTICAL)
    if (!alt) j--; else j++;
  alt=(alt)?FALSE:TRUE;
  follow_the_line(P,SM,i,j,alt,cuttype,FALSE);
}

lop *connexity_find_paths_monolithic(lop *Lorg,supermatrix *SM)
{
  lop *L=Lorg;
  path P; 
  int i=0,j=0;
  bool a,b,c; 

  /* if this works, I eat my mouse */
  /* beware fallthru's in this kind of code */

  for (i=0;i<SM->h-1;i++)
    for (j=0;j<SM->w-1;j++)
      {
	if (!SM->lines[i][j].topleft)
	  {
	    a=SM->lines[i][j].z>0;
	    b=SM->lines[i+1][j].z>0;
	    c=SM->lines[i][j+1].z>0;
	    if ( a&& b&& c) { SM->lines[i][j].topleft=TRUE; goto next0; }
	    if (!a&&!b&&!c) { SM->lines[i][j].topleft=TRUE; goto next0; }
	    /* beware! we have to find an *oriented* path */
	    if (!a&& b&& c) follow_the_line_bi(&P,SM,i,j,FALSE,CUTVERTICAL);
	    if ( a&&!b&& c) follow_the_line_bi(&P,SM,i,j,FALSE,CUTDIAGONAL);
	    if (!a&&!b&& c) follow_the_line_bi(&P,SM,i,j,FALSE,CUTDIAGONAL);
	    if ( a&& b&&!c) follow_the_line_bi(&P,SM,i,j,FALSE,CUTHORIZONTAL);
	    if (!a&& b&&!c) follow_the_line_bi(&P,SM,i,j,FALSE,CUTVERTICAL);
	    if ( a&&!b&&!c) follow_the_line_bi(&P,SM,i,j,FALSE,CUTHORIZONTAL);
	    L=lop_add(L,&P);
	  next0:
	  }
	if (!SM->lines[i][j].bottomright)
	  {
	    a=SM->lines[i+1][j+1].z>0;
	    b=SM->lines[i+1][j].z>0;
	    c=SM->lines[i][j+1].z>0;
	    if ( a&& b&& c) { SM->lines[i][j].bottomright=TRUE; goto next1; }
	    if (!a&&!b&&!c) { SM->lines[i][j].bottomright=TRUE; goto next1; }
	    /* beware! we have to find an *oriented* path */
	    if (!a&& b&& c) follow_the_line_bi(&P,SM,i,j,TRUE,CUTVERTICAL);
	    if ( a&&!b&& c) follow_the_line_bi(&P,SM,i,j,TRUE,CUTHORIZONTAL);
	    if (!a&&!b&& c) follow_the_line_bi(&P,SM,i,j,TRUE,CUTVERTICAL);
	    if ( a&& b&&!c) follow_the_line_bi(&P,SM,i,j,TRUE,CUTDIAGONAL);
	    if (!a&& b&&!c) follow_the_line_bi(&P,SM,i,j,TRUE,CUTDIAGONAL);
	    if ( a&&!b&&!c) follow_the_line_bi(&P,SM,i,j,TRUE,CUTHORIZONTAL);
	    L=lop_add(L,&P);
	  next1:
	  }
      }

  return L;
}

void connexity_find_paths(top *T,matrix *M,int cutlevel,double avalaible_megabytes)
{
  supermatrix SM;
  lop *L=NULL;

  if (1e-6*M->h*M->w*sizeof(triangle)<avalaible_megabytes)
    {
      VERBMESSAGE(" enough space to treat the whole matrix\n");
      supermatrix_malloc_and_fill(&SM,M,cutlevel,0,0,M->h,M->w);
      L=connexity_find_paths_monolithic(L,&SM);
      supermatrix_free(&SM);
    }
  else
    {
      int I,J,h,w,square;

      VERBMESSAGE(" cutting the matrix in patches\n");
      
      /* we cut the scanned text into little squares fitting in memory */
      square=sqrt(1e6*avalaible_megabytes/sizeof(triangle));

      /* there is an overlapping of exactly 1 line or column */
      for (I=0;I<M->h-1;I+=square) 
	{
	  if (I+square<=M->h-1)
	    h=square+1;
	  else
	    h=M->h-I;

	  for (J=0;J<M->w-1;J+=square)
	  {
	    if (J+square<=M->w-1)
	      w=square+1;
	    else
	      w=M->w-J;
	    
	    VERBMESSAGE(" treating patch[%d-%d]x[%d-%d]\n",I,I+h-1,J,J+w-1);
	    supermatrix_malloc_and_fill(&SM,M,cutlevel,I,J,h,w);
	    L=connexity_find_paths_monolithic(L,&SM);
	    supermatrix_free(&SM);
	  }
	}
    }
  connexity_reassemble_paths(L,T);
  connexity_smooth_paths(T);
  connexity_integrate(T);
  connexity_attribute_holes(T);
  connexity_final_computations(T);
}

/* let's reassemble all non closed paths and bust what remains.
 the result is stored in a top (table of paths) */
void connexity_reassemble_paths(lop *L,top *T)
{
  int size=0,realsize=0,i,j;
  lop *U;
  double *x,*y;
  path **paths;
  int *previous,*next;
  int rejected=0,accepted=0;

  /* let's count the number of open paths to prepare malloc */
  for (U=L;U;U=U->next) if (!U->data.closed) size++;

  /* each open path has two ends */
  size*=2;

  MYOWN_MALLOC(x,double,size);
  MYOWN_MALLOC(y,double,size);
  MYOWN_MALLOC(paths,path *,size);
  MYOWN_MALLOC(previous,int,size);
  MYOWN_MALLOC(next,int,size);

#define NONE -1
#define TREATED -2
  /* NONE is used in previous to mark an index without predecessor. 
     TREATED is not used for the array previous */
  /* NONE is used in next to mark the end of a cluster of paths
     TREATED is used to mark already treated indices */

  for (i=0;i<size;i++) previous[i]=NONE;
  for (i=0;i<size;i++) next[i]=NONE;
  
  for (U=L;U;U=U->next)
    if (!U->data.closed)
      {
	/* examining first pair */
	for (i=0;(i<realsize)
	       &&(fabs(x[i]-U->data.first->x)+fabs(y[i]-U->data.first->y)>SAMEPOINT);i++); /* boolean shortcut! */
	if (i==realsize)
	  {
	    x[i]=U->data.first->x;
	    y[i]=U->data.first->y;
	    realsize++;
	  }

	/* examining last pair */
	for (j=0;(j<realsize)
	       &&(fabs(x[j]-U->data.last->x)+fabs(y[j]-U->data.last->y)>SAMEPOINT);j++); /* boolean shortcut! */
	if (j==realsize)
	  {
	    x[j]=U->data.last->x;
	    y[j]=U->data.last->y;
	    realsize++;
	  }
	
	paths[i]=&U->data;
	next[i]=j;
	previous[j]=i;
      }

  /* destroy non closed paths */
  for (i=0;i<realsize;i++)
    if (previous[i]==NONE)
      {
	int pj=0;
	
	rejected++;
    
	for (j=i;next[j]!=NONE;j=next[j])
	  {
	    if (j!=i) 
	      {
		next[pj]=TREATED;
		path_free(paths[pj]);
	      }
	    pj=j;
	  }
	next[pj]=TREATED;
	path_free(paths[pj]);
	next[j]=TREATED;
	/* next[j] is NONE so don't try to free paths[j]... */
      }
  VERBMESSAGE(" %d not closed rejected\n",rejected);
  
  /* reassemble new closed paths */
  for (i=0;i<realsize;i++)
    if (next[i]!=TREATED)
      {
	int pj=0;
	
	accepted++;
    
	for (j=i;next[j]!=TREATED;j=next[j])
	  {
	    if (j!=i) 
	     {
	       path_assemble(paths[pj],paths[j]);
	       next[pj]=TREATED;
	      }
	    pj=j;
	  }
	next[pj]=TREATED;
	
	/* first and last point of this closed path are the same.
	   let's remove the first to have a clean result. mark as closed. */
	{
	  pair *u;
	  u=paths[pj]->first;
	  paths[pj]->first=paths[pj]->first->next;
	  MYOWN_FREE(u,pair,1);
	  paths[pj]->length--;
	  paths[pj]->closed=TRUE;
	}

      }
  VERBMESSAGE(" %d new closed accepted\n",accepted);

  MYOWN_FREE(next,int,size);
  MYOWN_FREE(previous,int,size);
  MYOWN_FREE(paths,path *,size);
  MYOWN_FREE(y,double,size);
  MYOWN_FREE(x,double,size);
      
  /* let's change the linked list lop into a more compact table of path */
  i=0;
  for (U=L;U;U=U->next)
    if (U->data.closed) i++;

  T->size=i;
  MYOWN_MALLOC(T->table,path,i);

  i=0;
  for (U=L;U;U=U->next)
    {
      if (U->data.closed) 
	{
	  T->table[i]=U->data;
	  i++;
	} 
    }
  lop_free(L);
}

void connexity_smooth_paths(top *T)
{
  int i;
  pair *u,*pu;
  DECLARE_UNINITIALIZED(double,firstx);
  DECLARE_UNINITIALIZED(double,firsty);
  
  for (i=0;i<T->size;i++)
    {
      pu=NULL;
      for (u=T->table[i].first;u;u=u->next)
	{
	  if (pu)
	    {
	      pu->x=(pu->x+u->x)/2;
	      pu->y=(pu->y+u->y)/2;
	    }
	  else
	    {
	      firstx=u->x;
	      firsty=u->y;
	    }
	  pu=u;
	}
      pu->x=(pu->x+firstx)/2;
      pu->y=(pu->y+firsty)/2;
    }
}
    
/* \int_{along contour}Pdx+Qdy=\iint_{on surface}(dQ/dx-dP/dy)dxdy 
   so: we get area by integrating P=-y
              xg                  P=-xy
	      yg                  Q=xy */
void connexity_integrate(top *T)
{
  int i;
  pair *u,*pu;
  double x1,x2,y1,y2,qty;
  DECLARE_UNINITIALIZED(double,firstx);
  DECLARE_UNINITIALIZED(double,firsty);
  
  for (i=0;i<T->size;i++)
    {
      T->table[i].area=0;
      T->table[i].xg=0;
      T->table[i].yg=0;

      pu=NULL;
      for (u=T->table[i].first;u;u=u->next)
	{
	  if (pu)
	    {
	      x1=pu->x;
	      x2=u->x;
	      y1=pu->y;
	      y2=u->y;
	      T->table[i].area+=-(y1+y2)/2*(x2-x1);
	      qty=(x1*y1+x2*y2+x2*y1/2+x1*y2/2)/3;
	      T->table[i].xg+=-qty*(x2-x1);
	      T->table[i].yg+= qty*(y2-y1);

	      if (u->x<T->table[i].top)    T->table[i].top=u->x;
	      if (u->x>T->table[i].bottom) T->table[i].bottom=u->x;
	      if (u->y<T->table[i].left)   T->table[i].left=u->y;
	      if (u->y>T->table[i].right)  T->table[i].right=u->y;
	    }
	  else
	    {
	      firstx=u->x;
	      firsty=u->y;
	      
	      T->table[i].top=u->x;
	      T->table[i].bottom=u->x;
	      T->table[i].left=u->y;
	      T->table[i].right=u->y;
	    }
	  pu=u;
	}
     x1=pu->x;
     x2=firstx;
     y1=pu->y;
     y2=firsty;
     T->table[i].area+=-(y1+y2)*(x2-x1)/2;
     qty=(x1*y1+x2*y2+x2*y1/2+x1*y2/2)/3;
     T->table[i].xg+=-qty*(x2-x1);
     T->table[i].yg+= qty*(y2-y1);

     T->table[i].xg/=T->table[i].area;
     T->table[i].yg/=T->table[i].area;
    }
}

int compare_areas (const path *a,const path *b)
{
  if (a->area<b->area)
    return 1;
  else
    if (a->area>b->area)
      return -1;
    else 
      return 0;
}

int compare_tops (const path *a,const path *b)
{
  if (a->top>b->top)
    return 1;
  else
    if (a->top<b->top)
      return -1;
    else 
      return 0;
}

/* beware, this function can fail in pathologic cases. See error.jpg
   to see when this happens. I think it's not a trouble */
void connexity_attribute_holes(top *T)
{
  int i,ni,oldsize;
  DECLARE_UNINITIALIZED(int,inegative);
  top negativeT;

  /* separation of characters and holes */
  qsort(T->table,T->size,sizeof(path),(comparison_fn_t)compare_areas);

  for (i=0;(i<T->size)&&(T->table[i].area>0);i++); /* boolean shortcut! */ 
  inegative=i;
  
  negativeT.size=T->size-inegative;
  negativeT.table=T->table +inegative; /* pointer arithmetic! */
  oldsize=T->size;
  T->size=inegative;
  
  /* sorting according to top criterium */
  qsort(T->table,T->size,sizeof(path),(comparison_fn_t)compare_tops);
  qsort(negativeT.table,negativeT.size,sizeof(path),(comparison_fn_t)compare_tops);

  /* affectation of holes... not very optimized. beuarf.
     equality cases do not exist thanks to randomization */
  for (ni=0;ni<negativeT.size;ni++)
    {
      int imin=NONE;
      double areamin=DBL_MAX;

      /* find boxes containing this hole, and among them, the more little */
      /* boolean shortcut! */
      for (i=0;(i<T->size)&&(T->table[i].top<negativeT.table[ni].top);i++)
	  if ((T->table[i].bottom>negativeT.table[ni].bottom)
	      &&(T->table[i].left<negativeT.table[ni].left)
	      &&(T->table[i].right>negativeT.table[ni].right))
	    {
	      if (areamin>T->table[i].area) 
		{
		  areamin=T->table[i].area;
		  imin=i;
		}
	    }
      
      /* register the new hole */
      if (imin!=NONE)
	T->table[imin].holes=lop_add(T->table[imin].holes,&negativeT.table[ni]);
      else
	path_free(&negativeT.table[ni]);
    }

  /* memory management */
  MYOWN_REALLOC(T->table,path,oldsize,T->size);
}

static void connexity_final_computations(top *T)
{
  int i;
  lop *L;
  double area,xg,yg;
  
  for (i=0;i<T->size;i++)
    {
      area=T->table[i].area;
      xg=area*T->table[i].xg;
      yg=area*T->table[i].yg;

      for (L=T->table[i].holes;L;L=L->next) 
	{
	  area+=L->data.area;
	  xg+=L->data.area*L->data.xg;
	  yg+=L->data.area*L->data.yg;
	}
      
      T->table[i].area=area;
      T->table[i].xg=xg/area;
      T->table[i].yg=yg/area;
    }
}
