/* in this file, we propose an algorithm to compute the little
   orientation error people usually make while scanning.
   It's called guess_alpha_4 since it's the fourth I tried
   before an acceptable result. It tries to mimic what one
   does when trying to find lines orientation with a ruler.

   Explanations are given in the code, but you'd better
   see some sketches in docs/ to understand it.

   You can hope alpha precision of +-0.002 rad (2 pixel on a w=1000 page)
   with BREADTH=50 NTRIES=20 NNEIGHBORS=5. 
   [applying the correction algorithm twice leads to alpha=0.000000000
   answer, since alpha is determined by discrete i...!]
 */

#include "defs.h"

#include "matrix.h"
#include "accumulation.h"
#include "correct_orientation.h"

#include <math.h>

static double guess_alpha_4(matrix *M,int cutlevel,double alphamin,double alphamax);

/* this is a function to count ``black'' pixel along
   a given line (i1,j1)-(i2-j2) with |j2-j1|>|i2-i1|
   We say a pixel is black when it's below cutlevel. */
int countblack(matrix *M,int cutlevel,int i1,int j1,int i2,int j2)
{
  int i,j;
  int ct=0;

  for (j=j1;j<=j2;j++)  
    {
      i=i1+(double)(i2-i1)*(j-j1)/(double)(j2-j1);
      if (matrix_secure_get(M,i,j)<cutlevel) ct++;
    }

  return ct;
}

#define MAXCARAMBA 50000
#define BREADTH 50

double guess_alpha_4(matrix *M,int cutlevel,double alphamin,double alphamax)
{
  int carambacount=0;
  int i,j;
  int i0,j0;
  int hwhite;

  int iblack[BREADTH],imin,imax;

  int rep[BREADTH],repmax;
  
  int i1,j1,deltaj,ihigh,ilow,preced,current,fallmax;

  DECLARE_UNINITIALIZED(int,irepmax);
  DECLARE_UNINITIALIZED(int,ifallmax);

  VERBMESSAGE(" ");

 caramba:
  /* this algorithm is probabilist, so in case of failure,
     it comes back to caramba with goto and tries again. 
     No this is not a spaghetti. */
  carambacount++;
  VERBMESSAGE("-");
  if (carambacount>MAXCARAMBA) 
    {
      VERBMESSAGE("\n");
      FATALMESSAGE(ERROR_NOLINES);
    }
  
  /* pick up a random point in the 25% left of the page */
  i0=(double)(rand()%GRANY)/GRANY*M->h;
  j0=(double)(rand()%GRANY)/GRANY/4*M->w;

  /* look for an horizontal segment of BREADTH white pixels */
  hwhite=1;
  for (j=0;hwhite&&(j<BREADTH);j++)
    hwhite=matrix_secure_get(M,i0,j0+j)>cutlevel;
  if (j<BREADTH) goto caramba;
  
  /* success. now, for each of these white pixels, we
     find the first black pixel on top of it, and store
     the results in iblack[j], and also compute imin and imax.
     See the file climb_climb_climb.pnm */
  
  imax=0;
  imin=M->h;
  for (j=0;j<BREADTH;j++)
    {
      i=i0;
      do i--; while ((i>=0)&&(matrix_secure_get(M,i,j0+j)>cutlevel));
      if (i<0) goto caramba; /* overrun... */
      iblack[j]=i;
      
      if (i<imin) imin=i;
      if (i>imax) imax=i;
    }
  
  /* if this submatrix is more high than wide, this certainly is a failure */
  if ((imax-imin)>=BREADTH) goto caramba;
  
  /* we have found a proper submatrix
     (imin,j0)-(imax,j0+BREATH-1) 
     we compute the repartition function of iblack and keep
     the i realizing the first lower max */
  for (i=0;i<BREADTH;i++) rep[i]=0;
  for (j=0;j<BREADTH;j++) rep[imax-iblack[j]]++;
  repmax=0;
  for (i=0;i<BREADTH;i++) 
    if (rep[i]>repmax) { repmax=rep[i]; irepmax=i; }
  
  i1=imax-irepmax;
  j1=j0+BREADTH/2;
  /* the pixel (i1,j1) is very likely to be a pixel on a baseline */
  
  /* now, we consider lines beginning at (i1,j1) going to the right edge
     of the sheet with a slope in the range (alphamin,alphamax), see
     the image sweeping_slopes.pnm */
  deltaj=M->w-1-j1;
  ilow=tan(alphamin)*deltaj+i1;
  ihigh=tan(alphamax)*deltaj+i1;
  
  /* when looping through i, there will be a big fall of countblack
     corresponding to the instant when the line brutally quits 
     intersecting all characters to run free (free!). 
     See guessed_lines.pnm */
  fallmax=0;
  preced=countblack(M,cutlevel,i1,j1,ilow,M->w-1);
  for (i=ilow+1;i<=ihigh;i++)
    {
      current=countblack(M,cutlevel,i1,j1,i,M->w-1);
      if (preced-current>fallmax) 
	{
	  fallmax=preced-current;
	  ifallmax=i;
	}
      preced=current;
    }

  VERBMESSAGE("\n");

  return atan2(ifallmax-i1,deltaj);
}

#define NTRIES 20
#define NNEIGHBORS 5

double guess_alpha(matrix *M,int cutlevel,double alphamin,double alphamax)
{
  int k;
  double alpha[NTRIES];
  
  for (k=0;k<NTRIES;k++)
      alpha[k]=guess_alpha_4(M,cutlevel,alphamin,alphamax);

  return guess_accumulation(NTRIES,alpha,NNEIGHBORS);
}
