/*
 * Copyright (c) 1995 John Kohl.  All rights reserved.
 * Copyright (c) 1995 Rolf Grossmann
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by John Kohl and
 *	Rolf Grossmann.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */


#include "forms.h"
#include "mixer.h"
#include "mixer_c.h"

int debug;

#define DPRINTF(x) if (debug) printf x

int
get_encoding(const char *encstring)
{
    if (strcmp(encstring, AudioNlinear) == 0)
	return AUDIO_ENCODING_LINEAR;
    if (strcmp(encstring, AudioNulaw) == 0)
	return AUDIO_ENCODING_ULAW;
    if (strcmp(encstring, AudioNalaw) == 0)
	return AUDIO_ENCODING_ALAW;
    if (strcmp(encstring, AudioNpcm16) == 0)
	return AUDIO_ENCODING_PCM16;
    if (strcmp(encstring, AudioNpcm8) == 0)
	return AUDIO_ENCODING_PCM8;
    if (strcmp(encstring, AudioNadpcm) == 0)
	return AUDIO_ENCODING_ADPCM;
    warnx("encoding `%s' not recognized", encstring);
    return -1;
}

const char *
encoding_text(int val)
{
    switch (val) {
/* PCM16 is the same as linear
   case AUDIO_ENCODING_LINEAR: */
    default:
	return  AudioNlinear;
    case AUDIO_ENCODING_ULAW:
	return  AudioNulaw;
    case AUDIO_ENCODING_ALAW:
	return  AudioNalaw;
    case AUDIO_ENCODING_PCM16:
	return  AudioNpcm16;
    case AUDIO_ENCODING_PCM8:
	return  AudioNpcm8;
    case AUDIO_ENCODING_ADPCM:
	return  AudioNadpcm;
    }
}

mixer_ctrl_t masterinfo;
mixer_ctrl_t playinfo;
mixer_ctrl_t recinfo;

void
get_devinfo(int mixer, DB **namedb, DB **indexdb)
{
    DB *dev_db, *indx_db;
    DBT key, datum;
    mixer_devinfo_t di, storedi;
    int error;
    devdata_db_t dbitem;
    int first;

    dev_db = dbopen(NULL, O_RDWR|O_CREAT, 0666, DB_HASH, NULL);
    if (dev_db == NULL)
	err(1, "open name database");
    
    indx_db = dbopen(NULL, O_RDWR|O_CREAT, 0666, DB_HASH, NULL);
    if (indx_db == NULL)
	err(1, "open devindex database");
    
    di.index = 0;
    while (((error = ioctl(mixer, AUDIO_MIXER_DEVINFO, &di)) == 0) ||
	   di.index < 255) {
	if (error == 0) {
	    dbitem.index = di.index;
	    strcpy(dbitem.name, di.label.name);
	    dbitem.type = di.type;
	    dbitem.devclass = di.mixer_class;
	    /* mark master volume specially, so we don't put it in the
	       list of things eligible for generic handling */
	    if (strcmp(di.label.name, AudioNvolume) == 0) {
		DPRINTF(("found masterinfo index %d\n", di.index));
		masterinfo.un.value.num_channels = 2;
		masterinfo.dev = di.index;
		masterinfo.type = AUDIO_MIXER_VALUE;
		if (ioctl(mixer, AUDIO_MIXER_READ, &masterinfo)) {
		    masterinfo.un.value.num_channels = 1;
		    if (ioctl(mixer, AUDIO_MIXER_READ, &masterinfo))
			warn("can't get master info");
		}
		dbitem.srcflags = MIXER_MASTER_VOL;
	    } else if (strcmp(di.label.name, AudioNspeaker) == 0) {
		/* speaker can't be manipulated */
		dbitem.srcflags = MIXER_MASTER_VOL;
	    } else if (first && di.type == AUDIO_MIXER_VALUE) {
		char mixtxt[MAX_AUDIO_DEV_LEN + 16];
		playinfo.un.value.num_channels = 2;
		playinfo.dev = dbitem.index;
		playinfo.type = AUDIO_MIXER_VALUE;
		if (ioctl(mixer_fd, AUDIO_MIXER_READ, &playinfo)) {
		    playinfo.un.value.num_channels = 1;
		    if (ioctl(mixer, AUDIO_MIXER_READ, &playinfo))
			warn("can't fetch mixer level for default `%s'",
			     dbitem.name);
		}
		sprintf(mixtxt, "%s volume", dbitem.name);
		fl_set_object_label(fd_mixerform->srcvol_text, mixtxt);
		sprintf(mixtxt, "%s mode", dbitem.name);
		fl_set_object_label(fd_mixerform->srcmix_text, mixtxt);
		dbitem.srcflags = 0;
		first = 0;
	    } else
		dbitem.srcflags = 0;

	    datum.data = &dbitem;
	    datum.size = sizeof(dbitem);
	    key.data = di.label.name;
	    key.size = strlen(di.label.name)+1; /* include the NUL */

	    DPRINTF(("storing key `%s' (%d) dev %d\n",
		     key.data, key.size, dbitem.index));
	    if ((*dev_db->put)(dev_db, &key, &datum, 0) != 0)
		err(1, "build database");
	    key.data = &di.index;
	    key.size = sizeof(di.index);

	    DPRINTF(("storing key %d name `%s'\n",
		     *(int *)key.data, dbitem.name));
	    if ((*indx_db->put)(indx_db, &key, &datum, 0) != 0)
		err(1, "build database");

	    /* XXX Assume the source list comes after all the elements. */
	    if (strcmp(di.label.name, AudioNsource) == 0 &&
		di.type == AUDIO_MIXER_ENUM) {
		register int i;
		DPRINTF(("found sources: %d\n", di.index));
		for (i = 0; i < di.un.e.num_mem; i++) {
		    key.data = &di.un.e.member[i].ord;
		    key.size = sizeof(di.un.e.member[i].ord);
		    if ((*indx_db->get)(indx_db, &key, &datum, 0) == 0) {
			bcopy(datum.data, &dbitem, sizeof(dbitem));
			dbitem.srcflags |= MIXER_REC_SRC;
			datum.data = &dbitem;
			datum.size = sizeof(dbitem);
			if ((*indx_db->put)(indx_db, &key, &datum, 0) != 0)
			    err(1, "build database");
			key.data = dbitem.name;
			key.size = strlen(dbitem.name)+1;
			if ((*dev_db->put)(dev_db, &key, &datum, 0) != 0)
			    err(1, "build database");
			DPRINTF(("marked `%s' as recordable\n", dbitem.name));
		    } else
			warnx("entry ordering problem for `%s' in source list",
			      di.un.e.member[i].label);
		}
		break;
	    }
	}
	/* some devices have holes in their number space.  argh. */
	if (di.next == AUDIO_MIXER_LAST)
	    di.index = di.index + 1;
	else if (di.index == di.next)
	    di.index++;
	else
	    di.index = di.next;
	di.label.name[0] = '\0';
    }

    *namedb = dev_db;
    *indexdb = indx_db;
}

void
set_sliders(FL_OBJECT *slider_l,
	    FL_OBJECT *slider_r,
	    FL_OBJECT *m_button,
	    FL_OBJECT *st_button,
	    mixer_ctrl_t *mixinfo)
{
    fl_freeze_form(fd_mixerform->mixerform);
    if (mixinfo->un.value.num_channels == 2) {
	fl_set_slider_value(slider_l,
			    mixinfo->un.value.level[AUDIO_MIXER_LEVEL_LEFT]);
	fl_set_slider_value(slider_r,
			    mixinfo->un.value.level[AUDIO_MIXER_LEVEL_RIGHT]);
	fl_set_button(st_button,1);
	fl_set_button(m_button,0);
    } else {
	fl_set_slider_value(slider_l,
			    mixinfo->un.value.level[AUDIO_MIXER_LEVEL_MONO]);
	fl_set_slider_value(slider_r,
			    mixinfo->un.value.level[AUDIO_MIXER_LEVEL_MONO]);
	fl_set_button(st_button,0);
	fl_set_button(m_button,1);
    }
    fl_unfreeze_form(fd_mixerform->mixerform);
}

DB *dev_db_name, *dev_db_index;

FD_mixerform *fd_mixerform;
int mixer_fd = -1;
int snd_fd = -1;

int
open_snd(int mustwork)
{
    register int fd;
    fd = open("/dev/sound", O_RDWR, 0); /* XXX define _PATH_SOUND ? */
    if (fd < 0) {
	if (mustwork)
	    err(1, "Can't open sound device");
	else
	    warn("Can't open sound device");
    }
    return fd;
}

void
close_snd(int fd)
{
    (void) close(fd);
    if (fd == snd_fd)
	snd_fd = -1;
}

void
usage(const char *progname)
{
    err(1, "%s: Usage: %s [-d] { xforms arguments }\n", progname, progname);
}

int
main(int argc, char *argv[])
{
    DBT key, datum;
    mixer_ctrl_t ctl;
    int i, j, v, error;
    char *p;
    extern int optind;
    struct audio_info ainfo, ascratch;
    char inputval[32];
    devdata_db_t dbitem;

    mixer_fd = open("/dev/mixer", O_RDONLY, 0); /* XXX define _PATH_MIXER ? */
    if (mixer_fd < 0)
	err(1, "Can't open mixer device");

    snd_fd = open_snd(1);
    if (ioctl(snd_fd, AUDIO_GETINFO, &ainfo))
	err(1, "can't get current audio info");


    fl_initialize(argv[0], 0, 0, 0, &argc, argv);
    while ((i = getopt(argc, argv, "d")) != -1) {
	switch (i) {
	case 'd':
	    debug = 1;
	    break;
	default:
	    usage(argv[0]);
	}
    }

    fd_mixerform = create_form_mixerform();

    get_devinfo(mixer_fd, &dev_db_name, &dev_db_index);


    /* fill-in form initialization code */
    fl_set_slider_bounds(fd_mixerform->master_l,255.0,0.0);
    fl_set_slider_return(fd_mixerform->master_l,0);
    fl_set_slider_bounds(fd_mixerform->master_r,255.0,0.0);
    fl_set_slider_return(fd_mixerform->master_r,0);
    fl_set_slider_bounds(fd_mixerform->rec_l,255.0,0.0);
    fl_set_slider_return(fd_mixerform->rec_l,0);
/*    fl_set_slider_bounds(fd_mixerform->rec_r,255.0,0.0);*/
    fl_set_slider_bounds(fd_mixerform->play_l,255.0,0.0);
    fl_set_slider_return(fd_mixerform->play_l,0);
    fl_set_slider_bounds(fd_mixerform->play_r,255.0,0.0);
    fl_set_slider_return(fd_mixerform->play_r,0);

    fl_set_slider_step(fd_mixerform->master_l,1.0);
    fl_set_slider_step(fd_mixerform->master_r,1.0);
    fl_set_slider_step(fd_mixerform->rec_l,1.0);
/*    fl_set_slider_step(fd_mixerform->rec_r,1.0);*/
    fl_set_slider_step(fd_mixerform->play_l,1.0);
    fl_set_slider_step(fd_mixerform->play_r,1.0);

    fl_set_slider_precision(fd_mixerform->master_l,0);
    fl_set_slider_precision(fd_mixerform->master_r,0);
    fl_set_slider_precision(fd_mixerform->rec_l,0);
/*    fl_set_slider_precision(fd_mixerform->rec_r,0);*/
    fl_set_slider_precision(fd_mixerform->play_l,0);
    fl_set_slider_precision(fd_mixerform->play_r,0);

    set_sliders(fd_mixerform->master_l, fd_mixerform->master_r,
		    fd_mixerform->master_mono, fd_mixerform->master_stereo,
		    &masterinfo);

    fl_set_slider_value(fd_mixerform->rec_l,255.0);
/*    fl_set_slider_value(fd_mixerform->rec_r,255.0);*/

    set_sliders(fd_mixerform->play_l, fd_mixerform->play_r,
		fd_mixerform->play_mono, fd_mixerform->play_stereo,
		&playinfo);

    if (ainfo.record.channels == 2)
	fl_set_button(fd_mixerform->rec_stereo,1);
    else
	fl_set_button(fd_mixerform->rec_mono,1);

    /* XXX do anything about full duplex stuff? AUDIO_GETFD/SETFD ? */

    /*
     * XXX must match bit order  in <sys/audioio.h>
     */
    fl_addto_choice(fd_mixerform->dac_mode,"Off");
    fl_addto_choice(fd_mixerform->dac_mode,"Play");
    fl_addto_choice(fd_mixerform->dac_mode,"Record");
    fl_addto_choice(fd_mixerform->dac_mode,"Play & Record");
    if (ioctl(snd_fd, AUDIO_GETFD, &i) == 0 && i == 0)
	fl_set_choice_item_mode(fd_mixerform->dac_mode, 4, FL_PUP_GRAY);

    /*
     * ainfo.mode is set to AUMODE_PLAY or AUMODE_RECORD, not to a
     * bitmask.  sigh.
     */
    fl_set_choice(fd_mixerform->dac_mode, (1<<ainfo.mode) + 1);
    DPRINTF(("initial duplexmode %d\n", 1<<ainfo.mode));

    /* check out each encoding before putting into the list */

    DPRINTF(("initial encodings: play %s record %s\n",
	     encoding_text(ainfo.play.encoding),
	     encoding_text(ainfo.record.encoding)));
    /*
     * XXX must match order in <sys/audioio.h>
     */
    fl_addto_choice(fd_mixerform->rec_encoding,AudioNulaw);
    fl_addto_choice(fd_mixerform->rec_encoding,AudioNalaw);
    fl_addto_choice(fd_mixerform->rec_encoding,AudioNpcm16);
    fl_addto_choice(fd_mixerform->rec_encoding,AudioNpcm8);
    fl_addto_choice(fd_mixerform->rec_encoding,AudioNadpcm);

    fl_addto_choice(fd_mixerform->play_encoding,AudioNulaw);
    fl_addto_choice(fd_mixerform->play_encoding,AudioNalaw);
    fl_addto_choice(fd_mixerform->play_encoding,AudioNpcm16);
    fl_addto_choice(fd_mixerform->play_encoding,AudioNpcm8);
    fl_addto_choice(fd_mixerform->play_encoding,AudioNadpcm);

    for (i = AUDIO_ENCODING_ULAW; i <= AUDIO_ENCODING_ADPCM; i++) {
	AUDIO_INITINFO(&ascratch);
	ascratch.record.encoding = i;
	if (ioctl(snd_fd, AUDIO_SETINFO, &ascratch) == -1)
	    fl_set_choice_item_mode(fd_mixerform->rec_encoding, i,
				    FL_PUP_GRAY);
	if (i == ainfo.record.encoding)
	    fl_set_choice(fd_mixerform->rec_encoding, i);
	AUDIO_INITINFO(&ascratch);
	ascratch.play.encoding = i;
	if (ioctl(snd_fd, AUDIO_SETINFO, &ascratch) == -1)
	    fl_set_choice_item_mode(fd_mixerform->play_encoding, i,
				    FL_PUP_GRAY);
	if (i == ainfo.play.encoding)
	    fl_set_choice(fd_mixerform->play_encoding, i);
    }

    AUDIO_INITINFO(&ascratch);
    ascratch.play.encoding = ainfo.play.encoding;
    ascratch.record.encoding = ainfo.record.encoding;
    if (ioctl(snd_fd, AUDIO_SETINFO, &ascratch))
	warn("can't reset encodings to prior values");
    

    sprintf(inputval, "%d", ainfo.play.sample_rate);
    fl_set_input(fd_mixerform->playrate, inputval);
    sprintf(inputval, "%d", ainfo.record.sample_rate);
    fl_set_input(fd_mixerform->recrate, inputval);

    /* set up the recording, mixer source lists */
    for (i = 1, j = 1,
	     error = (*dev_db_name->seq)(dev_db_name, &key, &datum, R_FIRST);
	 error == 0;
	 error = (*dev_db_name->seq)(dev_db_name, &key, &datum, R_NEXT)) {
	bcopy(datum.data, &dbitem, sizeof(dbitem));
	DPRINTF(("`%s'(%d) is index %d\n", key.data, key.size, i));
	if (dbitem.type == AUDIO_MIXER_VALUE &&
	    dbitem.srcflags & MIXER_REC_SRC) {
	    /* eligible as recording source */
	    fl_addto_choice(fd_mixerform->rec_source, dbitem.name);
	    if (dbitem.index == ainfo.record.port)
		fl_set_choice(fd_mixerform->rec_source, i);
	    i++;
	}
	if (dbitem.type == AUDIO_MIXER_VALUE &&
	    (dbitem.srcflags & MIXER_MASTER_VOL) == 0) {
	    /* eligible for generic volume control */
	    fl_addto_choice(fd_mixerform->mix_source, dbitem.name);
	    if (dbitem.index == playinfo.dev)
		fl_set_choice(fd_mixerform->mix_source, j);
	    j++;
	}
    }

    close(snd_fd);

    /* show the first form */
    fl_show_form(fd_mixerform->mixerform,FL_PLACE_CENTER,FL_FULLBORDER,"mixerform");
    while (fl_do_forms())
	;
    return 0;
}
