/** @file wlan_tx.c
  * @brief This file contains the handling of TX in wlan
  * driver.
  * 
  *  Copyright (c) Marvell Semiconductor, Inc., 2003-2005
  */
/********************************************************
Change log:
	09/28/05: Add Doxygen format comments
	12/13/05: Add Proprietary periodic sleep support
********************************************************/

#include	"include.h"

typedef struct TxPD TxPD;
#define OS_INT_DISABLE          cli()
#define OS_INT_RESTORE          sti()
#define OS_INTERRUPT_SAVE_AREA

/********************************************************
		Local Variables
********************************************************/

/* temporary buffer to hold TX packet */
static u8 tx_temp_buf[MRVDRV_ETH_TX_PACKET_BUFFER_SIZE];

/********************************************************
		Global Variables
********************************************************/

/********************************************************
		Local Functions
********************************************************/

/** 
 *  @brief This function processes a single packet and sends
 *  to IF layer
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @param skb     A pointer to skb which includes TX packet
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int SendSinglePacket(wlan_private *priv, struct sk_buff *skb)
{
	wlan_adapter   *Adapter = priv->adapter;
	int             ret = WLAN_STATUS_SUCCESS;
	TxPD		LocalTxPD;
	TxPD		*pLocalTxPD = &LocalTxPD;
	u8		*ptr = tx_temp_buf;

	ENTER();

	if (!skb->len || (skb->len > MRVDRV_ETH_TX_PACKET_BUFFER_SIZE)) {
		PRINTM(INFO, "Tx Error: Bad skb length %d : %d\n", skb->len, 
					MRVDRV_ETH_TX_PACKET_BUFFER_SIZE);
		ret = WLAN_STATUS_FAILURE;
		goto done;
	}

	memset(pLocalTxPD, 0, sizeof(TxPD));

	pLocalTxPD->TxPacketLength = skb->len;

#ifdef WMM
	if(Adapter->wmm.enabled) {
		/* 
		 * original skb->priority has been overwritten 
		 * by wmm_map_and_add_skb()
		 */
#ifdef PACKET_PRIORITY
		pLocalTxPD->Priority = (u8)skb->priority;
#endif /* PACKET_PRIORITY */
	}
#endif /* WMM */
	
	if(Adapter->PSState != PS_STATE_FULL_POWER){
#ifdef WMM_UAPSD
		//UAPSD		
		if ((Adapter->CurBssParams.wmm_uapsd_enabled == TRUE)&& (Adapter->wmm.qosinfo != 0) &&
			 (Adapter->sleep_period.period != 0)) {
       			if (wmm_lists_empty(priv)) {
				Adapter->wmm.no_more_packet = 1;
#ifdef PACKET_PRIORITY
				pLocalTxPD->PowerMgmt = MRVDRV_TxPD_POWER_MGMT_LAST_PACKET;
#endif /* PACKET_PRIORITY */
			}
		}
#ifdef PROP_PERIODIC_SLEEP
#ifdef TX_QUEUE
#ifdef PACKET_PRIORITY	    
		else
#endif  //PACKET_PRIORITY		    
#endif	//TX_QUEUE		
#endif  //PROP_PERIOD_SLEEP

#endif /* WMM_UAPSD */	

#ifdef PROP_PERIODIC_SLEEP
#ifdef TX_QUEUE
#ifdef PACKET_PRIORITY		    
		{
			//Proprietary periodic sleep 
			if ( !Adapter->TxSkbNum  && (Adapter->sleep_period.period != 0)){
				pLocalTxPD->PowerMgmt = MRVDRV_TxPD_POWER_MGMT_LAST_PACKET;
			}
		}
#endif  //PACKET_PRIORITY	    
#endif	//TX_QUEUE		
#endif  //PROP_PERIOD_SLEEP		    
	}
	
	/* offset of actaul data */
	pLocalTxPD->TxPacketLocation = sizeof(TxPD);

	/* TxCtrl set by user or default */
	pLocalTxPD->TxControl = Adapter->PktTxCtrl;

#ifdef	BIG_ENDIAN 
	endian_convert_pLocalTxPD(pLocalTxPD);
#endif	

	memcpy(pLocalTxPD->TxDestAddrHigh, skb->data, MRVDRV_ETH_ADDR_LEN);

	HEXDUMP("TxPD", (u8 *) pLocalTxPD, sizeof(TxPD));

	memcpy(ptr, pLocalTxPD, sizeof(TxPD));
	
	ptr += sizeof(TxPD);
		
	HEXDUMP("Tx Data", (u8 *) skb->data, skb->len);
	memcpy(ptr, skb->data, skb->len);

	ret = sbi_host_to_card(priv, MVMS_DAT,
		       		tx_temp_buf,  skb->len + sizeof(TxPD));

	if (ret) {
		PRINTM(INFO, "Tx Error: sbi_host_to_card failed: 0x%X\n",ret);
		goto done;
	}

	PRINTM(INFO, "SendSinglePacket succeeds\n");

done:
	if (!ret) {
		priv->stats.tx_packets++;
		priv->stats.tx_bytes += skb->len;
	}
	else {
		priv->stats.tx_dropped++;
		priv->stats.tx_errors++;
	}

	/* need to be freed in all cases */
	os_free_tx_packet(priv);
	os_start_queue(priv);

	LEAVE();
	return ret;
}

/********************************************************
		Global functions
********************************************************/

/** 
 *  @brief This function checks the conditions and sends packet to IF
 *  layer if everything is ok.
 *  
 *  @param priv    A pointer to wlan_private structure
 *  @return 	   n/a
 */
void wlan_process_tx(wlan_private *priv)
{
	wlan_adapter   *Adapter = priv->adapter;

	OS_INTERRUPT_SAVE_AREA;	

	ENTER();
    
	if(priv->wlan_dev.dnld_sent) {
		PRINTM(MSG, "TX Error: dnld_sent = %d, not sending\n",
						priv->wlan_dev.dnld_sent);
		goto done;
	}

	SendSinglePacket(priv, Adapter->CurrentTxSkb);

	OS_INT_DISABLE;
	priv->adapter->HisRegCpy &= ~HIS_TxDnLdRdy;
	OS_INT_RESTORE;
	
done:
	LEAVE();
}

/** 
 *  @brief This function queues the packet received from
 *  kernel/upper layer and wake up the main thread to handle it.
 *  
 *  @param priv    A pointer to wlan_private structure
  * @param skb     A pointer to skb which includes TX packet
 *  @return 	   WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
int wlan_tx_packet(wlan_private *priv, struct sk_buff *skb)
{
	u32		flags;
	wlan_adapter 	*Adapter = priv->adapter;
	int		ret = WLAN_STATUS_SUCCESS;

	ENTER();

	spin_lock_irqsave(&Adapter->CurrentTxLock, flags);

#ifdef WMM
	if(Adapter->wmm.enabled) {
		wmm_map_and_add_skb(priv, skb);
		wake_up_interruptible(&priv->MainThread.waitQ);
	} else
#endif /* WMM */
#ifdef TX_QUEUE
	{
		if(Adapter->TxSkbNum < MAX_NUM_IN_TX_Q)	{
			Adapter->TxSkbNum ++;
			list_add_tail((struct list_head *) skb,(struct list_head *) &priv->adapter->TxSkbQ);
			wake_up_interruptible(&priv->MainThread.waitQ);
		}else
			ret = WLAN_STATUS_FAILURE;
	}
#else	
	if (!Adapter->CurrentTxSkb) {
		Adapter->CurrentTxSkb = skb;
		wake_up_interruptible(&priv->MainThread.waitQ);
	} else {
		ret = WLAN_STATUS_FAILURE;
	}
#endif 	
	spin_unlock_irqrestore(&Adapter->CurrentTxLock, flags);

	LEAVE();

	return ret;
}


/** 
 *  @brief This function tells firmware to send a NULL data packet.
 *  
 *  @param priv     A pointer to wlan_private structure
 *  @param pwr_mgmt indicate if power management bit should be 0 or 1
 *  @return 	    n/a
 */
int SendNullPacket(wlan_private *priv, u8 pwr_mgmt)
{
	wlan_adapter 	*Adapter = priv->adapter;
  	TxPD txpd;
  	int ret = WLAN_STATUS_SUCCESS;

	ENTER();

	if (priv->adapter->SurpriseRemoved == TRUE) {
    		ret = WLAN_STATUS_FAILURE;
		goto done;
   	}

	if (priv->adapter->MediaConnectStatus == WlanMediaStateDisconnected) {
    		ret = WLAN_STATUS_FAILURE;
		goto done;
   	}

  	memset(&txpd, 0, sizeof(TxPD));

  	txpd.TxControl   = Adapter->PktTxCtrl;
#ifdef PACKET_PRIORITY
 	txpd.PowerMgmt   = pwr_mgmt;
#endif /* PACKET_PRIORITY */
 	txpd.TxPacketLocation = sizeof(TxPD);

#ifdef BIG_ENDIAN 
 	endian_convert_pLocalTxPD(&txpd);
#endif  

   	memcpy(tx_temp_buf, &txpd, sizeof(TxPD));

   	ret = sbi_host_to_card(priv, MVMS_DAT,
		       		tx_temp_buf, sizeof(TxPD));

   	if (ret != 0)
  	{
   		PRINTM(INFO, "TX Error: SendNullPacket failed!\n");
		goto done;
   	}

done:
  	LEAVE();
	return ret;
}


/**
 *  @brief This function check if we need send last packet indication.
 *
 *  @param priv     A pointer to wlan_private structure
 *
 *  @return        TRUE or FALSE
 */
u8 CheckLastPacketIndication(wlan_private *priv)
{
        wlan_adapter    *Adapter = priv->adapter;
        u8 ret = FALSE;
        if(Adapter->sleep_period.period == 0)
                goto done;
done:
        return ret;
}
