/** @file  wlanconfig.c
  * @brief Program to configure addition paramters into the wlan driver
  * 
  *  Usage: wlanconfig <ethX> <cmd> [...] 
  *
  * (c) Copyright  2003-2006, Marvell International Ltd. 
  * All Rights Reserved
  *
  * This software file (the "File") is distributed by Marvell International 
  * Ltd. under the terms of the GNU General Public License Version 2, June 1991 
  * (the "License").  You may use, redistribute and/or modify this File in 
  * accordance with the terms and conditions of the License, a copy of which 
  * is available along with the File in the license.txt file or by writing to 
  * the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 
  * 02111-1307 or on the worldwide web at http://www.gnu.org/licenses/gpl.txt.
  *
  * THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE 
  * IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE 
  * ARE EXPRESSLY DISCLAIMED.  The License provides additional details about 
  * this warranty disclaimer.
  *
  */
/********************************************************
Change log:
	10/12/05: Add Doxygen format comments
	11/03/05: Load priv ioctls on demand, ifdef code for features in driver
	11/04/05: Add crypto_test 
	12/14/05: Support wildcard SSID in BGSCAN	
	01/11/06: Add getscantable, setuserscan, setmrvltlv, getassocrsp 
	01/31/06: Add support to selectively enabe the FW Scan channel filter	
	02/24/06: fix getscanlist function not work on linux 2.6.15 X86
	04/06/06: Add TSPEC, queue metrics, and MSDU expiry support
	04/10/06: Add hostcmd generic API and power_adapt_cfg_ext command
	04/18/06: Remove old Subscrive Event and add new Subscribe Event
		  implementation through generic hostcmd API
********************************************************/

#include    <stdio.h>
#include    <unistd.h>
#include    <time.h>
#include    <ctype.h>
#include    <sys/types.h>
#include    <sys/socket.h>
#include    <string.h>
#include    <stdlib.h>
#include    <linux/if.h>
#include    <sys/ioctl.h>
#include    <linux/wireless.h>
#include    <linux/if_ether.h>
#include    <linux/byteorder/swab.h>
#include    <errno.h>

typedef unsigned char		u8;
typedef unsigned short  	u16;
typedef unsigned long		u32;
typedef unsigned long long	u64;
typedef signed char		s8;
typedef signed short  		s16;
typedef signed long		s32;

#ifdef 	BYTE_SWAP
#define 	cpu_to_le16(x)	__swab16(x)
#else
#define		cpu_to_le16(x)	(x)
#endif

#ifndef __ATTRIB_ALIGN__
#define __ATTRIB_ALIGN__ __attribute__((aligned(4)))
#endif

#ifndef __ATTRIB_PACK__
#define __ATTRIB_PACK__  __attribute__((packed))
#endif

/*
 *  ctype from older glib installations defines BIG_ENDIAN.  Check it 
 *   and undef it if necessary to correctly process the wlan header
 *   files
 */
#if (BYTE_ORDER == LITTLE_ENDIAN)
#undef BIG_ENDIAN
#endif

#include	"wlan_defs.h"
#include	"wlan_types.h"
#include    "wlan_11d.h"
#include	"host.h"
#include	"hostcmd.h"
#include    "wlan_scan.h"
#include	"wlan_wext.h"
#include	"wlanconfig.h"

#ifndef MIN
#define MIN(a, b) ((a) < (b) ? (a) : (b))
#endif /* MIN */

enum COMMANDS {
	CMD_HOSTCMD,
	CMD_RDMAC,
	CMD_WRMAC,
	CMD_RDBBP,
	CMD_WRBBP,
	CMD_RDRF,
	CMD_WRRF,
	CMD_RDEEPROM,
	CMD_CMD52R,
	CMD_CMD52W,
	CMD_CMD53R,
	CMD_CMD53W,
	CMD_CFREGR,
	CMD_CFREGW,
	CMD_GETRATE,
	CMD_SLEEPPARAMS,
    CMD_BCA_TS,
	CMD_EXTSCAN,
	CMD_SCAN_LIST,
	CMD_GET_SCAN_RSP,
	CMD_SET_USER_SCAN,
};

#define IW_MAX_PRIV_DEF		128

/********************************************************
		Local Variables
********************************************************/
static s8    *commands[] = {
	"hostcmd",
	"rdmac",
	"wrmac",
	"rdbbp",
	"wrbbp",
	"rdrf",
	"wrrf",
	"rdeeprom",
	"sdcmd52r",
	"sdcmd52w",
	"sdcmd53r",
	"sdcmd53w",
	"rdcfreg",
	"wrcfreg",
	"getrate",
	"sleepparams",
	"bca-ts",
	"extscan",
	"getscanlist",
	"getscantable",
	"setuserscan",
};

static s8    *usage[] = {
	"Usage: wlanconfig <ethX> <cmd> [...]",
	"where",
	"	ethX	: wireless network interface",
	"	cmd	: hostcmd, rdmac, wrmac, rdbbp, wrbbp, rdrf, wrrf",
	"		: sdcmd52r, sdcmd52w, sdcmd53r",
	"		: caldataext, rdcfreg, wrcfreg, rdeeprom",
    "		: sleepparams, bca-ts",
	"		: setadhocch, getadhocch",
	"		: getscantable, setuserscan",
	"	[...]	: additional parameters for read registers are",
	"		:	<offset>",
	"		: additional parameters for write registers are",
	"		:	<offset> <value>",
	"		: additonal parameter for hostcmd",
	"		: 	<filename> <cmd>",
	"		: addition parameters for caldataext",
	"		: 	<filename>",
};

static s32  sockfd;
static s8	dev_name[IFNAMSIZ + 1];
static struct iw_priv_args	Priv_args[IW_MAX_PRIV_DEF];
static int we_version_compiled=0;
#define MRV_EV_POINT_OFF (((char *) &(((struct iw_point *) NULL)->length)) - \
			  (char *) NULL)



static s8 * wlan_config_get_line(s8 *s, s32 size, FILE *stream, int *line);

/********************************************************
		Global Variables
********************************************************/


/********************************************************
		Local Functions
********************************************************/
/** 
 *  @brief convert char to hex integer
 * 
 *  @param chr 		char to convert
 *  @return      	hex integer or 0
 */
static int hexval(s32 chr)
{
	if (chr >= '0' && chr <= '9')
		return chr - '0';
	if (chr >= 'A' && chr <= 'F')
		return chr - 'A' + 10;
	if (chr >= 'a' && chr <= 'f')
		return chr - 'a' + 10;

	return 0;
}

/** 
 *  @brief Hump hex data
 *
 *  @param prompt	A pointer prompt buffer
 *  @param p		A pointer to data buffer
 *  @param len		the len of data buffer
 *  @param delim	delim char
 *  @return            	hex integer
 */
static void hexdump(s8 *prompt, void *p, s32 len, s8 delim)
{
	s32             i;
	u8  *s = p;
	
    if (prompt) {
        printf("%s: ", prompt);
    }
	for (i = 0; i < len; i++) {
		if (i != len - 1)
			printf("%02x%c", *s++, delim);
		else
			printf("%02x\n", *s);
	}
}

/** 
 *  @brief convert char to hex integer
 *
 *  @param chr		char
 *  @return            	hex integer
 */
static u8 hexc2bin(s8 chr)
{
	if (chr >= '0' && chr <= '9')
		chr -= '0';
	else if (chr >= 'A' && chr <= 'F')
		chr -= ('A' - 10);
	else if (chr >= 'a' && chr <= 'f')
		chr -= ('a' - 10);

	return chr;
}

/** 
 *  @brief convert string to hex integer
 *
 *  @param s		A pointer string buffer
 *  @return            	hex integer
 */
static u32 a2hex(s8 *s)
{
	u32    val = 0;
	while (*s && isxdigit(*s)) {
		val = (val << 4) + hexc2bin(*s++);
	}

	return val;
}


/* 
 *  @brief convert String to integer
 *  
 *  @param value	A pointer to string
 *  @return             integer
 */
static u32 a2hex_or_atoi(s8 *value)
{
	if (value[0] == '0' && (value[1] == 'X' || value[1] == 'x'))
		return a2hex(value + 2);
	else
		return atoi(value);
}


/** 
 *  @brief convert string to integer
 * 
 *  @param ptr		A pointer to data buffer
 *  @param chr 		A pointer to return integer
 *  @return      	A pointer to next data field
 */
s8 *convert2hex(s8 *ptr, u8 *chr)
{
	u8	val;

	for (val = 0; *ptr && isxdigit(*ptr); ptr++) {
		val = (val * 16) + hexval(*ptr);
	}

	*chr = val;

	return ptr;
}

/** 
 *  @brief Get private info.
 *   
 *  @param ifname   A pointer to net name
 *  @return 	    WLAN_STATUS_SUCCESS--success, otherwise --fail
 */
static int get_private_info(const s8 *ifname)
{
	/* This function sends the SIOCGIWPRIV command which is
	 * handled by the kernel. and gets the total number of
	 * private ioctl's available in the host driver.
	 */
	struct iwreq iwr;
	int s, ret = WLAN_STATUS_SUCCESS;
	struct iw_priv_args *pPriv = Priv_args;

	s = socket(PF_INET, SOCK_DGRAM, 0);
	if (s < 0) {
		perror("socket[PF_INET,SOCK_DGRAM]");
		return WLAN_STATUS_FAILURE;
	}

	memset(&iwr, 0, sizeof(iwr));
	strncpy(iwr.ifr_name, ifname, IFNAMSIZ);
	iwr.u.data.pointer = (caddr_t) pPriv;
	iwr.u.data.length = IW_MAX_PRIV_DEF;
	iwr.u.data.flags = 0;
	      
	if (ioctl(s, SIOCGIWPRIV, &iwr) < 0) {
		perror("ioctl[SIOCGIWPRIV]");
		ret = WLAN_STATUS_FAILURE;
	} else {
		/* Return the number of private ioctls */
		ret = iwr.u.data.length;
	}

	close(s);

	return ret;
}

/** 
 *  @brief Get Sub command ioctl number
 *   
 *  @param i        command index
 *  @param priv_cnt     Total number of private ioctls availabe in driver
 *  @param ioctl_val    A pointer to return ioctl number
 *  @param subioctl_val A pointer to return sub-ioctl number
 *  @return 	        WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int marvell_get_subioctl_no(s32 i, 
                                   s32 priv_cnt,
                                   int *ioctl_val,
                                   int *subioctl_val)
{
	s32 j;

	if (Priv_args[i].cmd >= SIOCDEVPRIVATE) {
        *ioctl_val    = Priv_args[i].cmd;
        *subioctl_val = 0;
		return WLAN_STATUS_SUCCESS;
	}

	j = -1;

	/* Find the matching *real* ioctl */

	while ((++j < priv_cnt) 
           && ((Priv_args[j].name[0] != '\0') ||
               (Priv_args[j].set_args != Priv_args[i].set_args) ||
               (Priv_args[j].get_args != Priv_args[i].get_args))) {
    }

	/* If not found... */
	if (j == priv_cnt) {
		printf("%s: Invalid private ioctl definition for: 0x%x\n",
					dev_name, Priv_args[i].cmd);
		return  WLAN_STATUS_FAILURE;
	}

	/* Save ioctl numbers */
	*ioctl_val    = Priv_args[j].cmd;
	*subioctl_val = Priv_args[i].cmd;

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get ioctl number
 *   
 *  @param ifname   	A pointer to net name
 *  @param priv_cmd	A pointer to priv command buffer
 *  @param ioctl_val    A pointer to return ioctl number
 *  @param subioctl_val A pointer to return sub-ioctl number
 *  @return 	        WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int marvell_get_ioctl_no(const s8 *ifname, 
                                const s8 *priv_cmd, 
                                int *ioctl_val,
                                int *subioctl_val)
{
	s32	i;
    s32 priv_cnt;

    priv_cnt = get_private_info(ifname);
    
	/* Are there any private ioctls? */
	if (priv_cnt <= 0) {
		/* Could skip this message ? */
		printf("%-8.8s  no private ioctls.\n", ifname);
	} else {
		for (i = 0; i < priv_cnt; i++) {
			if (Priv_args[i].name[0] && !strcmp(Priv_args[i].name, priv_cmd)) {
                return marvell_get_subioctl_no(i, priv_cnt, 
                                               ioctl_val, subioctl_val);
			}
		}
	}

    return WLAN_STATUS_FAILURE;
}

/** 
 *  @brief Retrieve the ioctl and sub-ioctl numbers for the given ioctl string
 *   
 *  @param ifname       Private IOCTL string name
 *  @param ioctl_val    A pointer to return ioctl number
 *  @param subioctl_val A pointer to return sub-ioctl number
 *
 *  @return             WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int get_priv_ioctl(char* ioctl_name, int* ioctl_val, int* subioctl_val)
{
    int retval;

    retval = marvell_get_ioctl_no(dev_name,
                                  ioctl_name,
                                  ioctl_val,
                                  subioctl_val);
    
#if 0
    /* Debug print discovered IOCTL values */
    printf("ioctl %s: %x, %x\n", ioctl_name, *ioctl_val, *subioctl_val);
#endif
    
    return retval;
}


/** 
 *  @brief  get range 
 *   
 *  @return	WLAN_STATUS_SUCCESS--success, otherwise --fail
 */
static int get_range(void) 
{
	struct iw_range *range;
	struct iwreq	iwr;
	size_t buflen;
	WCON_HANDLE	mhandle, *pHandle = &mhandle;

	buflen = sizeof(struct iw_range) + 500;
	range = malloc(buflen);
	if (range == NULL)
		return WLAN_STATUS_FAILURE;
	memset(range, 0, buflen);
	
	memset(pHandle, 0, sizeof(WCON_HANDLE));
	memset(&iwr, 0, sizeof(struct iwreq));
	
	iwr.u.data.pointer = (caddr_t) range;
	iwr.u.data.length = buflen;

	strncpy(iwr.ifr_name, dev_name, IFNAMSIZ);

	if ((ioctl(sockfd, SIOCGIWRANGE, &iwr)) < 0) {
		printf("Get Range Results Failed\n");
		free(range);
		return WLAN_STATUS_FAILURE;
	}
	we_version_compiled = range->we_version_compiled;
	printf("Driver build with Wireless Extension %d\n",range->we_version_compiled);
	free(range);
	return WLAN_STATUS_SUCCESS;
}
#define WLAN_MAX_RATES	14
#define	GIGA		1e9
#define	MEGA		1e6
#define	KILO		1e3

/** 
 *  @brief print bit rate
 *   
 *  @param rate  	rate to be print
 *  @param current      if current is TRUE, data rate not need convert
 *  @param fixed        not used
 *  @return 	        WLAN_STATUS_SUCCESS
 */
static int print_bitrate(double rate, s32 current, s32 fixed) 
{
	s8	scale = 'k', buf[128];
	s32	divisor = KILO;
	
	if (!current)
		rate *= 500000;

	if (rate >= GIGA) {
		scale = 'G';
		divisor = GIGA;
	} else if (rate >= MEGA) {
		scale = 'M';
		divisor = MEGA;
	}

	snprintf(buf, sizeof(buf), "%g %cb/s", rate/divisor, scale);

	if (current) {
		printf("\t  Current Bit Rate%c%s\n\n",
					(fixed) ? '=' : ':', buf);
	} else {
		printf("\t  %s\n", buf);
	}

	return WLAN_STATUS_SUCCESS;
}

/* 
 *  @brief get hostcmd data
 *  
 *  @param fp			A pointer to file stream
 *  @param ln			A pointer to line number
 *  @param buf			A pointer to hostcmd data
 *  @param size			A pointer to the return size of hostcmd buffer
 *  @return      		WLAN_STATUS_SUCCESS
 */
static int wlan_get_hostcmd_data(FILE *fp, int *ln, u8 *buf, u16 *size)
{
	s32	errors = 0, i;
	s8	line[256], *pos, *pos1, *pos2, *pos3;
	u16	len;


	while ((pos = wlan_config_get_line(line, sizeof(line), fp, ln))) {
		(*ln)++;
		if (strcmp(pos, "}") == 0) {
			break;
		}

		pos1 = strchr(pos, ':');
		if (pos1 == NULL) {
			printf("Line %d: Invalid hostcmd line '%s'\n", *ln, pos);
			errors++;
			continue;
		}
		*pos1++ = '\0';

		pos2 = strchr(pos1, '=');
		if (pos2 == NULL) {
			printf("Line %d: Invalid hostcmd line '%s'\n", *ln, pos);
			errors++;
			continue;
		}
		*pos2++ = '\0';

		len = a2hex_or_atoi(pos1);
		if (len < 1 || len > MRVDRV_SIZE_OF_CMD_BUFFER) {
			printf("Line %d: Invalid hostcmd line '%s'\n", *ln, pos);
			errors++;
			continue;
		}

		*size += len;

		if (*pos2 == '"') {
			pos2++;
			if ((pos3=strchr(pos2, '"')) == NULL) {
				printf("Line %d: invalid quotation '%s'\n", *ln, pos);
				errors++;
				continue;
			}
			*pos3 = '\0';
			memset(buf, 0, len);
			memmove(buf, pos2, MIN(strlen(pos2),len));
			buf += len;
		}
		else if (*pos2 == '\'') {
			pos2++;
			if ((pos3=strchr(pos2, '\'')) == NULL) {
				printf("Line %d: invalid quotation '%s'\n", *ln, pos);
				errors++;
				continue;
			}
			*pos3 = ',';
			for (i=0; i<len; i++) {
				if ((pos3=strchr(pos2, ',')) != NULL) {
					*pos3 = '\0';
					*buf++ = (u8)a2hex_or_atoi(pos2);
					pos2 = pos3 + 1;
				}
				else
					*buf++ = 0;
			}
		}
		else if (*pos2 == '{') {
			u16 *tlvlen = (u16 *)buf;
			wlan_get_hostcmd_data(fp, ln, buf+len, tlvlen);
			*size += *tlvlen;
			buf += len + *tlvlen;
		}
		else {
			u32 value = a2hex_or_atoi(pos2);
			while (len--) {
				*buf++ = (u8)(value & 0xff);
				value >>= 8;
			}
		}
	}
	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Process host_cmd 
 *  @param hostcmd      A pointer to HostCmd_DS_GEN data structure
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_host_cmd(int argc, char *argv[])
{
	u8		line[256], cmdname[256], *buf, *pos;
	FILE		*fp;
	HostCmd_DS_GEN	*hostcmd;
	struct ifreq	userdata;
	int 		ln = 0;
	int		cmdname_found = 0, cmdcode_found = 0;
	int		ret = WLAN_STATUS_SUCCESS;

	if (argc < 5) {
		printf("Error: invalid no of arguments\n");
		printf("Syntax: ./wlanconfig eth1 hostcmd <hostcmd.conf> <cmdname>\n");
		exit(1);
	}
	
	if ((fp = fopen(argv[3], "r")) == NULL) {
		fprintf(stderr, "Cannot open file %s\n", argv[4]);
		exit(1);
	}

	buf = (u8 *)malloc(MRVDRV_SIZE_OF_CMD_BUFFER);
	memset(buf, 0, MRVDRV_SIZE_OF_CMD_BUFFER);
	hostcmd = (PHostCmd_DS_GEN)buf;

	hostcmd->Command = 0xffff;

	sprintf(cmdname, "%s={", argv[4]);
	cmdname_found = 0;
	while ((pos = wlan_config_get_line(line, sizeof(line), fp, &ln))) {
		if (strcmp(pos, cmdname) == 0) {
			cmdname_found = 1;
			sprintf(cmdname, "CmdCode=");
			cmdcode_found = 0;
			while ((pos = wlan_config_get_line(line, sizeof(line), fp, &ln))) {
				if (strncmp(pos, cmdname, strlen(cmdname)) == 0) {
					cmdcode_found = 1;
					hostcmd->Command = a2hex_or_atoi(pos+strlen(cmdname));
					hostcmd->Size = S_DS_GEN;
					wlan_get_hostcmd_data(fp, &ln, buf+hostcmd->Size, &hostcmd->Size);
					break;
				}
			}
			if (!cmdcode_found) {
				fprintf(stderr, "wlanconfig: CmdCode not found in file '%s'\n", argv[3]);
			}
			break;
		}
	}

	fclose(fp);

	if (!cmdname_found)
		fprintf(stderr, "wlanconfig: cmdname '%s' not found in file '%s'\n", argv[4],argv[3]);

	if (!cmdname_found || !cmdcode_found) {
		ret = -1;
		goto _exit_;
	}

	buf = (u8 *)hostcmd;

	hostcmd->SeqNum = 0;
	hostcmd->Result = 0;

	strncpy(userdata.ifr_name, dev_name, IFNAMSIZ);
	userdata.ifr_data = (u8 *)hostcmd;

	if (ioctl(sockfd, WLANHOSTCMD, &userdata)) {
		fprintf(stderr, "wlanconfig: WLANHOSTCMD is not supported by %s\n", dev_name);
		ret = -1;
		goto _exit_;
   	}

	if (!hostcmd->Result) {
		switch (hostcmd->Command) {
		case HostCmd_RET_802_11_SUBSCRIBE_EVENT:
		{
			HostCmd_DS_802_11_SUBSCRIBE_EVENT *se = (HostCmd_DS_802_11_SUBSCRIBE_EVENT *)(buf + S_DS_GEN);
			if (se->Action == HostCmd_ACT_GET) {
				int len = S_DS_GEN + sizeof(HostCmd_DS_802_11_SUBSCRIBE_EVENT);
				printf("\nEvent\t\tValue\tFreq\tsubscribed\n\n");
				while (len < hostcmd->Size) {
					MrvlIEtypesHeader_t *header = (MrvlIEtypesHeader_t *)(buf + len);
					switch (header->Type) {
					case TLV_TYPE_RSSI_LOW:
					{
						MrvlIEtypes_RssiParamSet_t *LowRssi = (MrvlIEtypes_RssiParamSet_t *)(buf + len);
						printf("Low RSSI\t%d\t%d\t%s\n",LowRssi->RSSIValue,LowRssi->RSSIFreq,(se->Events & 0x0001)?"yes":"no");
						len += sizeof(MrvlIEtypes_RssiParamSet_t);
						break;
					}
					case TLV_TYPE_SNR_LOW:
					{
						MrvlIEtypes_SnrThreshold_t *LowSnr = (MrvlIEtypes_SnrThreshold_t *)(buf + len);
						printf("Low SNR\t\t%d\t%d\t%s\n",LowSnr->SNRValue,LowSnr->SNRFreq,(se->Events & 0x0002)?"yes":"no");
						len += sizeof(MrvlIEtypes_SnrThreshold_t);
						break;
					}
					case TLV_TYPE_FAILCOUNT:
					{
						MrvlIEtypes_FailureCount_t *FailureCount = (MrvlIEtypes_FailureCount_t *)(buf + len);
						printf("Failure Count\t%d\t%d\t%s\n",FailureCount->FailValue,FailureCount->FailFreq,(se->Events & 0x0004)?"yes":"no");
						len += sizeof(MrvlIEtypes_FailureCount_t);
						break;
					}
					case TLV_TYPE_BCNMISS:
					{
						MrvlIEtypes_BeaconsMissed_t *BcnMissed = (MrvlIEtypes_BeaconsMissed_t *)(buf + len);
						printf("Beacon Missed\t%d\tN/A\t%s\n",BcnMissed->BeaconMissed,(se->Events & 0x0008)?"yes":"no");
						len += sizeof(MrvlIEtypes_BeaconsMissed_t);
						break;
					}
					case TLV_TYPE_RSSI_HIGH:
					{
						MrvlIEtypes_RssiParamSet_t *HighRssi = (MrvlIEtypes_RssiParamSet_t *)(buf + len);
						printf("High RSSI\t%d\t%d\t%s\n",HighRssi->RSSIValue,HighRssi->RSSIFreq,(se->Events & 0x0010)?"yes":"no");
						len += sizeof(MrvlIEtypes_RssiParamSet_t);
						break;
					}
					case TLV_TYPE_SNR_HIGH:
					{
						MrvlIEtypes_SnrThreshold_t *HighSnr = (MrvlIEtypes_SnrThreshold_t *)(buf + len);
						printf("High SNR\t%d\t%d\t%s\n",HighSnr->SNRValue,HighSnr->SNRFreq,(se->Events & 0x0020)?"yes":"no");
						len += sizeof(MrvlIEtypes_SnrThreshold_t);
						break;
					}
					default:
						printf("unknown subscribed event TLV Type=%#x, Len=%d\n", header->Type, header->Len);
						len += sizeof(MrvlIEtypesHeader_t) + header->Len;
						break;
					}
				}
			}
			break;
		}
		default:
			printf("HOSTCMD_RESP: ReturnCode=%#04x, Result=%#04x\n",hostcmd->Command,hostcmd->Result);
			break;
		}
	}
	else {
		printf("HOSTCMD failed: ReturnCode=%#04x, Result=%#04x\n",hostcmd->Command,hostcmd->Result);
	}

_exit_:
	if (buf)
		free(buf);

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Get Rate
 *   
 *  @return      WLAN_STATUS_SUCCESS--success, otherwise --fail
 */
static int process_get_rate(void)
{
	u32	bitrate[WLAN_MAX_RATES];
	struct iwreq	iwr;
	s32		i = 0;
    int ioctl_val, subioctl_val;

    if (get_priv_ioctl("getrate", 
                       &ioctl_val, &subioctl_val) == WLAN_STATUS_FAILURE) {
		return -EOPNOTSUPP;
    }
    
	memset(&iwr, 0, sizeof(iwr));
	strncpy(iwr.ifr_name, dev_name, IFNAMSIZ);
	iwr.u.data.pointer = (caddr_t) bitrate;
	iwr.u.data.length = sizeof(bitrate);
	iwr.u.data.flags = subioctl_val;

	if (ioctl(sockfd, ioctl_val, &iwr) < 0) {
		perror("wlanconfig");
		return WLAN_STATUS_FAILURE;
	}

	printf("%-8.16s  %d available bit-rates :\n",
					dev_name, iwr.u.data.length);
	
	for (i = 0; i < iwr.u.data.length; i++) {
		print_bitrate(bitrate[i], 0, 0);
	}

	if (ioctl(sockfd, SIOCGIWRATE, &iwr)) {
		perror("wlanconfig");
		return WLAN_STATUS_FAILURE;
	}

	print_bitrate(iwr.u.bitrate.value, 1, iwr.u.bitrate.fixed);
	
	return  WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Check the Hex String
 *  @param s		A pointer to the string      
 *  @return      	0--HexString, -1--not HexString
 */
static int ishexstring(s8 *s)
{
	int ret = -1;
	s32 tmp;
	
	while(*s) {
		tmp = toupper(*s);
		if (tmp >= 'A' && tmp <= 'F') {
			ret = 0;
			break;
		}
		s++;
	}
	
	return ret;
}

/** 
 *  @brief Convert String to Integer
 *  @param buf		A pointer to the string      
 *  @return      	Integer
 */
static int atoval(s8 *buf)
{
	if (!strncasecmp(buf, "0x", 2))
		return a2hex(buf+2);
	else if (!ishexstring(buf))
		return a2hex(buf);
	else 	
		return atoi(buf);
}


/** 
 *  @brief Display sleep params
 *  @param sp		A pointer to wlan_ioctl_sleep_params_config structure    
 *  @return      	NA
 */
void display_sleep_params(wlan_ioctl_sleep_params_config *sp)
{
	printf("Sleep Params for %s:\n", sp->Action ? "set" : "get");
   	printf("----------------------------------------\n");
	printf("Error		: %u\n", sp->Error);
	printf("Offset		: %u\n", sp->Offset);
	printf("StableTime	: %u\n", sp->StableTime);
	printf("CalControl	: %u\n", sp->CalControl);
	printf("ExtSleepClk	: %u\n", sp->ExtSleepClk);
	printf("Reserved	: %u\n", sp->Reserved);
}

/** 
 *  @brief Process sleep params
 *  @param argc		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_sleep_params(int argc, char *argv[])
{
	struct iwreq			iwr;
	int				ret;
	wlan_ioctl_sleep_params_config 	sp;
    int ioctl_val, subioctl_val;

    if (get_priv_ioctl("sleepparams",
                       &ioctl_val, &subioctl_val) == WLAN_STATUS_FAILURE) {
		return -EOPNOTSUPP;
    }

	if (argc < 4) {
		printf("Error: invalid no of arguments\n");
		printf("Syntax: ./wlanconfig eth1 sleepparams get/set <p1>"
				" <p2> <p3> <p4> <p5> <p6>\n");
		exit(1);
	}

	memset(&sp, 0, sizeof(wlan_ioctl_sleep_params_config));
	if (!strcmp(argv[3], "get")) {
		sp.Action = 0;
	} else if (!strncmp(argv[3], "set", 3)) {
		if (argc != 10) {
			printf("Error: invalid no of arguments\n");
			printf("Syntax: ./wlanconfig eth1 sleepparams get/set" 
					"<p1> <p2> <p3> <p4> <p5> <p6>\n");
			exit(1);
		}

		sp.Action = 1;
		if ((ret = atoval(argv[4])) < 0)
			return -EINVAL;
		sp.Error = (u16) ret;
		if ((ret = atoval(argv[5])) < 0)
			return -EINVAL;
		sp.Offset = (u16) ret;
		if ((ret = atoval(argv[6])) < 0)
			return -EINVAL;
		sp.StableTime = (u16) ret;
		if ((ret = atoval(argv[7])) < 0)
			return -EINVAL;
		sp.CalControl = (u8) ret;
		if ((ret = atoval(argv[8])) < 0)
			return -EINVAL;
		sp.ExtSleepClk = (u8) ret;
		if ((ret = atoval(argv[9])) < 0)
			return -EINVAL;
		sp.Reserved = (u16) ret;
	} else 	{
		return -EINVAL;
	}

	memset(&iwr, 0, sizeof(iwr));
	
	strncpy(iwr.ifr_name, dev_name, IFNAMSIZ);
	iwr.u.data.pointer = (caddr_t) &sp;
	iwr.u.data.length = sizeof(wlan_ioctl_sleep_params_config);
    iwr.u.data.flags = subioctl_val;

	if (ioctl(sockfd, ioctl_val, &iwr) < 0) {
		perror("wlanconfig");
		return -1;
	}

	display_sleep_params(&sp);

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief Display BCA Time Share Params
 *  @param sp		A point to wlan_ioctl_bca_timeshare_config structure    
 *  @return      	NA
 */
static void display_bca_ts_params(wlan_ioctl_bca_timeshare_config *bca_ts)
{
	printf("BCA Time Share Params for %s:\n", bca_ts->Action?"set" : "get");
   	printf("----------------------------------------\n");
	printf("TrafficType		: %u\n", bca_ts->TrafficType);	
	printf("TimeShareInterval	: %lu\n", bca_ts->TimeShareInterval);	
	printf("BTTime			: %lu\n", bca_ts->BTTime);	
}

/** 
 *  @brief Process BCA Time Share Params
 *  @param argc		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_bca_ts(int argc, char *argv[]) 
{
	int				ret, i;
	struct iwreq			iwr;
	wlan_ioctl_bca_timeshare_config	bca_ts;
    int ioctl_val, subioctl_val;

    if (get_priv_ioctl("bca-ts",
                       &ioctl_val, &subioctl_val) == WLAN_STATUS_FAILURE) {
		return -EOPNOTSUPP;
    }

	if (argc < 5) {
		printf("Error: invalid no of arguments\n");
		printf("Syntax: ./wlanconfig eth1 bca_ts get/set <p1>"
								" <p2> <p3>\n");
		exit(1);
	}

	memset(&bca_ts, 0, sizeof(wlan_ioctl_bca_timeshare_config));
	
	if ((ret = atoval(argv[4])) < 0)
		return -EINVAL;
	if (ret > 1)
		return -EINVAL;
	bca_ts.TrafficType = (u16) ret; // 0 or 1
	
	if (!strcmp(argv[3], "get")) {
		bca_ts.Action = 0;
	} else if (!strncmp(argv[3], "set", 3)) {
		if (argc != 7) {
			printf("Error: invalid no of arguments\n");
			printf("Syntax: ./wlanconfig eth1 bca_ts get/set" 
							" <p1> <p2> <p3>\n");
			exit(1);
		}

		bca_ts.Action = 1;
		
		if ((ret = atoval(argv[5])) < 0)
			return -EINVAL;
		/* If value is not multiple of 10 then take the floor value */
		i = ret % 10;
		ret -= i;
		/* Valid Range for TimeShareInterval: < 20 ... 60_000 > ms */
		if (ret < 20  || ret > 60000) {
			printf("Invalid TimeShareInterval Range:"
						" < 20 ... 60000 > ms\n");
			return -EINVAL;
		}
		bca_ts.TimeShareInterval = (u32) ret;
		
		if ((ret = atoval(argv[6])) < 0)
			return -EINVAL;
		/* If value is not multiple of 10 then take the floor value */
		i = ret % 10;
		ret -= i;
		
		if (ret > bca_ts.TimeShareInterval) {
			printf("Invalid BTTime"
				"  Range: < 0 .. TimeShareInterval > ms\n");
			return -EINVAL;
		}
		bca_ts.BTTime = (u32) ret;
	} else 	{
		return -EINVAL;
	}

	memset(&iwr, 0, sizeof(iwr));
	
	strncpy(iwr.ifr_name, dev_name, IFNAMSIZ);
	iwr.u.data.pointer = (caddr_t) &bca_ts;
	iwr.u.data.length = sizeof(wlan_ioctl_bca_timeshare_config);
    iwr.u.data.flags = subioctl_val;

	if (ioctl(sockfd, ioctl_val, &iwr) < 0) {
		perror("wlanconfig");
		return -1;
	}

	display_bca_ts_params(&bca_ts);

	return WLAN_STATUS_SUCCESS;
}





/**
 *  @brief Retrieve and display the contents of the driver scan table.
 *
 *  The ioctl to retrieve the scan table contents will be invoked, and portions
 *   of the scan data will be displayed on stdout.  The entire beacon or 
 *   probe response is also retrieved (if available in the driver).  This 
 *   data would be needed in case the application was explicitly controlling
 *   the association (inserting IEs, TLVs, etc).
 *
 *  @param argc     number of arguments
 *  @param argv     A pointer to arguments array    
 *
 *  @return         WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_getscantable(int argc, char *argv[])
{
    int ioctl_val, subioctl_val;
	struct iwreq iwr;
    u8 scanRspBuffer[500]; /* Stack buffer can be as large as ioctl allows */

    uint scanStart;
    uint idx;

    u8* pCurrent;
    u8* pNext;
    IEEEtypes_ElementId_e* pElementId;
    u8* pElementLen;
    int bssInfoLen;
    int ssidIdx;
    u16 tmpCap;
    u8* pByte;

    IEEEtypes_CapInfo_t capInfo;
    u8 tsf[8];
    u16 beaconInterval;

    wlan_ioctl_get_scan_table_info* pRspInfo;
    wlan_ioctl_get_scan_table_entry* pRspEntry;

    pRspInfo = (wlan_ioctl_get_scan_table_info*)scanRspBuffer;
    
    if (get_priv_ioctl("getscantable", 
                       &ioctl_val, &subioctl_val) == WLAN_STATUS_FAILURE) {
		return -EOPNOTSUPP;
    }

    scanStart = 0;

    printf("---------------------------------------");
    printf("---------------------------------------\n");
    printf("# | ch  | ss  |       bssid       |   cap    |   SSID \n");
    printf("---------------------------------------");
    printf("---------------------------------------\n");

    do {
        pRspInfo->scanNumber = scanStart;
        
        /* 
         * Set up and execute the ioctl call
         */
        strncpy(iwr.ifr_name, dev_name, IFNAMSIZ);
        iwr.u.data.pointer = (caddr_t)pRspInfo;
        iwr.u.data.length  = sizeof(scanRspBuffer);
        iwr.u.data.flags   = subioctl_val;
        
        if (ioctl(sockfd, ioctl_val, &iwr) < 0) {
            perror("wlanconfig: getscantable ioctl");
            return -EFAULT;
        }

        pCurrent = 0;
        pNext = pRspInfo->scan_table_entry_buffer;

        for (idx = 0; idx < pRspInfo->scanNumber; idx++) {
            
            /* 
             * Set pCurrent to pNext in case pad bytes are at the end
             *   of the last IE we processed.
             */
            pCurrent = pNext;  

            pRspEntry = (wlan_ioctl_get_scan_table_entry*)pCurrent;
            
            printf("%02u| %03d | %03d | %02x:%02x:%02x:%02x:%02x:%02x |",
                   scanStart + idx,
                   pRspEntry->fixedFields.channel,
                   pRspEntry->fixedFields.rssi,
                   pRspEntry->fixedFields.bssid[0],
                   pRspEntry->fixedFields.bssid[1],
                   pRspEntry->fixedFields.bssid[2],
                   pRspEntry->fixedFields.bssid[3],
                   pRspEntry->fixedFields.bssid[4],
                   pRspEntry->fixedFields.bssid[5]);

#if 0
            printf("fixed = %u, bssInfo = %u\n", 
                   (unsigned int)pRspEntry->fixedFieldLength,
                   (unsigned int)pRspEntry->bssInfoLength);
#endif
                
            pCurrent += (sizeof(pRspEntry->fixedFieldLength) +
                         pRspEntry->fixedFieldLength);

            bssInfoLen = pRspEntry->bssInfoLength;
            pCurrent  += sizeof(pRspEntry->bssInfoLength);
            pNext      = pCurrent + pRspEntry->bssInfoLength;

            if (bssInfoLen >= (sizeof(tsf) 
                               + sizeof(beaconInterval) + sizeof(capInfo))) {
                /* time stamp is 8 byte long */
                memcpy(tsf, pCurrent, sizeof(tsf));
                pCurrent   += sizeof(tsf);
                bssInfoLen -= sizeof(tsf);
            
                /* beacon interval is 2 byte long */
                memcpy(&beaconInterval, pCurrent, sizeof(beaconInterval));
                pCurrent   += sizeof(beaconInterval);
                bssInfoLen -= sizeof(beaconInterval);
                
                /* capability information is 2 byte long */
                memcpy(&capInfo, pCurrent, sizeof(capInfo));
                memcpy(&tmpCap, pCurrent, sizeof(tmpCap));
                pCurrent   += sizeof(capInfo);
                bssInfoLen -= sizeof(capInfo);
            
                printf(" %04x-", tmpCap);

                printf("%c%c%c | ",
                       capInfo.Ibss ? 'A' : 'I',
                       capInfo.Privacy ? 'P' : ' ',
                       capInfo.SpectrumMgmt ? 'S' : ' ');
            } else {
                printf("          | ");
            }

            while (bssInfoLen >= 2) {
                pElementId  = (IEEEtypes_ElementId_e*)pCurrent;
                pElementLen = pCurrent + 1;
                pCurrent   += 2;
                
                switch (*pElementId) {
                    
                case SSID:
                    if (*pElementLen && 
                        *pElementLen <= MRVDRV_MAX_SSID_LENGTH) {
                        for (ssidIdx = 0; ssidIdx < *pElementLen; ssidIdx++) {
                            if (isprint(*(pCurrent + ssidIdx))) {
                                printf("%c", *(pCurrent + ssidIdx));
                            } else {
                                printf("\\%02x", *(pCurrent + ssidIdx));
                            }
                        }
                    }
                    break;
                    
                default:
#if 0
                    printf("% d(%d), bil=%d\n", 
                           *pElementId, *pElementLen, bssInfoLen);
#endif
                    break;
                }
                
                pCurrent   += *pElementLen;
                bssInfoLen -= (2 + *pElementLen);
            }
            
            printf("\n");
            
            if (argc > 3) {
                /* TSF is a u64, some formatted printing libs have
                 *   trouble printing long longs, so cast and dump as bytes
                 */
                pByte = (u8*)&pRspEntry->fixedFields.networkTSF;
                printf("    TSF=%02x%02x%02x%02x%02x%02x%02x%02x\n",
                       pByte[7], pByte[6], pByte[5], pByte[4], 
                       pByte[3], pByte[2], pByte[1], pByte[0]);
            }
        }
        
        scanStart += pRspInfo->scanNumber;
        
        
    } while (pRspInfo->scanNumber);
    
    return WLAN_STATUS_SUCCESS;
}

/**
 *  @brief Request a scan from the driver and display the scan table afterwards
 *
 *  Command line interface for performing a specific immediate scan based
 *    on the following keyword parsing:
 *
 *     chan=[chan#][band][mode] where band is [a,b,g] and mode is 
 *                              blank for active or 'p' for passive
 *     bssid=xx:xx:xx:xx:xx:xx  specify a BSSID filter for the scan
 *     ssid="[SSID]"            specify a SSID filter for the scan
 *     keep=[0 or 1]            keep the previous scan results (1), discard (0)
 *     dur=[scan time]          time to scan for each channel in milliseconds
 *     probes=[#]               number of probe requests to send on each chan
 *     type=[1,2,3]             BSS type: 1 (Infra), 2(Adhoc), 3(Any)
 *
 *  Any combination of the above arguments can be supplied on the command line.
 *    If the chan token is absent, a full channel scan will be completed by 
 *    the driver.  If the dur or probes tokens are absent, the driver default
 *    setting will be used.  The bssid and ssid fields, if blank, 
 *    will produce an unfiltered scan. The type field will default to 3 (Any)
 *    and the keep field will default to 0 (Discard).  
 *
 *  @param argc     number of arguments
 *  @param argv     A pointer to arguments array    
 *
 *  @return         WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_setuserscan(int argc, char *argv[]) 
{
    wlan_ioctl_user_scan_cfg scanReq;
    int ioctl_val, subioctl_val;
	struct iwreq iwr;
    char* pArgTok;
    char* pChanTok;
    char* pArgCookie;
    char* pChanCookie;
    int argIdx;
    int chanParseIdx;
    int chanCmdIdx;
    char chanScratch[10];
    char* pScratch;
    int tmpIdx;
	unsigned int mac[MRVDRV_ETH_ADDR_LEN];
    int scanTime;

    memset(&scanReq, 0x00, sizeof(scanReq));
    chanCmdIdx = 0;
    scanTime = 0;
    
    if (get_priv_ioctl("setuserscan",
                       &ioctl_val, &subioctl_val) == WLAN_STATUS_FAILURE) {
		return -EOPNOTSUPP;
    }
    
    for (argIdx = 0; argIdx < argc; argIdx++) {
        if (strncmp(argv[argIdx], "chan=", strlen("chan=")) == 0) {
            /* 
             *  "chan" token string handler
             */
            pArgTok = argv[argIdx] + strlen("chan=");

            while ((pArgTok = strtok_r(pArgTok, ",", &pArgCookie)) != NULL) {
                
                memset(chanScratch, 0x00, sizeof(chanScratch));
                pScratch = chanScratch;

                for (chanParseIdx = 0; 
                     chanParseIdx < strlen(pArgTok); chanParseIdx++) {
                    if (isalpha(*(pArgTok + chanParseIdx))) {
                        *pScratch++ = ' ';
                    }

                    *pScratch++ = *(pArgTok + chanParseIdx);
                }
                *pScratch = 0;
                pArgTok = NULL;
                
                pChanTok = chanScratch;
                
                while ((pChanTok = strtok_r(pChanTok, " ",
                                            &pChanCookie)) != NULL) {
                    if (isdigit(*pChanTok)) {
                        scanReq.chanList[chanCmdIdx].chanNumber
                            = atoi(pChanTok);
                    } else {
                        switch (toupper(*pChanTok)) 
                        {
                        case 'A':
                            scanReq.chanList[chanCmdIdx].radioType = 1;
                            break;
                        case 'B':
                        case 'G':
                            scanReq.chanList[chanCmdIdx].radioType = 0;
                            break;
                        case 'P':
                            scanReq.chanList[chanCmdIdx].scanType = 1;
                            break;
                        }
                    }
                    pChanTok = NULL;
                }
                chanCmdIdx++;
            }
        } else if (strncmp(argv[argIdx], "bssid=", strlen("bssid=")) == 0) {
            /* 
             *  "bssid" token string handler
             */
            sscanf(argv[argIdx] + strlen("bssid="), "%2x:%2x:%2x:%2x:%2x:%2x",
					 mac + 0, mac + 1, mac + 2, mac + 3, mac + 4, mac + 5);

            for (tmpIdx = 0; tmpIdx < NELEMENTS(mac); tmpIdx++){
                scanReq.specificBSSID[tmpIdx] = (u8)mac[tmpIdx];
            }
        } else if (strncmp(argv[argIdx], "keep=", strlen("keep=")) == 0) {
            /* 
             *  "keep" token string handler
             */
            scanReq.keepPreviousScan = atoi(argv[argIdx] + strlen("keep="));
        } else if (strncmp(argv[argIdx], "dur=", strlen("dur=")) == 0) {
            /* 
             *  "dur" token string handler
             */
            scanTime = atoi(argv[argIdx] + strlen("dur="));
        } else if (strncmp(argv[argIdx], "ssid=", strlen("ssid=")) == 0) {
            /* 
             *  "ssid" token string handler
             */
            strncpy(scanReq.specificSSID, argv[argIdx] + strlen("ssid="),
                    sizeof(scanReq.specificSSID));
        } else if (strncmp(argv[argIdx], "probes=", strlen("probes=")) == 0) {
            /* 
             *  "probes" token string handler
             */
            scanReq.numProbes = atoi(argv[argIdx] + strlen("probes="));
        } else if (strncmp(argv[argIdx], "type=", strlen("type=")) == 0) {
            /* 
             *  "type" token string handler
             */
            scanReq.bssType = atoi(argv[argIdx] + strlen("type="));
            switch (scanReq.bssType) 
            {
            case WLAN_SCAN_BSS_TYPE_BSS:
            case WLAN_SCAN_BSS_TYPE_IBSS:
                break;

            default:
            case WLAN_SCAN_BSS_TYPE_ANY:
                /* Set any unknown types to ANY */
                scanReq.bssType = WLAN_SCAN_BSS_TYPE_ANY;
            }
        }
    }

    /*
     * Update all the channels to have the same scan time
     */
    for (tmpIdx = 0; tmpIdx < chanCmdIdx; tmpIdx++){
        scanReq.chanList[tmpIdx].scanTime = scanTime;
    }

    strncpy(iwr.ifr_name, dev_name, IFNAMSIZ);
    iwr.u.data.pointer = (caddr_t)&scanReq;
    iwr.u.data.length  = sizeof(scanReq);
    iwr.u.data.flags   = subioctl_val;

    if (ioctl(sockfd, ioctl_val, &iwr) < 0) {
        perror("wlanconfig: setuserscan ioctl");
        return -EFAULT;
    }

    process_getscantable(0, 0);

    return WLAN_STATUS_SUCCESS;
}



/** 
 *  @brief scan network with specific ssid
 *  @param argc		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_extscan(int argc, char *argv[]) 
{
	struct iwreq			iwr;
	WCON_SSID 			Ssid;
    int ioctl_val, subioctl_val;

    if (get_priv_ioctl("extscan",
                       &ioctl_val, &subioctl_val) == WLAN_STATUS_FAILURE) {
		return -EOPNOTSUPP;
    }
    
    if (argc != 4) {
		printf("Error: invalid no of arguments\n");
		printf("Syntax: ./wlanconfig eth1 extscan <SSID>\n");
		exit(1);
	}

    printf("Ssid: %s\n", argv[3]);

	memset(&Ssid, 0, sizeof(Ssid));
	memset(&iwr, 0, sizeof(iwr));

	Ssid.ssid_len = strlen(argv[3]);
	memcpy(Ssid.ssid, argv[3], Ssid.ssid_len);

	strncpy(iwr.ifr_name, dev_name, IFNAMSIZ);
	iwr.u.data.pointer = (caddr_t) &Ssid;
	iwr.u.data.length = sizeof(Ssid);
    iwr.u.data.flags = subioctl_val;

	if (ioctl(sockfd, ioctl_val, &iwr) < 0) {
		perror("wlanconfig");
		return -1;
	}

	return WLAN_STATUS_SUCCESS;
}

#if WIRELESS_EXT > 14
/** 
 *  @brief parse custom info
 *  @param pHandle	A pointer to WCON_HANDLE
 *  @param data         A pointer to iw_point structure
 *  @param idx          AP index
 *  @return      	NA
 */
static void parse_custom_info(WCON_HANDLE *pHandle, struct iw_point *data, s32 idx)
{
	s32	i = 0;
	s8 	*custom_cmd[] = { "wpa_ie", "rsn_ie", NULL };
	
	if (!data->pointer || !data->length) {
		printf("iw_point: Invalid Pointer/Length\n");
		return;
	}
	
	if (!strncmp(data->pointer, "wmm_ie", strlen("wmm_ie"))) {
			pHandle->ScanList[idx].Wmm = WCON_WMM_ENABLED;
	}

	while (custom_cmd[i]) {
		if (!strncmp(data->pointer, custom_cmd[i], 
					strlen(custom_cmd[i]))) {
			pHandle->ScanList[idx].WpaAP = WCON_WPA_ENABLED;
			break;
		}
		i++;
	}

	printf("Wpa:\t %s\n", pHandle->ScanList[idx].WpaAP ?
					"enabled" : "disabled");
	printf("Wmm:\t %s\n", pHandle->ScanList[idx].Wmm ?
					"enabled" : "disabled");
}
#endif 

/** 
 *  @brief parse scan info
 *  @param pHandle	A pointer to WCON_HANDLE
 *  @param buffer       A pointer to scan result buffer
 *  @param length       length of scan result buffer
 *  @return      	NA
 */
static void parse_scan_info(WCON_HANDLE *pHandle, u8 buffer[], s32 length)
{
	s32			len = 0;
	s32			ap_index = -1;
	s8			*mode[3] = {"auto", "ad-hoc", "infra"};
	struct iw_event		iwe;
	struct iw_point		iwp;

	memset(pHandle->ScanList, 0, sizeof(pHandle->ScanList));
	pHandle->ApNum = 0;

	while (len + IW_EV_LCP_LEN < length) {
		memcpy((s8 *)&iwe, buffer + len, sizeof(struct iw_event));
		if ((iwe.cmd == SIOCGIWESSID)||(iwe.cmd ==SIOCGIWENCODE)||
			(iwe.cmd ==IWEVCUSTOM)){
				if(we_version_compiled > 18)	
					memcpy((s8 *)&iwp, buffer + len + IW_EV_LCP_LEN - MRV_EV_POINT_OFF,
						       	sizeof(struct iw_point));
				else
					memcpy((s8 *)&iwp, buffer + len + IW_EV_LCP_LEN,sizeof(struct iw_point));
			iwp.pointer = buffer + len + IW_EV_POINT_LEN;				
		}
		switch (iwe.cmd) {
		case SIOCGIWAP:
			ap_index++;
			memcpy(pHandle->ScanList[ap_index].Bssid, 
					iwe.u.ap_addr.sa_data, ETH_ALEN);
			printf("\nBSSID:\t %02X:%02X:%02X:%02X:%02X:%02X\n",
				HWA_ARG(pHandle->ScanList[ap_index].Bssid));
			break;
			
		case SIOCGIWESSID:
			if ((iwp.pointer) && (iwp.length)) {
				memcpy(pHandle->ScanList[ap_index].Ssid.ssid,
						(s8 *)iwp.pointer,
						iwp.length);
				pHandle->ScanList[ap_index].Ssid.ssid_len = 
							iwp.length;
			}
			printf("SSID:\t %s\n",
				pHandle->ScanList[ap_index].Ssid.ssid);
			break;
		
		case SIOCGIWENCODE:
			if (!(iwp.flags & IW_ENCODE_DISABLED)) {
				pHandle->ScanList[ap_index].Privacy =
							WCON_ENC_ENABLED;
			}
			printf("Privacy: %s\n", 
					pHandle->ScanList[ap_index].Privacy ?
					"enabled": "disabled");
			break;

		case SIOCGIWMODE:
			pHandle->ScanList[ap_index].NetMode = iwe.u.mode;
			printf("NetMode: %s\n", 
				mode[pHandle->ScanList[ap_index].NetMode]);
			break;
	
#if WIRELESS_EXT > 14
		case IWEVCUSTOM:
			parse_custom_info(pHandle, &iwp, ap_index);
			break;
#endif

		case IWEVQUAL:
			pHandle->ScanList[ap_index].Rssi = iwe.u.qual.level;
			printf("Quality: %d\n", 
					pHandle->ScanList[ap_index].Rssi);
			break;
		}
		
		len += iwe.len;
	}
	
	pHandle->ApNum = ap_index + 1;
	printf("\nNo of AP's = %d\n", pHandle->ApNum);

	return;
}

/* 
 *  @brief Process scan results
 *  @param argc		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_scan_results(int argc, char *argv[]) 
{
	u8		buffer[IW_SCAN_MAX_DATA];
	struct iwreq	iwr;
	WCON_HANDLE	mhandle, *pHandle = &mhandle;

	memset(pHandle, 0, sizeof(WCON_HANDLE));
	memset(&iwr, 0, sizeof(struct iwreq));
	
	iwr.u.data.pointer = buffer;
	iwr.u.data.length = sizeof(buffer);
	strncpy(iwr.ifr_name, dev_name, IFNAMSIZ);

	if ((ioctl(sockfd, SIOCGIWSCAN, &iwr)) < 0) {
		printf("Get Scan Results Failed\n");
		return -1;
	}

	parse_scan_info(pHandle, buffer, iwr.u.data.length);

	return WLAN_STATUS_SUCCESS;
}


/** 
 *  @brief Process read eeprom
 *
 *  @param stroffset	A pointer to the offset string
 *  @param strnob	A pointer to NOB string
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_read_eeprom(s8 *stroffset, s8 *strnob)
{
	s8 			buffer[MAX_EEPROM_DATA];
	struct ifreq    	userdata;
	wlan_ioctl_regrdwr 	*reg = (wlan_ioctl_regrdwr *)buffer;

	memset(buffer, 0, sizeof(buffer));
	reg->WhichReg = REG_EEPROM;
	reg->Action = 0;

	if (!strncasecmp(stroffset, "0x", 2))
		reg->Offset = a2hex((stroffset + 2));
	else
		reg->Offset = atoi(stroffset);

	if (!strncasecmp(strnob, "0x", 2))
		reg->NOB = a2hex((strnob + 2));
	else
		reg->NOB = atoi(strnob);

	if (reg->NOB > MAX_EEPROM_DATA) {
		fprintf(stderr, "Number of bytes exceeds MAX EEPROM Read size\n");
		return WLAN_STATUS_FAILURE;
	}

	strncpy(userdata.ifr_name, dev_name, IFNAMSIZ);
	userdata.ifr_data = buffer;

	if (ioctl(sockfd, WLANREGRDWR, &userdata)) {
		perror("wlanconfig");
		fprintf(stderr, 
			"wlanconfig: EEPROM read not possible "
			"by interface %s\n", dev_name);
		return WLAN_STATUS_FAILURE;
	}

	hexdump("RD EEPROM", &reg->Value, reg->NOB, ' '); 

	return WLAN_STATUS_SUCCESS;
}

/* 
 *  @brief Display usage
 *  
 *  @return       NA
 */
static void display_usage(void)
{
	s32 i;

	for (i = 0; i < NELEMENTS(usage); i++)
		fprintf(stderr, "%s\n", usage[i]);
}

/* 
 *  @brief Find command
 *  
 *  @param maxcmds	max command number
 *  @param cmds		A pointer to commands buffer
 *  @param cmd		A pointer to command buffer
 *  @return      	index of command or WLAN_STATUS_FAILURE
 */
static int findcommand(s32 maxcmds, s8 *cmds[], s8 *cmd)
{
	s32 i;

	for (i = 0; i < maxcmds; i++) {
		if (!strcasecmp(cmds[i], cmd)) {
			return i;
		}
	}

	return WLAN_STATUS_FAILURE;
}

/* 
 *  @brief SD comand52 read
 *  @param argc		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_sdcmd52r(int argc, char *argv[])
{
	struct ifreq    userdata;
	u8	 	buf[6];
	u32	 	tmp;

	buf[0] = 0;			//CMD52 read
	if (argc == 5) {
		buf[1] = atoval(argv[3]);	//func
		tmp = 	 atoval(argv[4]);	//reg
		buf[2] = tmp & 0xff;
		buf[3] = (tmp >> 8) & 0xff;
		buf[4] = (tmp >> 16) & 0xff;
		buf[5] = (tmp >> 24) & 0xff;
	} else {
		fprintf(stderr, "Invalid number of parameters!\n");
		return WLAN_STATUS_FAILURE;
	}
	
	strncpy(userdata.ifr_name, dev_name, IFNAMSIZ);
	userdata.ifr_data = buf;

	if (ioctl(sockfd, WLANCMD52RDWR, &userdata)) {
		perror("wlanconfig");
		fprintf(stderr,
			"wlanconfig: CMD52 R/W not supported by "
				"interface %s\n", dev_name);
		return WLAN_STATUS_FAILURE;
   	}
   	printf("sdcmd52r returns 0x%02X\n", buf[0]);

   	return WLAN_STATUS_SUCCESS;
}

/* 
 *  @brief SD comand52 write
 *  @param argc		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_sdcmd52w(int argc, char *argv[])
{
   	struct ifreq    userdata;
   	u8	 	buf[7];
   	u32	 	tmp;

	buf[0] = 1;			//CMD52 write
	if (argc == 6) {
		buf[1] = atoval(argv[3]);		//func
		tmp =    atoval(argv[4]);		//reg
		buf[2] = tmp & 0xff;
		buf[3] = (tmp >> 8) & 0xff;
		buf[4] = (tmp >> 16) & 0xff;
		buf[5] = (tmp >> 24) & 0xff;
		buf[6] = atoval(argv[5]);		//dat
	} else {
		fprintf(stderr, "Invalid number of parameters!\n");
		return WLAN_STATUS_FAILURE;
	}

	strncpy(userdata.ifr_name, dev_name, IFNAMSIZ);
	userdata.ifr_data = buf;

	if (ioctl(sockfd, WLANCMD52RDWR, &userdata)) {
		perror("wlanconfig");
		fprintf(stderr,
			"wlanconfig: CMD52 R/W not supported by "
				"interface %s\n", dev_name);
			return WLAN_STATUS_FAILURE;
   	}
   	printf("sdcmd52w returns 0x%02X\n",buf[0]);

   	return WLAN_STATUS_SUCCESS;
}

/* 
 *  @brief SD comand53 read
 *  
 *  @param argc		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
static int process_sdcmd53r(void)
{
	struct ifreq    userdata;
	s8 		buf[CMD53BUFLEN];
	int 		i;

	strncpy(userdata.ifr_name, dev_name, IFNAMSIZ);
	userdata.ifr_data = buf;

	for(i=0; i < NELEMENTS(buf); i++)
		buf[i] = i & 0xff;

	if (ioctl(sockfd, WLANCMD53RDWR, &userdata)) {
		perror("wlanconfig");
		fprintf(stderr,
			"wlanconfig: CMD53 R/W not supported by "
			"interface %s\n", dev_name);
		return WLAN_STATUS_FAILURE;
	}

	for(i=0; i < NELEMENTS(buf); i++) {
		if (buf[i] != (i ^ 0xff))
			printf("i=%02X  %02X\n",i,buf[i]);
	}

	return WLAN_STATUS_SUCCESS;
}



/* 
 *  @brief Get one line from the File
 *  
 *  @param s	        Storage location for data.
 *  @param size 	Maximum number of characters to read. 
 *  @param stream 	File stream	  	
 *  @param line		A pointer to return current line number
 *  @return             returns string or NULL 
 */
static s8 * wlan_config_get_line(s8 *s, s32 size, FILE *stream, int *line)
{
	s8 *pos, *end, *sstart;

	while (fgets(s, size, stream)) {
		(*line)++;
		s[size - 1] = '\0';
		pos = s;

		while (*pos == ' ' || *pos == '\t')
			pos++;
		if (*pos == '#' || (*pos == '\r' && *(pos+1) == '\n') || 
						*pos == '\n' || *pos == '\0')
			continue;

		/* Remove # comments unless they are within a double quoted
		* string. Remove trailing white space. */
		sstart = strchr(pos, '"');
		if (sstart)
			sstart = strchr(sstart + 1, '"');
		if (!sstart)
			sstart = pos;
		end = strchr(sstart, '#');
		if (end)
			*end-- = '\0';
		else
			end = pos + strlen(pos) - 1;
		while (end > pos && (*end == '\r' || *end == '\n' || 
						*end == ' ' || *end == '\t')) {
			*end-- = '\0';
		}
		if (*pos == '\0')
			continue;
		return pos;
	}

	return NULL;
}



/** 
 *  @brief read register
 *  @param cmd 		the type of register
 *  @param stroffset	A pointer to register index string
 *  @return            	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int process_read_register(s32 cmd, s8 *stroffset)
{
	struct ifreq    userdata;
	wlan_ioctl_regrdwr reg;
	s8           *whichreg;

	switch (cmd) {
		case CMD_RDMAC:
			/*
			 * HostCmd_CMD_MAC_REG_ACCESS 
			 */
			reg.WhichReg = REG_MAC;
			whichreg = "MAC";
			break;
		case CMD_RDBBP:
			/*
			 * HostCmd_CMD_BBP_REG_ACCESS 
			 */
			reg.WhichReg = REG_BBP;
			whichreg = "BBP";
			break;
		case CMD_RDRF:
			/*
			 * HostCmd_CMD_RF_REG_ACCESS 
			 */
			reg.WhichReg = REG_RF;
			whichreg = "RF";
			break;
		default:
			fprintf(stderr, 
				"Invalid Register set specified.\n");
			return -1;
	}

	reg.Action = 0;		/* READ */

	if (!strncasecmp(stroffset, "0x", 2))
		reg.Offset = a2hex((stroffset + 2));
	else
		reg.Offset = atoi(stroffset);

	strncpy(userdata.ifr_name, dev_name, IFNAMSIZ);
	userdata.ifr_data = (s8 *) &reg;

	if (ioctl(sockfd, WLANREGRDWR, &userdata)) {
		perror("wlanconfig");
		fprintf(stderr,
			"wlanconfig: Register Reading not supported by"
			"interface %s\n", dev_name);
		return WLAN_STATUS_FAILURE;
	}

	printf("%s[0x%04lx] = 0x%08lx\n", 
			whichreg, reg.Offset, reg.Value);

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief write register
 *  @param cmd 		the type of register
 *  @param stroffset	A pointer to register index string
 *  @param strvalue	A pointer to the register value
 *  @return            	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int process_write_register(s32 cmd, s8 *stroffset, s8 *strvalue)
{
	struct ifreq    	userdata;
	wlan_ioctl_regrdwr 	reg;
	s8           		*whichreg;

	switch (cmd) {
		case CMD_WRMAC:
			/*
			 * HostCmd_CMD_MAC_REG_ACCESS 
			 */
			reg.WhichReg = REG_MAC;
			whichreg = "MAC";
			break;
		case CMD_WRBBP:
			/*
			 * HostCmd_CMD_BBP_REG_ACCESS 
			 */
			reg.WhichReg = REG_BBP;
			whichreg = "BBP";
			break;
		case CMD_WRRF:
			/*
			 * HostCmd_CMD_RF_REG_ACCESS 
			 */
			reg.WhichReg = REG_RF;
			whichreg = "RF";
			break;
		default:
			fprintf(stderr, 
				"Invalid register set specified.\n");
			return -1;
	}

	reg.Action = 1;		/* WRITE */

	if (!strncasecmp(stroffset, "0x", 2))
		reg.Offset = a2hex((stroffset + 2));
	else
		reg.Offset = atoi(stroffset);

	if (!strncasecmp(strvalue, "0x", 2))
		reg.Value = a2hex((strvalue + 2));
	else
		reg.Value = atoi(strvalue);

	printf("Writing %s Register 0x%04lx with 0x%08lx\n", whichreg,
			reg.Offset, reg.Value);

	strncpy(userdata.ifr_name, dev_name, IFNAMSIZ);
	userdata.ifr_data = (s8 *) &reg;

	if (ioctl(sockfd, WLANREGRDWR, &userdata)) {
		perror("wlanconfig");
		fprintf(stderr, 
			"wlanconfig: Register Writing not supported "
			"by interface %s\n", dev_name);
		return WLAN_STATUS_FAILURE;
	}

	printf("%s[0x%04lx] = 0x%08lx\n",
			whichreg, reg.Offset, reg.Value);

	return WLAN_STATUS_SUCCESS;
}

/** 
 *  @brief read CF register
 *
 *  @param stroffset	A pointer to register index string
 *  @return            	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int process_read_cfreg(s8 *stroffset)
{
	struct ifreq    	userdata;
	wlan_ioctl_cfregrdwr 	reg;
	
	reg.Action = 0; //Read register

	if (!strncasecmp(stroffset, "0x", 2))
		reg.Offset = a2hex((stroffset + 2));
	else
		reg.Offset = atoi(stroffset);

	strncpy(userdata.ifr_name, dev_name, IFNAMSIZ);
	userdata.ifr_data = (s8 *) &reg;

	if (ioctl(sockfd, WLANREGCFRDWR, &userdata)) {
		perror("wlanconfig");
		fprintf(stderr, 
			"wlanconfig: Register reading not supported "
			"by interface %s\n", dev_name);
		return  WLAN_STATUS_FAILURE;
	}

	printf("CFREG[0x%04X] = 0x%04X\n",
				reg.Offset, reg.Value);

	return  WLAN_STATUS_SUCCESS;

}

/** 
 *  @brief write CF register
 *
 *  @param stroffset	A pointer to register index string
 *  @param strvalue	A pointer to the register value 
 *  @return            	WLAN_STATUS_SUCCESS or WLAN_STATUS_FAILURE
 */
static int process_write_cfreg(s8 *stroffset, s8 *strvalue)
{
	struct ifreq    	userdata;
	wlan_ioctl_cfregrdwr 	reg;

	reg.Action = 1; //Write register

	if (!strncasecmp(stroffset, "0x", 2))
		reg.Offset = a2hex((stroffset + 2));
	else
		reg.Offset = atoi(stroffset);

	if (!strncasecmp(strvalue, "0x", 2))
		reg.Value = a2hex((strvalue + 2));
	else
		reg.Value = atoi(strvalue);

	strncpy(userdata.ifr_name, dev_name, IFNAMSIZ);
	userdata.ifr_data = (s8 *) &reg;

	if (ioctl(sockfd, WLANREGCFRDWR, &userdata)) {
		perror("wlanconfig");
		fprintf(stderr, 
			"wlanconfig: Register writing not supported "
			"by interface %s\n", dev_name);
		return WLAN_STATUS_FAILURE;
	}

	return WLAN_STATUS_SUCCESS;
}

/********************************************************
		Global Functions
********************************************************/
/* 
 *  @brief Entry function for wlanconfig
 *  @param argc		number of arguments
 *  @param argv         A pointer to arguments array    
 *  @return      	WLAN_STATUS_SUCCESS--success, otherwise--fail
 */
int main(int argc, char *argv[])
{
	s32             cmd;

	if (argc < 3) {
		fprintf(stderr, "Invalid number of parameters!\n");
		display_usage();
		exit(1);
	}

	strncpy(dev_name, argv[1], IFNAMSIZ);

	/*
	 * create a socket 
	 */
	if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		fprintf(stderr, "wlanconfig: Cannot open socket.\n");
		exit(1);
	}
	if(get_range() < 0){
		fprintf(stderr, "wlanconfig: Cannot get range.\n");
		exit(1);
	}
	switch ((cmd = findcommand(NELEMENTS(commands), commands, argv[2]))) {
	case CMD_HOSTCMD:
		process_host_cmd(argc, argv);
		break;
	case CMD_RDMAC:
	case CMD_RDBBP:
	case CMD_RDRF:
		if (argc < 4) {
			fprintf(stderr, "Register offset required!\n");
			display_usage();
			exit(1);
		}

		if (process_read_register(cmd, argv[3])) {
			fprintf(stderr, "Read command failed!\n");
			exit(1);
		}
		break;
	case CMD_WRMAC:
	case CMD_WRBBP:
	case CMD_WRRF:
		if (argc < 5) {
			fprintf(stderr, "Register offset required & value!\n");
			display_usage();
			exit(1);
		}
		if (process_write_register(cmd, argv[3],
					argv[4])) {
			fprintf(stderr, "Write command failed!\n");
			exit(1);
		}
		break;
	case CMD_CMD52R:
		process_sdcmd52r(argc,argv);
		break;
	case CMD_CMD52W:
		process_sdcmd52w(argc,argv);
		break;
	case CMD_CMD53R:
		process_sdcmd53r();
		break;
	case CMD_CFREGR:
		printf("process read cfreg\n");
		if (argc < 4) {
			fprintf(stderr,	"Register offset required!\n");
			display_usage();
			exit(1);
		}
		if (process_read_cfreg(argv[3])) {
			fprintf(stderr, "Read CF register failed\n");
			display_usage();
			exit(1);
		}
		break;
	case CMD_CFREGW:
		printf("process write cfreg\n");
		if (argc < 5) {
			fprintf(stderr,	"Register offset required!\n");
			display_usage();
			exit(1);
		}
		if (process_write_cfreg(argv[3], argv[4])) {
			fprintf(stderr, "Read CF register failed\n");
			display_usage();
			exit(1);
		}
		break;
	case CMD_RDEEPROM:
		printf("proces read eeprom\n");

		if(argc < 5) {
			fprintf(stderr, "Register offset, number of bytes required\n");
			display_usage();
			exit(1);
		}
		
		if(process_read_eeprom(argv[3], argv[4])) {
			fprintf(stderr, "EEPROM Read failed\n");
			display_usage();
			exit(1);
		}
		break;
	case CMD_GETRATE:
		if (process_get_rate()) {
			fprintf(stderr, "Get Rate Failed\n");
			display_usage();
			exit(1);
		}
		break;
	case CMD_SLEEPPARAMS:
		if (process_sleep_params(argc, argv)) {
			fprintf(stderr, "Sleep Params Failed\n");
			display_usage();
			exit(1);
		}
		break;
	case CMD_BCA_TS:
		if (process_bca_ts(argc, argv)) {
			fprintf(stderr, "SetBcaTs Failed\n");
			display_usage();
			exit(1);
		}
		break;
	case CMD_EXTSCAN:
		if (process_extscan(argc, argv)) {
			fprintf(stderr, "ExtScan Failed\n");
			display_usage();
			exit(1);
		}
		break;
	case CMD_SCAN_LIST:
		if (process_scan_results(argc, argv)) {
			fprintf(stderr, "getscanlist Failed\n");
			display_usage();
			exit(1);
		}
		break;
	case CMD_GET_SCAN_RSP:
        if (process_getscantable(argc, argv)) {
			exit(1);
        }
        break;

    case CMD_SET_USER_SCAN:
        if (process_setuserscan(argc, argv)) {
            exit(1);
        }
        break;

	default:
		fprintf(stderr, "Invalid command specified!\n");
		display_usage();
		exit(1);
	}

	return WLAN_STATUS_SUCCESS;
}
