/* fdesc.c  - user-level file descriptor information for the system 
 * call wrappers */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <errno.h>

#include "scout_semaphore.h"

#define IS_OPEN (1 << 31)

struct fdesc {
    int f;              /* fcntl/open flags - we set the high order bit 
			 * if the file has been opened */
    struct Semaphore s; /* semaphore to wait on */
};

struct Semaphore selectSema;

static int maxopen = -1;
static struct fdesc files[FD_SETSIZE];


void fdescInit() 
{
    int i;
    for (i = 0; i < FD_SETSIZE; i++) {
	semaphoreInit(&files[i].s, 0);
	files[i].f = 0;
    }
    semaphoreInit(&selectSema, 0);
}

int fdescGetFlags(unsigned int fd) 
{
    if (fd > FD_SETSIZE) {
	return -EBADF;
    } else if ((files[fd].f & IS_OPEN) == 0) {
	return -EBADF;
    } else {
	return files[fd].f & ~IS_OPEN;
    }
}


/* The IS_OPEN flag is set through a seperate call to fdescOpen */
int fdescSetFlags(unsigned int fd, unsigned int flags) 
{
    if (fd > FD_SETSIZE) {
	return -EBADF;
    } else if ((files[fd].f & IS_OPEN) == 0) {
	return -EBADF;
    } else {
	return files[fd].f |= (flags & ~IS_OPEN);
    }
}

/* The IS_OPEN flag is set through a seperate call to fdescClose */
int fdescClearFlags(unsigned int fd, unsigned int flags) 
{
    if (fd > FD_SETSIZE) {
	return -EBADF;
    } else if ((files[fd].f & IS_OPEN) == 0) {
	return -EBADF;
    } else {
	return files[fd].f &= (~flags | IS_OPEN);
    }
}

/* Mark fd as an open file, with some initial flags */
int fdescOpen(unsigned int fd, unsigned int flags) 
{
    if (fd > FD_SETSIZE) {
	return -EBADF;
    } else if ((files[fd].f & IS_OPEN) != 0) {
	return -EBADF;
    } else {
	if ((int)fd > maxopen)
	    maxopen = fd;
	return files[fd].f |= (flags | IS_OPEN);
    }
}

/* Mark fd as an open file, with some initial flags */
int fdescClose(unsigned int fd) 
{
    if (fd > FD_SETSIZE) {
	return -EBADF;
    } else if ((files[fd].f & IS_OPEN) == 0) {
	return -EBADF;
    } else {
	return files[fd].f = 0;
    }
}

int fdescIsOpen(unsigned int fd) {
    if (fd > FD_SETSIZE)
	return 0;
    else if (files[fd].f & IS_OPEN)
	return 1;
    else
	return 0;
}

int fdescMaxOpen() 
{
    int i;
    for (i = maxopen; i >= 0; i--) {
	if (fdescIsOpen(i))
	    return i;
    }
    return -1;
}

/* Signal or wait for a change in a file descriptor */
void fdescSignal(unsigned int fd)
{
    if (fd > FD_SETSIZE) {
	return;
    }
    semaphoreSignal(&files[fd].s);
}

void fdescWait(unsigned int fd)
{
    if (fd > FD_SETSIZE) {
	return;
    }
    semaphoreWait(&files[fd].s);
}

void fdescAnyWait() 
{
    semaphoreWait(&selectSema);
}

void fdescAnySignal()
{
    semaphoreSignal(&selectSema);
}

/* Routines tro iterate through all open file descriptors */
int fdescGetFirstOpen()
{
    int i;
    for (i = 0; i < FD_SETSIZE; i++)
	if ((files[i].f & IS_OPEN) != 0)
	    return i;
    return -1;
}

int fdescGetNextOpen(int curr) 
{
    int i;
    for (i = curr + 1; i < FD_SETSIZE; i++)
	if ((files[i].f & IS_OPEN) != 0)
	    return i;
    return -1;
}
