//  ClassFile.java -- locate a Java .class file
//
//   Note:  these methods just open files; they don't check class names.

package toba.classfile;

import java.io.*;
import java.util.*;
import sun.tools.zip.*;



public class ClassFile extends DataInputStream {



static final int classBufferSize = 8192;	// class file buffer size

public static boolean trace = false;		// tracing flag



// instance variables

    public File file;			// file handle (for name, age, etc.)
    public String path;			// path found on 

    public ZipFile zipfile;		// opened Zip File



// new ClassFile(InputStream input) -- open a class file given the input stream

private ClassFile(InputStream input)
{
    super(new BufferedInputStream(input, classBufferSize));
}


//  new ClassFile(f) -- open a class file given a file handle.

public ClassFile(File f)
    throws FileNotFoundException
{
    this(new FileInputStream(f));
    file = f;
    path = null;
}


//  new ClassFile(filename) -- open a class file given the exact file name.

public ClassFile(String filename)
    throws FileNotFoundException
{
    this(new File(filename));
}



// find(path, classname) -- find a class in a search path

public static ClassFile find(String path, String classname)
    throws ClassNotFoundException
{
    if (path == null)
        return null;

    String classpath = path + File.pathSeparator;
    String filepath = classname.replace('.', File.separatorChar) + ".class";

    int i = 0;
    int j;
    while ((j = classpath.indexOf(File.pathSeparator, i)) >= 0) {
	String dname = classpath.substring(i, j); // next field of path
	if (dname.length() == 0)
	    dname = null;			// if empty, use current dir

        ClassFile cf = open(dname, filepath);
        if (cf != null) {
            cf.path = path;
            return cf;
        }
	i = j + 1;
    }

    if (trace)
        System.out.println("    CLASS NOT FOUND: " + classname);
    throw new ClassNotFoundException(classname);	// no luck
}


//  find(classname) -- find a class given its name, checking usual places.

public static ClassFile find(String classname)
    throws ClassNotFoundException
{
    ClassFile f;

    // check for name ending in .class, which indicates a file name
    int len = classname.length();
    if (len > 6 && classname.substring(len - 6).equals(".class")) {
	try {
	    if (trace)
		System.out.println("  filename load: " + classname);
	    return new ClassFile(classname);
	} catch (FileNotFoundException e) {
	    throw new ClassNotFoundException(classname);
	}
    }

    if (trace)
    	System.out.println("  searching for class " + classname);

    // check for illegal class name that might map into a legal file name
    if (classname.indexOf(File.separator) >= 0)
	throw new ClassNotFoundException(classname);

    // first try TOBAPATH and CLASSPATH
    if ((f = walkprop("toba.class.path", classname)) != null)
    	return f;
    if ((f = walkprop("java.class.path", classname)) != null)
    	return f;

    // finally strip off package name and try current directory
    int i = classname.lastIndexOf(".");
    String filename = classname.substring(i + 1) + ".class";
    if (trace)
    	System.out.println("    not on path; checking " + filename);
    try {
        f = new ClassFile(filename);
	if (trace)
	    System.out.println("    found " + filename);
        return f;
    } catch (FileNotFoundException e) {
        if (trace)
	    System.out.println("    CLASS NOT FOUND: " + classname);
	throw new ClassNotFoundException(classname);	// no luck
    }

}


// close() -- close the class file

public void close() 
    throws IOException
{
    super.close();
    if (zipfile != null) {
        zipfile.close();
        zipfile = null;
    }
}

//------------------------------- private ------------------------------------



//  walkprop(propname, classname) -- find .class file using path from property.

private static ClassFile walkprop(String propname, String classname)
{
    String path = System.getProperty(propname);

    if (trace)
    	System.out.println("    " + propname + ": " + 
	    ((path == null) ? "[null]" : ('"' + path + '"')));

    try {
        return find(path, classname);
    } catch(ClassNotFoundException e) {
        return null;
    }
}


// open(dir, file) -- open a class file, possibly in a .zip file
//
// If dir is null, the current directory is used.

private static ClassFile open(String dir, String file)
{
    if (trace)
         System.out.println("      checking for " + file + " in " + dir);

    // If dir exists and is a plain file, treat it as a zipfile
    if (dir != null && (new File(dir)).isFile() ) {
        ZipFile zf = null;

        // zipfiles seem to use '/' always
        file = file.replace(File.separatorChar, '/');
        try {
            File f = new File(dir);
            zf = new ZipFile(f);
            ZipEntry entry = zf.getEntry(file);
            if (entry != null) {
                if (trace)
	            System.out.println("      found " + 
                                       dir + File.separator + file);
                ClassFile ret = new ClassFile(zf.getInputStream(entry));
                ret.zipfile = zf;
                ret.file = f;
                return ret;
            }
            zf.close();
        } catch(IOException e1) {
            try {
                if (zf != null)
                    zf.close();
            } catch(IOException e2) {
            }
        }
    } else {
        File f = new File(dir, file);
        if (f.exists()) {
            if (trace)
	        System.out.println("      found " + f.getPath());
            try {
                return new ClassFile(f);
            } catch(FileNotFoundException e) {
            }
        }
    }
    return null;
}



} // class ClassFile
