/**
 ** util.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/


#include CONFIG

#include "util.h"
#include "stringlb.h"

void *util_unused;

#ifdef NEEDS_BZERO
NOPROTO void bzero(char *dest, int len)
{
    memset(dest, 0, len);
}
#endif

#ifdef NEEDS_STRICMP
#ifdef HAS_STRCASECMP
NOPROTO int stricmp(char *a, char *b)
{
    return strcasecmp(a, b);
}
#else
NOPROTO int stricmp(char *a, char *b)
{
	while (1) {
		Int c1= *a;
		Int c2= *b;
		if ('A' <= c1 && c1 <= 'Z') c1 += 32;
		if ('A' <= c2 && c2 <= 'Z') c2 += 32;
		if (c1 < c2) return -1;
		if (c1 > c2) return 1;
		if (!c1) return 0;
		a++; b++;
	}
}
#endif
#endif

#ifdef NEEDS_STRNICMP
#ifdef HAS_STRNCASECMP
NOPROTO int strnicmp(char *a, char *b, int n)
{
    return strncasecmp(a, b, n);
}
#else
NOPROTO int strnicmp(char *a, char *b, int n)
{
	while (n>0) {
		Int c1= *a;
		Int c2= *b;
		if ('A' <= c1 && c1 <= 'Z') c1 += 32;
		if ('A' <= c2 && c2 <= 'Z') c2 += 32;
		if (c1 < c2) return -1;
		if (c1 > c2) return 1;
		if (!c1) return 0;
		a++; b++; n--;
	}
	return 0;
}
#endif
#endif

#ifdef NEEDS_MEMMOVE
NOPROTO void memmove(void *dest, void *src, Int len)
{
    char *cdest= dest, *csrc= src;

    if (cdest < csrc) {
	/* Copy front to back */
	while (len--) *cdest++= *csrc++;
    }
    else {
	/* Copy back to front */
	cdest += len;
	csrc += len;
	while (len--) *--cdest= *--csrc;
    }
}
#endif

#ifdef MALLOC_DEBUG
void *debug_xmalloc(size_t size,const char *file,Int line)
{
  void *ret= debug_malloc(size,file,line);
  if (!ret) {
    fprintf(stderr,"Out of memory (tried to allocate size %ld)\n", (Int)size);
    exit(1);
  }
  return ret;
}
#else
void *xmalloc(size_t size)
{
  void *ret= malloc(size);
  if (!ret) {
    fprintf(stderr,"Out of memory (tried to allocate size %ld)\n", (Int)size);
    exit(1);
  }
  return ret;
}
#endif
void *xmalloc_clear(size_t size)
{
  void *ret= xmalloc(size);
  if (ret) memset(ret, 0, size);
  return ret;
}

void *xrealloc(void *old, size_t size)
{
  void *ret= realloc(old, size);
  if (!ret) {
    fprintf(stderr,"Out of memory (tried to reallocate size %ld)\n", (Int)size);
    exit(1);
  }
  return ret;
}

void xfree(void *pointer)
{
  free(pointer);
}

void out_of_memory(void)
{
    die(("Out of memory\n"));
}

void debugf(char *format, ...)
{
    va_list l;
    va_start(l, format);
    vfprintf(stderr, format, l);
    va_end(l);
}


void errorf(char *format, ...)
{
    va_list l;
    va_start(l, format);
    vfprintf(stderr, format, l);
    va_end(l);
    die(("fatal error\n"));
}


Int intlog2(unsigned long num)
{
    Int ret= -1;
    while (num > 128) {
	num >>= 8;
	ret += 8;
    }
    while (num > 8) {
	num >>= 4;
	ret += 4;
    }
    while (num > 0) {
	num >>= 1;
	ret ++;
    }
    return ret;
}

#ifdef HACK_VSPRINTF
/*NOPROTO*/ Int vsprintf(char *dest, char *src, void * _NOPROTO)
{
    long *args= (long*) _NOPROTO;
    return sprintf(dest, src, args[0], args[1], args[2], args[3], args[4], args[5],
  		              args[6], args[7], args[8], args[9]);
}
#endif

#ifdef MALLOC_DEBUG

#include "table.h"
#undef malloc
#undef free
#undef realloc

Malloc_ptrs malloc_tab[MAX_MALLOCS];
Int mtablen;

Int dmalloc_debug=0;
Int debug_malloc_lognum=0;
Table dmalloc_table;
static int in_dmalloc=1;

int dmalloc_init()
{
   in_dmalloc=1;
   table_hash_init(&dmalloc_table,sizeof(void *),sizeof(Malloc_ptrs),
		   table_ptrcmp,          table_ptr_hash1,
		   table_ptr_is_empty,    table_ptr_make_empty,
		   table_ptr_is_deleted,  table_ptr_make_deleted);
   in_dmalloc=0;	
   return(1);
}

int dmalloc_compare(Malloc_ptrs *m1,Malloc_ptrs *m2)
{
   int cmp;

   if((m1==NULL || m1->mptr==NULL) &&
      (m2==NULL || m2->mptr==NULL))
     return(0);
   if(m1==NULL || m1->mptr==NULL)     return(1);
   if(m2==NULL || m2->mptr==NULL)     return(-1);

   cmp=m1->logtime-m2->logtime;
   if(cmp!=0)   return(cmp);		/* not the same log time */

   /* is the same module */
   cmp=strcmp(m1->file,m2->file);
   if(cmp!=0)	return(cmp);		/* not the same module */
     
   /* is the same module */
   cmp=m1->line-m2->line;
   if(cmp!=0)    return(cmp);		/* not the same line */

   /* is the same line number */
   cmp=m1->c0 - m2->c0;
   if(cmp!=0)    return(cmp);		/* not the same caller */

   /* is the same caller */
   cmp=m1->c1 - m2->c1;
   if(cmp!=0)    return(cmp);		/* not the same caller */

   /* is the same caller */
   cmp=m1->c2 - m2->c2;
   if(cmp!=0)    return(cmp);		/* not the same caller */

   /* is the same caller */
   return(m1->time - m2->time);
}

   
void debug_malloc_print(FILE *f)
{
   Int i,index;
   Int *logsum;
   Int sum;
  char *lname;
  Int  lline,llog;
  Int  bsum,osum;
  Malloc_ptrs *m;
  TableIter ti;
  void *key;

  logsum= (Int*) malloc(sizeof(Int)*(debug_malloc_lognum+1));

  /* read table data into array for sorting and processing */
  table_rewind(&dmalloc_table,&ti);
  mtablen=0;
  
  while(table_get_next(&dmalloc_table,&ti,&key,&m))
    memcpy(&malloc_tab[mtablen++],m,sizeof(Malloc_ptrs));
  
  for(i=0;i<debug_malloc_lognum+1;i++)
    logsum[i]=0;

  qsort(malloc_tab,mtablen,sizeof(Malloc_ptrs),dmalloc_compare);

  lname=malloc_tab[0].file;
  lline=malloc_tab[0].line;
  llog=malloc_tab[0].logtime;
  bsum=0;
  osum=0;
  
  for(index=0,m=&malloc_tab[0];index<mtablen;index++,m++) {
     if(malloc_tab[index].mptr==NULL) /* found end of table */
       break;

     if(llog==m->logtime && m->file==lname && m->line==lline) {
	bsum+=m->size;
	osum++;
     }
     else {
	fprintf(f,"%s:%ld\t%ld: Total %ld bytes, %ld objects\n\n",
		lname,(long)lline,(long)llog,(long)bsum,(long)osum);
	bsum=m->size;;
	osum=1;
	lname=m->file;
	lline=m->line;
	llog=m->logtime;
     }

     fprintf(f,"%s:%ld\t%ld: %ld at %lx \t%s\t%s\t%s\n",
	     malloc_tab[index].file,
	     (long)malloc_tab[index].line,
	     (long)malloc_tab[index].logtime,
	     (long)malloc_tab[index].size,(long)malloc_tab[index].mptr,
	     func_name_from_pc(malloc_tab[index].c0),
	     func_name_from_pc(malloc_tab[index].c1),
	     func_name_from_pc(malloc_tab[index].c2));
     logsum[malloc_tab[index].logtime]+=malloc_tab[index].size;
  }
  fprintf(f,"%ld: Total %ld bytes, %ld objects in %s:%ld\n",
	  (long)llog,(long)bsum,(long)osum,lname,(long)lline);
  
  sum=0;
  for(i=0;i<debug_malloc_lognum+1;i++)
    {
       fprintf(f,"%ld Total: %ld\n",(long)i,(long)logsum[i]);
       sum+=logsum[i];
    }
   debug_malloc_lognum++;
   fprintf(f,"%ld Initial allocation %ld\n",
	   (long)debug_malloc_lognum,(long)sum);
   fflush(f);
   free(logsum);
}

void *debug_malloc(size_t size,const char *file,Int line)
{
   void *ret;
   Malloc_ptrs mps;
   
   ret=malloc(size);
   if(in_dmalloc) {
      return(ret);
   }
   in_dmalloc=1;
   __debug_malloc_create(&mps,ret,size,file,line);
   if(dmalloc_debug) fprintf(stderr,"%ld Malloc: %ld bytes at %lx from %s:%ld in %s:%ld\n",
			     (long) mps.time,
			     (long) mps.size, (long) mps.mptr,
			     mps.file,
			     (long) mps.line,file,(long) line);
   
   table_hash_set(&dmalloc_table,&ret,&mps);
   in_dmalloc=0;
   return(ret);
}

void debug_free(void *ptr,const char *file,Int line)
{
  Malloc_ptrs *mps;
  
  if(ptr==NULL)
    {
      fprintf(stderr, "Error: Free called on NULL in %s:%ld\n",
	      file,(long)line);
      return;
    }

  if(in_dmalloc) {
     free(ptr);
     return;
  }
  in_dmalloc=1;
  if((mps=table_get(&dmalloc_table,&ptr))==NULL) {
     fprintf(stderr,"Error: Freeing illegal address %lx in %s:%ld\n",
	     (long)ptr,file,(long)line);
     ASSERT(0);
  }
  else {
     if(dmalloc_debug) fprintf(stderr,"%ld Free: %ld bytes at %lx from %s:%ld in %s:%ld\n",
			       (long) mps->time,
			       (long) mps->size, (long) mps->mptr,
			       mps->file,
			       (long) mps->line,file, (long) line);
     if (mps->break_on_free) {
	fprintf(stderr, "Break on free\n");
	ASSERT(0);
     }
     memset(ptr,0xff,mps->size);
     ASSERT(!table_hash_remove(&dmalloc_table,&ptr));
     free(ptr);
  }
  in_dmalloc=0;
}

void dmalloc_break_on_free(void *ptr)
{
   Malloc_ptrs *mps;
   
   mps=table_get(&dmalloc_table,&ptr);
   ASSERT(mps);
   mps->break_on_free= 1;
}
       
void *debug_realloc(void *ptr, size_t size, const char *file, Int line)
{
   void *ret=NULL;
   Malloc_ptrs *mps,ms;

   if(in_dmalloc) {
      return(realloc(ptr,size));
   }

   if(ptr==NULL) {
      fprintf(stderr,"Error: Realloc called on NULL in %s:%ld\n",
	      file, (long) line);
      return(NULL);
   }
   
   in_dmalloc=1;
   
   if((mps=table_get(&dmalloc_table,&ptr))==NULL) {
      fprintf(stderr,"Error: Reallocing illegal address %lx in %s:%ld\n",
	      (long) ptr, file, (long) line);
   }
   else {
      ASSERT(!table_hash_remove(&dmalloc_table,&ptr));
      ret=realloc(ptr,size);
      if(dmalloc_debug)
	fprintf(stderr,"%ld Realloc: %ld->%ld bytes %lx->%lx in %s:%ld\n",
		(long) mps->time,
		(long) mps->size, (long) size,
		(long) mps->mptr, (long) ret,
		file, (long) line);
      memcpy(&ms,mps,sizeof(Malloc_ptrs));
      
      ms.mptr=ret;
      ms.size=size;
      table_hash_set(&dmalloc_table,&ret,&ms);
   }
   in_dmalloc=0;
   return(ret);
}
  
void __debug_malloc_create(Malloc_ptrs *ms,void *ptr,size_t size,
			   const char *file,Int line)
{
  static Int insert_time=0;
  
  ms->mptr=ptr;
  ms->size=size;
  ms->file=file;
  ms->line=line;
  ms->time=insert_time++;
  ms->logtime=debug_malloc_lognum;
  ms->c0=find_caller_pc(2);
  ms->c1=find_caller_pc(3);
  ms->c2=find_caller_pc(4);
  ms->break_on_free= 0;
}
#endif


#define FUNC_NAMELEN 40
#define MAX_FUNCS 2000
int func_pcs[MAX_FUNCS];
char func_names[MAX_FUNCS][FUNC_NAMELEN];
int nfuncs;

char executable_name[100];

void load_syms()
{
   int i;
   static int loaded= 0;
   FILE *in;
   char buf[200];
   char buf2[200];
   char buf3[200];

   if (!loaded) {
      sprintf(buf, "nm -n %s | grep ' T '", executable_name);
      in= popen(buf, "r");
      for (i= 0; 1; i++) {
	 if (!fgets(buf, 200, in)) break;
	 sscanf(buf, "%x %s %s", &func_pcs[i], buf3, buf2);
	 strncpy(func_names[i], buf2 + (buf2[0] == '_'), FUNC_NAMELEN);
	 func_names[i][FUNC_NAMELEN-1]= 0;
      }
      nfuncs= i;
      pclose(in);
      loaded= 1;
   }
}

long find_caller_pc(long x)
{
   long *fploc= (&x)-2;
   long retpc= 0;
   x++;
   while (x--) {
      /*printf("The return PC might be %lx\n", fploc[1]);*/
      retpc= fploc[1];
      /*printf("and I think the old FP might be %lx\n", fploc[0]);*/
      fploc= (long*)fploc[0];
   }
   return retpc;
}
#if 0
long find_caller_pc(long x) { return 0; }
#endif
char *func_name_from_pc(long pc)
{
   int i;
   char *ret= "unknown";
   load_syms();
   for (i= 0; i< nfuncs; i++) {
      if (func_pcs[i] < pc) ret= func_names[i];
   }
   return ret;
}


char *caller_name(long x)
{
   long pc= find_caller_pc(x+1);
   return func_name_from_pc(pc);
}
