#include <linux/delay.h>
#include <linux/gpio_keys.h>
#include <linux/init.h>
#include <linux/input.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>

#include <asm/hardware.h>
#include <asm/irq.h>
#include <asm/mach-types.h>
#include <asm/setup.h>

#include <asm/mach/irq.h>
#include <asm/mach/arch.h>
#include <asm/arch/bitfield.h>
#include <asm/arch/irda.h>
#include <asm/arch/pxafb.h>
#include <asm/arch/pxa27x_keyboard.h>
#include <asm/arch/pxa-regs.h>
#include <asm/arch/serial.h>
#include <asm/arch/htcuniversal-gpio.h>

#include "../generic.h"

static struct pxafb_mode_info o2atom_lcd_mode = {
        .pixclock               = 210000,
        .xres                   = 240,
        .yres                   = 320,
        .bpp                    = 16,
        .hsync_len              = 4,
        .vsync_len              = 6,
        .left_margin            = 3,
        .right_margin           = 2,
        .upper_margin           = 2,
        .lower_margin           = 2,
};

static struct pxafb_mach_info o2atom_lcd = {
        .modes                  = &o2atom_lcd_mode,
        .num_modes              = 1,
        .lccr0                  = LCCR0_LDDALT | LCCR0_Act,
};

#define O2ATOM_CF_BASE (0x14000000)
static struct resource o2atom_cf_resources[] = {
        [0] = {
                .start  = O2ATOM_CF_BASE,
                .end    = O2ATOM_CF_BASE + (0x020000000 - 1),
                .flags  = IORESOURCE_MEM | IORESOURCE_MEM_8AND16BIT,
        },
};

static struct platform_device o2atom_cf_device = {
        .name           = "o2atom_vlio_cf",
        .id             = -1,
        .resource       = o2atom_cf_resources,
        .num_resources  = ARRAY_SIZE(o2atom_cf_resources),
};

static struct platform_device o2atom_ts_device = {
        .name           = "pxa2xx-ac97",
        .id             = -1,
};

static void o2atom_irda_transceiver_mode(struct device *dev, int mode) {
}

static struct pxaficp_platform_data o2atom_ficp_platform_data = {
       .transceiver_cap  = IR_SIRMODE | IR_FIRMODE,
       .transceiver_mode = o2atom_irda_transceiver_mode,
};

/* Bluetooth and GSM support:                                    *
 * To enable Bluetooth, raise 86, wait for 96 to fall,           *
 * and raise 114. To disable, lower 114 and 86.                  *
 * To enable GSM, raise 82 until 37 goes high, then lower 82.    *
 * Comm is valid so long as 37 remains high (you can re-trigger  *
 * 82 to keep it going); transmissions last normally.            */

static void o2atom_bt_configure(int state)
{
        switch (state) {
                case PXA_UART_CFG_PRE_STARTUP:
                        break;
                case PXA_UART_CFG_POST_STARTUP:
                        pxa_gpio_mode(86 | GPIO_OUT | GPIO_DFLT_HIGH);
			while (GPLR(96) & GPIO_bit(96))
				msleep(10);
                        pxa_gpio_mode(114 | GPIO_OUT | GPIO_DFLT_HIGH);
                        break;
                case PXA_UART_CFG_PRE_SHUTDOWN:
			GPCR(114) = GPIO_bit(114);
			GPCR(86) = GPIO_bit(86);
                        break;
                case PXA_UART_CFG_POST_SHUTDOWN:
                        break;
                default:
                        break;
        }
}

static void o2atom_phone_configure(int state)
{
	static struct task_struct *watchdog_thread;
        switch (state) {
                case PXA_UART_CFG_PRE_STARTUP:
                        break;
                case PXA_UART_CFG_POST_STARTUP:
                        pxa_gpio_mode(82 | GPIO_OUT | GPIO_DFLT_HIGH); // RESET
			while (!(GPLR(37) & GPIO_bit(37)))
				msleep(10);
                        GPCR(82) = GPIO_bit(82);
			/* TODO */
                        break;
                case PXA_UART_CFG_PRE_SHUTDOWN:
			/* TODO */
                        break;
                case PXA_UART_CFG_POST_SHUTDOWN:
                        break;
                default:
                        break;
        }
}

static struct platform_pxa_serial_funcs o2atom_pxa_bt_funcs = {
        .configure = o2atom_bt_configure,
};

static struct platform_pxa_serial_funcs o2atom_pxa_ff_funcs = {
        .configure = o2atom_phone_configure,
};

/* GPIO button / PXA matrix keypad support:    *
 * Power, camera, and record are standard GPIO *
 * input pins. The rest are matrix-encoded.    */


static struct pxa27x_keyboard_platform_data o2atom_kbd = {
        .nr_rows = 3,
        .nr_cols = 5,
        .keycodes = {
                {
                        KEY_UP,
			KEY_DOWN,
			KEY_LEFT,
			KEY_RIGHT,
			KEY_ENTER,
                }, {    /* row 1 */
                        KEY_SEND,
			KEY_END,
			KEY_LEFTMETA, /* Win */
			KEY_PROG1, /* Media */
			/* NC */
                }, {    /* row 2 */
                        KEY_VOLUMEUP,
			KEY_VOLUMEDOWN,
			KEY_F1, /* Left soft key */
			KEY_F2, /* Right soft key */
			/* NC */
                },
        },
        .gpio_modes = {
                 GPIO_NR_HTCUNIVERSAL_KP_MKIN0_MD,
                 GPIO_NR_HTCUNIVERSAL_KP_MKIN1_MD,
                 GPIO_NR_HTCUNIVERSAL_KP_MKIN2_MD,
                 GPIO_NR_HTCUNIVERSAL_KP_MKOUT0_MD,
                 GPIO_NR_HTCUNIVERSAL_KP_MKOUT1_MD,
                 GPIO_NR_HTCUNIVERSAL_KP_MKOUT2_MD,
                 GPIO_NR_HTCUNIVERSAL_KP_MKOUT3_MD,
                 GPIO_NR_HTCUNIVERSAL_KP_MKOUT4_MD,
         },
};

static struct platform_device o2atom_pxa_keyboard = {
        .name   = "pxa27x-keyboard",
        .id     = -1,
        .dev    =  {
                .platform_data  = &o2atom_kbd,
        },
};

#define GPIO_O2ATOM_POWER 0
#define GPIO_O2ATOM_CAMERA 93
#define GPIO_O2ATOM_RECORD 94

static struct gpio_keys_button o2atom_button_table[] = {
        { KEY_POWER, GPIO_O2ATOM_POWER, 0 },
	{ KEY_CAMERA, GPIO_O2ATOM_CAMERA, 0 },
	{ KEY_RECORD, GPIO_O2ATOM_RECORD, 0 },
};

static struct gpio_keys_platform_data o2atom_pxa_keys_data = {
        .buttons = o2atom_button_table,
        .nbuttons = ARRAY_SIZE(o2atom_button_table),
};

static struct platform_device o2atom_pxa_keys = {
        .name = "gpio-keys",
        .dev = {
                .platform_data = &o2atom_pxa_keys_data,
        },
        .id = -1,
};

static void __init o2atom_init(void)
{
        set_pxa_fb_info(&o2atom_lcd);
        platform_device_register(&o2atom_cf_device);
        platform_device_register(&o2atom_ts_device);
        platform_device_register(&o2atom_pxa_keyboard);
	platform_device_register(&o2atom_pxa_keys);
}

static void __init o2atom_map_io(void)
{
        pxa_map_io();
        pxa_set_btuart_info(&o2atom_pxa_bt_funcs);
        pxa_set_ffuart_info(&o2atom_pxa_ff_funcs);
}

static void __init o2atom_init_irq(void)
{
        pxa_init_irq();
        pxa_set_ficp_info(&o2atom_ficp_platform_data);
}

MACHINE_START(O2ATOM, "O2 XDA Atom")
        /* Maintainer phone-hackers@mit.edu */
        .phys_io        = 0x40000000,
        .io_pg_offst    = (io_p2v(0x40000000) >> 18) & 0xfffc,
        .boot_params    = 0xa0000100,
        .map_io         = o2atom_map_io,
        .init_irq       = o2atom_init_irq,
        .init_machine   = o2atom_init,
        .timer          = &pxa_timer,
MACHINE_END
