/*
 * o2atom.c  --  SoC audio for O2 XDA Atom
 *
 * Copyright 2005 Wolfson Microelectronics PLC.
 * Copyright 2005 Openedhand Ltd.
 * Copyright 2007 Massachusetts Institute of Technology
 *
 * Authors: Liam Girdwood <liam.girdwood@wolfsonmicro.com>
 *          Richard Purdie <richard@openedhand.com>
 *          Geoffrey Thomas <geofft@mit.edu>
 *          Jeff Arnold <jbarnold@mit.edu>
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  Revision history
 *    30th Nov 2005   Initial version.
 *    July 2007       Copied from Mainstone version and adapted for O2 Atom.
 *
 *
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/device.h>

#include <sound/driver.h>
#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/soc.h>
#include <sound/soc-dapm.h>

#include <asm/mach-types.h>
//#include <asm/hardware/tmio.h>
#include <asm/arch/pxa-regs.h>
#include <asm/arch/hardware.h>
#include <asm/arch/audio.h>

#include "../codecs/wm9713.h"
#include "pxa2xx-pcm.h"
#include "pxa2xx-ac97.h"
#include "pxa2xx-ssp.h"

static struct snd_soc_machine o2atom;

#define O2ATOM_HP        0
#define O2ATOM_MIC_INT   1
#define O2ATOM_HEADSET   2
#define O2ATOM_HP_OFF    3
#define O2ATOM_SPK_ON    0
#define O2ATOM_SPK_OFF   1

static int o2atom_jack_func;
static int o2atom_spk_func;

//from mainstone_wm9713.c
static int o2atom_voice_hw_params(struct snd_pcm_substream *substream,
				struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_codec_dai *codec_dai = rtd->dai->codec_dai;
	struct snd_soc_cpu_dai *cpu_dai = rtd->dai->cpu_dai;
	unsigned int bclk = 0, pcmdiv = 0;
	int ret = 0;

	switch (params_rate(params)) {
	case 8000:
		pcmdiv = WM9713_PCMDIV(12); /* 2.048 MHz */
		bclk = WM9713_PCMBCLK_DIV_16; /* 128kHz */
		break;
	case 16001:
		pcmdiv = WM9713_PCMDIV(6); /* 4.096 MHz */
		bclk = WM9713_PCMBCLK_DIV_16; /* 256kHz */
		break;
	case 48000:
		pcmdiv = WM9713_PCMDIV(2); /* 12.288 MHz */
		bclk = WM9713_PCMBCLK_DIV_16; /* 512kHz */
		break;
	}

	if (codec_dai->dai_ops.set_fmt) {
		printk(KERN_WARNING "oh hai, i is callin codec_dai->dai_ops.set_fmt\n");
		/* set codec DAI configuration */
		ret = codec_dai->dai_ops.set_fmt(codec_dai, SND_SOC_DAIFMT_DSP_A |
				SND_SOC_DAIFMT_NB_NF | SND_SOC_DAIFMT_CBM_CFM);
		if (ret < 0)
			return ret;
	}

	if (cpu_dai->dai_ops.set_fmt) {
		printk(KERN_WARNING "oh hai, i is callin cpu_dai->dai_ops.set_fmt\n");
		/* set cpu DAI configuration */
		ret = cpu_dai->dai_ops.set_fmt(cpu_dai, SND_SOC_DAIFMT_DSP_A |
				SND_SOC_DAIFMT_NB_NF | SND_SOC_DAIFMT_CBM_CFM);
		if (ret < 0)
			return ret;
	}

	if (codec_dai->dai_ops.set_clkdiv) {
		printk(KERN_WARNING "oh hai, i is callin codec_dai->dai_ops.set_clkdiv\n");
		/* set codec BCLK division for sample rate */
		ret = codec_dai->dai_ops.set_clkdiv(codec_dai, WM9713_PCMBCLK_DIV, bclk);
		if (ret < 0)
			return ret;

		/* set codec PCM division for sample rate */
		ret = codec_dai->dai_ops.set_clkdiv(codec_dai, WM9713_PCMCLK_DIV, pcmdiv);
		if (ret < 0)
			return ret;
	}

	return 0;
}

static void o2atom_ext_control(struct snd_soc_codec *codec)
{
	int spk = 1, mic_int = 1, hp = 1, hs = 1;

	/* set up jack connection */
	switch (o2atom_jack_func) {
	case O2ATOM_HP:
		hp = 1;
		break;
	case O2ATOM_MIC_INT:
		mic_int = 1;
		break;
	case O2ATOM_HEADSET:
		hs = 1;
		break;
	}

	if (o2atom_spk_func == O2ATOM_SPK_ON)
		spk = 1;

	snd_soc_dapm_set_endpoint(codec, "Speaker", spk);
	snd_soc_dapm_set_endpoint(codec, "Mic (Internal)", mic_int);
	snd_soc_dapm_set_endpoint(codec, "Headphone Jack", hp);
	snd_soc_dapm_set_endpoint(codec, "Headset Jack", hs);
	snd_soc_dapm_sync_endpoints(codec);
}

static int o2atom_startup(struct snd_pcm_substream *substream)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_codec *codec = rtd->socdev->codec;

	/* check the jack status at stream startup */
	o2atom_ext_control(codec);
	return 0;
}

static struct snd_soc_ops o2atom_ops = {
	.startup = o2atom_startup,
	.hw_params = o2atom_voice_hw_params,
};

static int o2atom_get_jack(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = o2atom_jack_func;
	return 0;
}

static int o2atom_set_jack(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_codec *codec =  snd_kcontrol_chip(kcontrol);

	if (o2atom_jack_func == ucontrol->value.integer.value[0])
		return 0;

	o2atom_jack_func = ucontrol->value.integer.value[0];
	o2atom_ext_control(codec);
	return 1;
}

static int o2atom_get_spk(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = o2atom_spk_func;
	return 0;
}

static int o2atom_set_spk(struct snd_kcontrol *kcontrol,
	struct snd_ctl_elem_value *ucontrol)
{
	struct snd_soc_codec *codec =  snd_kcontrol_chip(kcontrol);

	if (o2atom_spk_func == ucontrol->value.integer.value[0])
		return 0;

	o2atom_spk_func = ucontrol->value.integer.value[0];
	o2atom_ext_control(codec);
	return 1;
}

#if 0
/* o2atom dapm event handlers */
static int o2atom_hp_event(struct snd_soc_dapm_widget *w, int event)
{
	if (SND_SOC_DAPM_EVENT_ON(event))
		set_tc6393_gpio(&tc6393_device.dev,O2ATOM_TC6393_L_MUTE);
	else
		reset_tc6393_gpio(&tc6393_device.dev,O2ATOM_TC6393_L_MUTE);
	return 0;
}
#endif

/* o2atom machine dapm widgets */
static const struct snd_soc_dapm_widget o2atom_dapm_widgets[] = {
//SND_SOC_DAPM_HP("Headphone Jack", o2atom_hp_event),
SND_SOC_DAPM_HP("Headphone Jack", NULL),
SND_SOC_DAPM_HP("Headset Jack", NULL),
SND_SOC_DAPM_MIC("Mic (Internal)", NULL),
SND_SOC_DAPM_SPK("Speaker", NULL),
};

/* o2atom audio map */
static const char *audio_map[][3] = {
	/* headphone connected to HPL, HPR */
	{"Headphone Jack", NULL, "HPL"},
	{"Headphone Jack", NULL, "HPR"},

	/* ext speaker connected to LOUT2, ROUT2 */
	{"Speaker", NULL, "SPKL"},
	{"Speaker", NULL, "SPKR"},

#if 0
	//{"MIC1", NULL, "Mic (Internal)"},
	/* internal mic is connected to mic1, mic2 differential - with bias */
	{"MIC2A", NULL, "Mic Bias"},
	{"MIC2B", NULL, "Mic Bias"},
	{"Mic Bias", NULL, "Mic (Internal)"},
#endif

	/* headset is connected to HPOUTR, and LINEINR with bias */
	{"Headset Jack", NULL, "HPR"},
	{"LINER", NULL, "Mic Bias"},
	{"Mic Bias", NULL, "Headset Jack"},
	{NULL, NULL, NULL},
};

static const char *jack_function[] = {"Headphone", "Mic", "Line", "Headset",
	"Off"};
static const char *spk_function[] = {"On", "Off"};
static const struct soc_enum o2atom_enum[] = {
	SOC_ENUM_SINGLE_EXT(5, jack_function),
	SOC_ENUM_SINGLE_EXT(2, spk_function),
};

static const struct snd_kcontrol_new o2atom_controls[] = {
	SOC_ENUM_EXT("Jack Function", o2atom_enum[0], o2atom_get_jack,
		o2atom_set_jack),
	SOC_ENUM_EXT("Speaker Function", o2atom_enum[1], o2atom_get_spk,
		o2atom_set_spk),
};

static int o2atom_ac97_init(struct snd_soc_codec *codec)
{
	int i, err;

//	snd_soc_dapm_set_endpoint(codec, "OUT3", 0);
//	snd_soc_dapm_set_endpoint(codec, "OUT4", 0);
//	snd_soc_dapm_set_endpoint(codec, "MONO", 0);

	/* add o2atom specific controls */
	for (i = 0; i < ARRAY_SIZE(o2atom_controls); i++) {
		err = snd_ctl_add(codec->card,
				snd_soc_cnew(&o2atom_controls[i],codec, NULL));
		if (err < 0)
			return err;
	}

	/* add o2atom specific widgets */
	for (i = 0; i < ARRAY_SIZE(o2atom_dapm_widgets); i++) {
		snd_soc_dapm_new_control(codec, &o2atom_dapm_widgets[i]);
	}

	/* set up o2atom specific audio path audio_map */
	for (i = 0; audio_map[i][0] != NULL; i++) {
		snd_soc_dapm_connect_input(codec, audio_map[i][0],
			audio_map[i][1], audio_map[i][2]);
	}

	snd_soc_dapm_sync_endpoints(codec);
	return 0;
}

static struct snd_soc_dai_link o2atom_dai[] = {
{
	.name = "AC97",
	.stream_name = "AC97 HiFi",
	.cpu_dai = &pxa_ac97_dai[PXA2XX_DAI_AC97_HIFI],
	.codec_dai = &wm9713_dai[WM9713_DAI_AC97_HIFI],
	.init = o2atom_ac97_init,
	.ops = &o2atom_ops,
},
{
	.name = "AC97 Aux",
	.stream_name = "AC97 Aux",
	.cpu_dai = &pxa_ac97_dai[PXA2XX_DAI_AC97_AUX],
	.codec_dai = &wm9713_dai[WM9713_DAI_AC97_AUX],
	.ops = &o2atom_ops,
},
{
	.name = "WM9713",
	.stream_name = "WM9713 Voice",
	.cpu_dai = &pxa_ssp_dai[PXA2XX_DAI_SSP2],
	.codec_dai = &wm9713_dai[WM9713_DAI_PCM_VOICE],
	.ops = &o2atom_ops,
},
};

static struct snd_soc_machine o2atom = {
	.name = "O2Atom",
	.dai_link = o2atom_dai,
	.num_links = ARRAY_SIZE(o2atom_dai),
};

static struct snd_soc_device o2atom_snd_devdata = {
	.machine = &o2atom,
	.platform = &pxa2xx_soc_platform,
	.codec_dev = &soc_codec_dev_wm9713,
};

static struct platform_device *o2atom_snd_device;

static int __init o2atom_init(void)
{
	int ret;

	o2atom_snd_device = platform_device_alloc("soc-audio", -1);
	if (!o2atom_snd_device)
		return -ENOMEM;

	platform_set_drvdata(o2atom_snd_device, &o2atom_snd_devdata);
	o2atom_snd_devdata.dev = &o2atom_snd_device->dev;
	ret = platform_device_add(o2atom_snd_device);

	if (ret)
		platform_device_put(o2atom_snd_device);

	return ret;
}

static void __exit o2atom_exit(void)
{
	platform_device_unregister(o2atom_snd_device);
}

module_init(o2atom_init);
module_exit(o2atom_exit);

/* Module information */
MODULE_AUTHOR("Richard Purdie");
MODULE_DESCRIPTION("ALSA SoC O2 Atom");
MODULE_LICENSE("GPL");
