/*
 *  Copyright (C) 2004 Florian Schirmer (jolt@tuxbox.org)
 *  Copyright (C) 2005 Waldemar Brodkorb <wbx@openwrt.org>
 *  Copyright (C) 2005 Felix Fietkau <nbd@openwrt.org>
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/init.h>
#include <linux/types.h>
#include <linux/tty.h>
#include <linux/serial.h>
#include <linux/serial_core.h>
#include <linux/serial_reg.h>
#include <asm/bootinfo.h>
#include <asm/time.h>
#include <asm/reboot.h>
#include <linux/pm.h>

#include <typedefs.h>
#include <osl.h>
#include <sbutils.h>
#include <sbmips.h>
#include <sbpci.h>
#include <sbconfig.h>
#include <bcmdevs.h>
#include <bcmutils.h>
#include <bcmnvram.h>

extern void bcm47xx_pci_init(void);
extern void bcm47xx_time_init(void);
extern void bcm47xx_timer_setup(struct irqaction *irq);
void *sbh;
spinlock_t sbh_lock = SPIN_LOCK_UNLOCKED;
int boardflags;

static int ser_line = 0;

typedef struct {
	void *regs;
	uint irq;
	uint baud_base;
	uint reg_shift;
} serial_port;

static serial_port ports[4];
static int num_ports = 0;

static void
serial_add(void *regs, uint irq, uint baud_base, uint reg_shift)
{
	ports[num_ports].regs = regs;
	ports[num_ports].irq = irq;
	ports[num_ports].baud_base = baud_base;
	ports[num_ports].reg_shift = reg_shift;
	num_ports++;
}

static void
do_serial_add(serial_port *port)
{
	void *regs;
	uint irq;
	uint baud_base;
	uint reg_shift;
	struct uart_port s;
	
	regs = port->regs;
	irq = port->irq;
	baud_base = port->baud_base;
	reg_shift = port->reg_shift;

	memset(&s, 0, sizeof(s));

	s.line = ser_line++;
	s.membase = regs;
	s.irq = irq + 2;
	s.uartclk = baud_base;
	s.flags = ASYNC_BOOT_AUTOCONF;
	s.iotype = SERIAL_IO_MEM;
	s.regshift = reg_shift;

	if (early_serial_setup(&s) != 0) {
		printk(KERN_ERR "Serial setup failed!\n");
	}
}

static void bcm47xx_machine_restart(char *command)
{
	printk("Please stand by while rebooting the system...\n");
	 
	/* Set the watchdog timer to reset immediately */
	local_irq_disable();
	sb_watchdog(sbh, 1);
	while (1);
}

static void bcm47xx_machine_halt(void)
{
	/* Disable interrupts and watchdog and spin forever */
	local_irq_disable();
	sb_watchdog(sbh, 0);
	while (1);
}

void __init plat_setup(void)
{
	char *s;
	int i;
	
	sbh = (void *) sb_kattach();
	sb_mips_init(sbh);

	bcm47xx_pci_init();

	sb_serial_init(sbh, serial_add);
	boardflags = getintvar(NULL, "boardflags");

	/* reverse serial ports if the nvram variable kernel_args starts with console=ttyS1 */
	s = early_nvram_get("kernel_args");
	if (!s)	s = "";
	if (!strncmp(s, "console=ttyS1", 13)) {
		for (i = num_ports; i; i--)
			do_serial_add(&ports[i - 1]);
	} else {
		for (i = 0; i < num_ports; i++)
			do_serial_add(&ports[i]);
	}
	
	_machine_restart = bcm47xx_machine_restart;
	_machine_halt = bcm47xx_machine_halt;
	pm_power_off = bcm47xx_machine_halt;
	
	board_time_init = bcm47xx_time_init;
	board_timer_setup = bcm47xx_timer_setup;
}

EXPORT_SYMBOL(sbh);
EXPORT_SYMBOL(sbh_lock);
EXPORT_SYMBOL(boardflags);
