/* linux/drivers/char/ar531x_gpio.c 

   Allows a user space process to play with the GPIO pins.

   Copyright (c) 2001,2002 Christer Weinigel <wingel@nano-system.com> */

#include <linux/config.h>
#include <linux/fs.h>
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <asm/io.h>

#define NAME "ar531x_gpio"

MODULE_AUTHOR("");
MODULE_DESCRIPTION("AR531X GPIO Pin Driver");
MODULE_LICENSE("GPL");

static int major = 254;		/* default to 254 */
module_param(major, int, 0);
MODULE_PARM_DESC(major, "Major device number");


typedef unsigned int AR531X_REG;

#define PHYS_TO_K1(physaddr) KSEG1ADDR(physaddr)

#define sysRegRead(phys)	\
	(*(volatile AR531X_REG *)PHYS_TO_K1(phys))

#define sysRegWrite(phys, val)	\
	((*(volatile AR531X_REG *)PHYS_TO_K1(phys)) = (val))

#define AR5315_DSLBASE          0xB1000000      /* RESET CONTROL MMR */

#define AR5315_GPIO_DI			(AR5315_DSLBASE + 0x0088)
#define AR5315_GPIO_DO			(AR5315_DSLBASE + 0x0090)
#define AR5315_GPIO_DIR			(AR5315_DSLBASE + 0x0098)
#define AR5315_GPIO_INTR_CR		(AR5315_DSLBASE + 0x00a0)

#define LED_BIT (1<<7)

static ssize_t ar531x_gpio_write(struct file *file, const char __user *data, 
				 size_t len, loff_t *ppos)
{
	unsigned m = iminor(file->f_dentry->d_inode);
	size_t i;
	AR531X_REG oldgpioval = sysRegRead(AR5315_GPIO_DO);

	for (i = 0; i < len; ++i) {
		char c;
		if (get_user(c, data+i))
			return -EFAULT;
		switch (c)
		{
		case '0':
			sysRegWrite(AR5315_GPIO_DO, oldgpioval & ~(1 << (m&31)));
			break;
		case '1': 
			sysRegWrite(AR5315_GPIO_DO, oldgpioval | (1 << (m&31)));
			break;
		default:
			break;
		}
	}

	return len;
}

static ssize_t ar531x_gpio_read(struct file *file, char __user *buf,
				size_t len, loff_t *ppos)
{
	unsigned m = iminor(file->f_dentry->d_inode);
	int value;

	value = sysRegRead(AR5315_GPIO_DO) & (1<< (m&31));
	if (put_user(value ? '1' : '0', buf))
		return -EFAULT;
	
	return 1;
}

static int ar531x_gpio_open(struct inode *inode, struct file *file)
{
	unsigned m = iminor(inode);
	if (m > 63)
		return -EINVAL;
	return nonseekable_open(inode, file);
}

static int ar531x_gpio_release(struct inode *inode, struct file *file)
{
	return 0;
}


static struct file_operations ar531x_gpio_fops = {
	.owner   = THIS_MODULE,
	.write   = ar531x_gpio_write,
	.read    = ar531x_gpio_read,
	.open    = ar531x_gpio_open,
	.release = ar531x_gpio_release,
};

static int __init ar531x_gpio_init(void)
{
	int r;

	printk(KERN_DEBUG NAME ": AR5315 GPIO Driver\n");

	r = register_chrdev(major, NAME, &ar531x_gpio_fops);
	if (r < 0) {
		printk(KERN_ERR NAME ": unable to register character device\n");
		return r;
	}
	if (!major) {
		major = r;
		printk(KERN_DEBUG NAME ": got dynamic major %d\n", major);
	}

	return 0;
}

static void __exit ar531x_gpio_cleanup(void)
{
	unregister_chrdev(major, NAME);
}

module_init(ar531x_gpio_init);
module_exit(ar531x_gpio_cleanup);

/*
    Local variables:
        compile-command: "make -k -C ../.. SUBDIRS=drivers/char modules"
        c-basic-offset: 8
    End:
*/
