/*-
 * Copyright (c) 2005 John Bicket
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: ieee80211_monitor.c 1483 2006-03-23 20:12:18Z jbicket $
 */

#ifndef EXPORT_SYMTAB
#define	EXPORT_SYMTAB
#endif

/*
 * IEEE 802.11 monitor mode 
 */
#include <linux/config.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/kmod.h>
#include <linux/init.h>
#include <linux/skbuff.h>
#include <linux/sysctl.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/if_vlan.h>

#include <net/iw_handler.h>
#include <linux/wireless.h>
#include <linux/if_arp.h>		/* XXX for ARPHRD_* */

#include <asm/uaccess.h>

#include "if_media.h"
#include "if_ethersubr.h"

#include <net80211/ieee80211_var.h>
#include <net80211/ieee80211_monitor.h>
#include <ath/if_athvar.h>

#define WIFI_EXTRA_MAGIC  0x74

enum {
  WIFI_EXTRA_TX			= (1<<0), /* packet transmission */
  WIFI_EXTRA_TX_FAIL		= (1<<1), /* transmission failed */
  WIFI_EXTRA_TX_USED_ALT_RATE	= (1<<2), /* used alternate bitrate */
  WIFI_EXTRA_RX_ERR		= (1<<3), /* failed crc check */
  WIFI_EXTRA_RX_MORE		= (1<<4), /* first part of a fragmented skb */
  WIFI_EXTRA_NO_TXF		= (1<<5),
  WIFI_EXTRA_DO_RTS_CTS		= (1<<6),
  WIFI_EXTRA_DO_CTS		= (1<<7)
};

struct click_wifi_extra {
  uint8_t magic;
  uint8_t flags;
  uint16_t len;

  uint8_t rssi;
  uint8_t silence;
  uint8_t power;
  uint8_t retries;

  uint8_t rate;			/* bitrate in Mbps*2 */
  uint8_t rate1;		/* bitrate in Mbps*2 */
  uint8_t rate2;		/* bitrate in Mbps*2 */
  uint8_t rate3;		/* bitrate in Mbps*2 */

  uint8_t max_tries;
  uint8_t max_tries1;
  uint8_t max_tries2;
  uint8_t max_tries3;
};


void
ieee80211_monitor_encap(struct ieee80211vap *vap, struct sk_buff *skb) 
{
	struct ieee80211_cb *cb = (struct ieee80211_cb *) skb->cb;
	struct ieee80211_phy_params *ph =
		(struct ieee80211_phy_params *) (skb->cb + sizeof(struct ieee80211_cb));
	cb = (struct ieee80211_cb *) skb->cb;
	cb->flags = M_RAW;
	cb->ni = NULL;
	cb->next = NULL;
	memset(ph, 0, sizeof(struct ieee80211_phy_params));

	/* send at a static rate if it is configured */
	ph->rate0 = vap->iv_fixed_rate > 0 ? vap->iv_fixed_rate : 2;
	ph->try0 = 11;
	ph->power = 60;

	switch (skb->dev->type) {
	case ARPHRD_IEEE80211: break;
	case ARPHRD_IEEE80211_PRISM: {
		wlan_ng_prism2_header *wh = 
			(wlan_ng_prism2_header *) skb->data;
		/* does it look like there is a prism header here? */
		if (skb->len > sizeof (wlan_ng_prism2_header) &&
                    wh->msgcode == DIDmsg_lnxind_wlansniffrm &&
		    wh->rate.did == DIDmsg_lnxind_wlansniffrm_rate) {
                        ph->rate0 = wh->rate.data;
                        skb_pull(skb, sizeof(wlan_ng_prism2_header));
		}
                break;
	}
	case ARPHRD_IEEE80211_ATHDESC: {
		if (skb->len > sizeof(struct click_wifi_extra)) {
			struct click_wifi_extra *desc = (struct click_wifi_extra *) skb->data;
			ph->power = desc->power;
			ph->rate0 = desc->rate;
			ph->rate1 = desc->rate1;
			ph->rate2 = desc->rate2;
			ph->rate3 = desc->rate3;
			ph->try0 = desc->max_tries;
			ph->try1 = desc->max_tries1;
			ph->try2 = desc->max_tries2;
			ph->try3 = desc->max_tries3;
			skb_pull(skb, sizeof(struct click_wifi_extra));
		}
		break;
	}		
	default:
		break;
	}

	if (!ph->rate0) {
		ph->rate0 = 0;
		ph->try0 = 11;
	}
}
EXPORT_SYMBOL(ieee80211_monitor_encap);

void
ieee80211_input_monitor(struct ieee80211com *ic, struct sk_buff *skb,
	struct ath_desc *ds, int tx, u_int32_t mactime, struct ath_softc *sc) 
{
	struct ieee80211vap *vap, *next;
	u_int32_t signal = 0;
	signal = tx ? ds->ds_txstat.ts_rssi : ds->ds_rxstat.rs_rssi;
	
	/* XXX locking */
	for (vap = TAILQ_FIRST(&ic->ic_vaps); vap != NULL; vap = next) {
		struct sk_buff *skb1;
		struct net_device *dev = vap->iv_dev;
		struct ieee80211_frame *wh = (struct ieee80211_frame *)skb->data;
		u_int8_t dir = wh->i_fc[1] & IEEE80211_FC1_DIR_MASK;

		next = TAILQ_NEXT(vap, iv_next);
		if (vap->iv_opmode != IEEE80211_M_MONITOR ||
		    vap->iv_state != IEEE80211_S_RUN)
			continue;
		if (vap->iv_monitor_nods_only &&
		    dir != IEEE80211_FC1_DIR_NODS) {
			/* don't rx fromds, tods, or dstods packets */
			continue;
		}		    
		skb1 = skb_copy(skb, GFP_ATOMIC);
		if (skb1 == NULL) {
			/* XXX stat+msg */
			continue;
		}		
		if (vap->iv_monitor_txf_len && tx) {
			/* truncate transmit feedback packets */
			skb_trim(skb1, vap->iv_monitor_txf_len);
			skb1->nh.raw = skb1->data;
		}
		switch (vap->iv_dev->type) {
		case ARPHRD_IEEE80211:
			break;
		case ARPHRD_IEEE80211_PRISM: {
			wlan_ng_prism2_header *ph;
			if (skb_headroom(skb1) < sizeof(wlan_ng_prism2_header)) {
				dev_kfree_skb(skb1);
				skb1 = NULL;
				break;
			}
			
			ph = (wlan_ng_prism2_header *)
				skb_push(skb1, sizeof(wlan_ng_prism2_header));
			memset(ph, 0, sizeof(wlan_ng_prism2_header));
			
			ph->msgcode = DIDmsg_lnxind_wlansniffrm;
			ph->msglen = sizeof(wlan_ng_prism2_header);
			strncpy(ph->devname, dev->name, sizeof(ph->devname));
			
			ph->hosttime.did = DIDmsg_lnxind_wlansniffrm_hosttime;
			ph->hosttime.status = 0;
			ph->hosttime.len = 4;
			ph->hosttime.data = jiffies;
			/* Pass up tsf clock in mactime */
			ph->mactime.did = DIDmsg_lnxind_wlansniffrm_mactime;
			ph->mactime.status = 0;
			ph->mactime.len = 4;
			ph->mactime.data = mactime;
			
			ph->istx.did = DIDmsg_lnxind_wlansniffrm_istx;
			ph->istx.status = 0;
			ph->istx.len = 4;
			ph->istx.data = tx ? P80211ENUM_truth_true : P80211ENUM_truth_false;
			
			ph->frmlen.did = DIDmsg_lnxind_wlansniffrm_frmlen;
			ph->frmlen.status = 0;
			ph->frmlen.len = 4;
			ph->frmlen.data = skb->len - sizeof(wlan_ng_prism2_header);
			
			ph->channel.did = DIDmsg_lnxind_wlansniffrm_channel;
			ph->channel.status = 0;
			ph->channel.len = 4;
			ph->channel.data =
				ieee80211_mhz2ieee(ic->ic_curchan->ic_freq, 
					ic->ic_curchan->ic_flags);
			
			ph->rssi.did = DIDmsg_lnxind_wlansniffrm_rssi;
			ph->rssi.status = 0;
			ph->rssi.len = 4;
			ph->rssi.data = signal;
			
			ph->noise.did = DIDmsg_lnxind_wlansniffrm_noise;
			ph->noise.status = 0;
			ph->noise.len = 4;
			ph->noise.data = -95;
			
			ph->signal.did = DIDmsg_lnxind_wlansniffrm_signal;
			ph->signal.status = 0;
			ph->signal.len = 4;
			ph->signal.data = signal;
			
			ph->rate.did = DIDmsg_lnxind_wlansniffrm_rate;
			ph->rate.status = 0;
			ph->rate.len = 4;
			ph->rate.data = sc->sc_hwmap[ds->ds_rxstat.rs_rate].ieeerate;
			break;
		}
		case ARPHRD_IEEE80211_RADIOTAP: {
			if (tx) {
				struct ath_tx_radiotap_header *th;
				if (skb_headroom(skb1) < sizeof(struct ath_tx_radiotap_header)) {
					printk("%s:%d %s\n", __FILE__, __LINE__, __func__);
					dev_kfree_skb(skb1);
					skb1 = NULL;
					break;
				}
				
				th = (struct ath_tx_radiotap_header *) skb_push(skb1, 
					sizeof(struct ath_tx_radiotap_header));
				memset(th, 0, sizeof(struct ath_tx_radiotap_header));
				th->wt_ihdr.it_version = 0;
				th->wt_ihdr.it_len = cpu_to_le16(sizeof(struct ath_tx_radiotap_header));
				th->wt_ihdr.it_present = ATH_TX_RADIOTAP_PRESENT;
				th->wt_flags = 0;
				th->wt_rate = sc->sc_hwmap[ds->ds_rxstat.rs_rate].ieeerate;
				th->wt_txpower = 0;
				th->wt_antenna = 0;
			} else {
				struct ath_rx_radiotap_header *th;
				if (skb_headroom(skb1) < sizeof(struct ath_rx_radiotap_header)) {
					printk("%s:%d %s\n", __FILE__, __LINE__, __func__);
					dev_kfree_skb(skb1);
					skb1 = NULL;
					break;
				}
				
				th = (struct ath_rx_radiotap_header *) skb_push(skb1, 
					sizeof(struct ath_rx_radiotap_header));
				memset(th, 0, sizeof(struct ath_rx_radiotap_header));
				th->wr_ihdr.it_version = 0;
				th->wr_ihdr.it_len = cpu_to_le16(sizeof(struct ath_rx_radiotap_header));
				th->wr_ihdr.it_present = ATH_RX_RADIOTAP_PRESENT;
				th->wr_flags = 0;
				th->wr_rate = sc->sc_hwmap[ds->ds_rxstat.rs_rate].ieeerate;
				th->wr_chan_freq = ic->ic_curchan->ic_freq;

				/* Define the channel flags for radiotap */
				switch(sc->sc_curmode) {
					case IEEE80211_MODE_11A:
						th->wr_chan_flags =
							cpu_to_le16(IEEE80211_CHAN_A);
						break;
					case IEEE80211_MODE_TURBO_A:
						th->wr_chan_flags = 
							cpu_to_le16(IEEE80211_CHAN_TA);
						break;
					case IEEE80211_MODE_11B:
						th->wr_chan_flags = 
							cpu_to_le16(IEEE80211_CHAN_B);
						break;
					case IEEE80211_MODE_11G:
						th->wr_chan_flags = IEEE80211_CHAN_G;
						break;
					case IEEE80211_MODE_TURBO_G:
						th->wr_chan_flags = IEEE80211_CHAN_TG;
						break;
					default:
						th->wr_chan_flags = 0; /* unknown */
						break;
				}

				th->wr_antenna = 0;
				th->wr_antsignal = signal;
				memcpy(&th->wr_fcs, &skb1->data[skb1->len - IEEE80211_CRC_LEN],
				       IEEE80211_CRC_LEN);
			}
			break;
		}
		case ARPHRD_IEEE80211_ATHDESC: {
			struct click_wifi_extra *e = NULL;
			if (skb_headroom(skb1) < sizeof(struct click_wifi_extra)) {
				printk("%s:%d %s\n", __FILE__, __LINE__, __func__);
				dev_kfree_skb(skb1);
				skb1 = NULL;
				break;
			}
			e = (struct click_wifi_extra *) skb_push(skb1, sizeof(struct click_wifi_extra));
			memset(e, 0, sizeof(struct click_wifi_extra));
			e->magic = WIFI_EXTRA_MAGIC;
			e->flags = 0;
			if (tx) {
				e->flags |= WIFI_EXTRA_TX;
				e->rssi = ds->ds_txstat.ts_rssi;
				e->power = 0;
				e->rate = 0;
				e->rate1 = 0;
				e->rate2 = 0;
				e->rate3 = 0;
				e->max_tries = 0;
				e->max_tries1 = 0;
				e->max_tries2 = 0;
				e->max_tries3 = 0;
				e->retries = ds->ds_txstat.ts_longretry;
			} else {
				e->rate = sc->sc_hwmap[ds->ds_rxstat.rs_rate].ieeerate;
				e->rssi = signal;
				e->silence = 0;
			}
			break;
		}
		default:
			break;
		}
		if (skb1) {
			if (!tx) {
				/* Remove FCS from end of rx frames*/
				skb_trim(skb1, skb1->len - IEEE80211_CRC_LEN);
			}
			skb1->dev = dev; /* NB: deliver to wlanX */
			skb1->mac.raw = skb1->data;
			skb1->ip_summed = CHECKSUM_NONE;
			skb1->pkt_type = PACKET_OTHERHOST;
			skb1->protocol = __constant_htons(0x0019); /* ETH_P_80211_RAW */
			
			netif_rx(skb1);
			
			vap->iv_devstats.rx_packets++;
			vap->iv_devstats.rx_bytes += skb1->len;
		}
	}
}
EXPORT_SYMBOL(ieee80211_input_monitor);
