/*-
 * Copyright (c) 2002-2005 John Bicket, Meraki Networks
 * All rights reserved.
 *
 */
#ifndef EXPORT_SYMTAB
#define	EXPORT_SYMTAB
#endif

/*
 * IEEE 802.11 station scanning support.
 */
#include <linux/config.h>
#include <linux/version.h>
#include <linux/module.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/init.h>
#include <linux/delay.h>

#include "if_media.h"

#include <net80211/ieee80211_var.h>

/*
 * Parameters for managing cache entries:
 *
 * o a station with AP_FAILS_MAX failures is not considered
 *   when picking a candidate
 * o a station that hasn't had an update in AP_PURGE_SCANS
 *   (background) scans is discarded
 * o after AP_FAILS_AGE seconds we clear the failure count
 */
#define	AP_FAILS_MAX	2		/* assoc failures before ignored */
#define	AP_FAILS_AGE	(2 * 60)	/* time before clearing fails (secs) */
#define	AP_PURGE_SCANS	2		/* age for purging entries (scans) */

/* XXX tunable */
#define	AP_RSSI_MIN	8		/* min acceptable rssi */

#define RSSI_LPF_LEN	10
#define	RSSI_EP_MULTIPLIER	(1<<7)	/* pow2 to optimize out * and / */
#define RSSI_IN(x)		((x) * RSSI_EP_MULTIPLIER)
#define LPF_RSSI(x, y, len)	(((x) * ((len) - 1) + (y)) / (len))
#define RSSI_LPF(x, y) do {						\
    if ((y) >= -20)							\
    	x = LPF_RSSI((x), RSSI_IN((y)), RSSI_LPF_LEN);			\
} while (0)
#define	EP_RND(x, mul) \
	((((x)%(mul)) >= ((mul)/2)) ? howmany(x, mul) : (x)/(mul))
#define	RSSI_GET(x)	EP_RND(x, RSSI_EP_MULTIPLIER)

struct ap_entry {
	struct ieee80211_scan_entry base;

	TAILQ_ENTRY(ap_entry) se_list;
	LIST_ENTRY(ap_entry) se_hash;
	u_int8_t	se_fails;		/* failure to associate count */
	u_int8_t	se_seen;		/* seen during current scan */
	u_int8_t	se_notseen;		/* not seen in previous scans */
	u_int32_t se_avgrssi;		/* LPF rssi state */
	unsigned long se_lastupdate;	/* time of last update */
	unsigned long se_lastfail;	/* time of last failure */
	unsigned long se_lastassoc;	/* time of last association */
	u_int se_scangen;		/* iterator scan gen# */

};

#define	AP_HASHSIZE	32
/* ap hash is enough for variation of macaddr */
#define	AP_HASH(addr)	\
	(((const u_int8_t *)(addr))[IEEE80211_ADDR_LEN - 1] % AP_HASHSIZE)

struct ap_table {
	spinlock_t st_lock;			/* on scan table */
	TAILQ_HEAD(, ap_entry) st_entry;	/* all entries */
	ATH_LIST_HEAD(, ap_entry) st_hash[AP_HASHSIZE];
	spinlock_t st_scanlock;			/* on st_scangen */
	u_int st_scangen;			/* gen# for iterator */
	int st_newscan;

	int as_maxrssi[IEEE80211_CHAN_MAX];
	struct IEEE80211_TQ_STRUCT as_actiontq;	/* tasklet for "action" */
	struct ieee80211_scan_entry as_selbss;	/* selected bss for action tasklet */
	int (*as_action)(struct ieee80211vap *, const struct ieee80211_scan_entry *);

};

static int ap_flush(struct ieee80211_scan_state *);
static void ap_flush_table(struct ap_table *);
static void action_tasklet(IEEE80211_TQUEUE_ARG);
/*
 * Attach prior to any scanning work.
 */
static int
ap_attach(struct ieee80211_scan_state *ss)
{
	struct ap_table *st;

	_MOD_INC_USE(THIS_MODULE, return 0);

	MALLOC(st, struct ap_table *, sizeof(struct ap_table),
		M_80211_SCAN, M_NOWAIT | M_ZERO);
	if (st == NULL)
		return 0;
	spin_lock_init(&st->st_lock);
	spin_lock_init(&st->st_scanlock);
	TAILQ_INIT(&st->st_entry);
	ss->ss_priv = st;
	IEEE80211_INIT_TQUEUE(&st->as_actiontq, action_tasklet, ss);
	ap_flush_table(st);

	return 1;
}

/*
 * Cleanup any private state.
 */
static int
ap_detach(struct ieee80211_scan_state *ss)
{
	struct ap_table *st = ss->ss_priv;

	if (st != NULL) {
		ap_flush_table(st);
		FREE(st, M_80211_SCAN);
	}

	_MOD_DEC_USE(THIS_MODULE);
	return 1;
}

/*
 * Flush all per-scan state.
 */
static int
ap_flush(struct ieee80211_scan_state *ss)
{
	struct ap_table *st = ss->ss_priv;

	spin_lock_bh(&st->st_lock);
	ap_flush_table(st);
	spin_unlock_bh(&st->st_lock);
	ss->ss_last = 0;
	return 0;
}

/*
 * Flush all entries in the scan cache.
 */
static void
ap_flush_table(struct ap_table *st)
{
	struct ap_entry *se, *next;

	TAILQ_FOREACH_SAFE(se, &st->st_entry, se_list, next) {
		TAILQ_REMOVE(&st->st_entry, se, se_list);
		LIST_REMOVE(se, se_hash);
		if (se->base.se_wme_ie)
			FREE(se->base.se_wme_ie, M_80211_SCAN);
		if (se->base.se_wpa_ie)
			FREE(se->base.se_wpa_ie, M_80211_SCAN);
		if (se->base.se_rsn_ie)
			FREE(se->base.se_rsn_ie, M_80211_SCAN);
		if (se->base.se_ath_ie)
			FREE(se->base.se_ath_ie, M_80211_SCAN);
		if (se->base.se_meraki_ie)
			FREE(se->base.se_meraki_ie, M_80211_SCAN);


		FREE(se, M_80211_SCAN);
	}
}

static void
saveie(u_int8_t **iep, const u_int8_t *ie)
{
	if (ie == NULL)
		*iep = NULL;
	else
		ieee80211_saveie(iep, ie);
}

/*
 * Process a beacon or probe response frame; create an
 * entry in the scan cache or update any previous entry.
 */
static int
ap_add(struct ieee80211_scan_state *ss, const struct ieee80211_scanparams *sp,
	const struct ieee80211_frame *wh, int subtype, int rssi, int rstamp)
{
#define	ISPROBE(_st)	((_st) == IEEE80211_FC0_SUBTYPE_PROBE_RESP)
#define	PICK1ST(_ss) \
	((ss->ss_flags & (IEEE80211_SCAN_PICK1ST | IEEE80211_SCAN_GOTPICK)) == \
	IEEE80211_SCAN_PICK1ST)
	struct ap_table *st = ss->ss_priv;
	const u_int8_t *macaddr = wh->i_addr2;
	struct ieee80211vap *vap = ss->ss_vap;
	struct ieee80211com *ic = vap->iv_ic;
	struct ap_entry *se;
	struct ieee80211_scan_entry *ise;
	int hash;

	hash = AP_HASH(macaddr);
	spin_lock_bh(&st->st_lock);  
	LIST_FOREACH(se, &st->st_hash[hash], se_hash)
		if (IEEE80211_ADDR_EQ(se->base.se_macaddr, macaddr) &&
		    sp->ssid[1] == se->base.se_ssid[1] && 
		    !memcmp(se->base.se_ssid+2, sp->ssid+2, se->base.se_ssid[1]))
			goto found;

	MALLOC(se, struct ap_entry *, sizeof(struct ap_entry),
		M_80211_SCAN, M_NOWAIT | M_ZERO);
	if (se == NULL) {
		spin_unlock_bh(&st->st_lock);
		return 0;
	}
	se->se_scangen = st->st_scangen-1;
	IEEE80211_ADDR_COPY(se->base.se_macaddr, macaddr);
	TAILQ_INSERT_TAIL(&st->st_entry, se, se_list);
	LIST_INSERT_HEAD(&st->st_hash[hash], se, se_hash);
found:
	ise = &se->base;
	/* XXX ap beaconing multiple ssid w/ same bssid */
	if (sp->ssid[1] != 0 &&
	    (ISPROBE(subtype) || ise->se_ssid[1] == 0))
		memcpy(ise->se_ssid, sp->ssid, 2 + sp->ssid[1]);
	KASSERT(sp->rates[1] <= IEEE80211_RATE_MAXSIZE,
		("rate set too large: %u", sp->rates[1]));
	memcpy(ise->se_rates, sp->rates, 2 + sp->rates[1]);
	if (sp->xrates != NULL) {
		/* XXX validate xrates[1] */
		KASSERT(sp->xrates[1] <= IEEE80211_RATE_MAXSIZE,
			("xrate set too large: %u", sp->xrates[1]));
		memcpy(ise->se_xrates, sp->xrates, 2 + sp->xrates[1]);
	} else
		ise->se_xrates[1] = 0;
	IEEE80211_ADDR_COPY(ise->se_bssid, wh->i_addr3);
	/*
	 * Record rssi data using extended precision LPF filter.
	 */
	if (se->se_lastupdate == 0)		/* first sample */
		se->se_avgrssi = RSSI_IN(rssi);
	else					/* avg w/ previous samples */
		RSSI_LPF(se->se_avgrssi, rssi);
	se->base.se_rssi = RSSI_GET(se->se_avgrssi);
	ise->se_rstamp = rstamp;
	memcpy(ise->se_tstamp.data, sp->tstamp, sizeof(ise->se_tstamp));
	ise->se_intval = sp->bintval;
	ise->se_capinfo = sp->capinfo;
	ise->se_chan = ic->ic_curchan;
	ise->se_fhdwell = sp->fhdwell;
	ise->se_fhindex = sp->fhindex;
	ise->se_erp = sp->erp;
	ise->se_timoff = sp->timoff;
	if (IEEE80211_ADDR_EQ(wh->i_addr1, vap->iv_myaddr)) {
		/* these were unicast to me, so the rssi readings are valid */
		if (sp->rcpi) {
			ise->se_rcpi = sp->rcpi;
		}
		if (sp->rsni) {
			ise->se_rsni = sp->rsni;
		}
		if (sp->rev_rssi) {
			ise->se_rev_rssi = sp->rev_rssi;
		}
		if (sp->meraki) {
			struct ieee80211_meraki_param *m = (struct ieee80211_meraki_param *) sp->meraki;
			if (m->rev_rssi) {
				ise->se_rev_rssi = m->rev_rssi;
			}
		}
	}

	if (sp->tim != NULL) {
		const struct ieee80211_tim_ie *tim =
		    (const struct ieee80211_tim_ie *) sp->tim;
		ise->se_dtimperiod = tim->tim_period;
	}
	saveie(&ise->se_wme_ie, sp->wme);
	saveie(&ise->se_wpa_ie, sp->wpa);
	saveie(&ise->se_rsn_ie, sp->rsn);
	saveie(&ise->se_ath_ie, sp->ath);

	if (ISPROBE(subtype)) {
		/* only save meraki ie on probe responses for now */
		saveie(&ise->se_meraki_ie, sp->meraki);
	}
	
	/* clear failure count after AP_FAIL_AGE passes */
	if (se->se_fails && (jiffies - se->se_lastfail) > AP_FAILS_AGE*HZ) {
		se->se_fails = 0;
		IEEE80211_NOTE_MAC(vap, IEEE80211_MSG_SCAN, macaddr,
			"%s: fails %u", __func__, se->se_fails);
	}

	se->se_lastupdate = jiffies;		/* update time */
	se->se_seen = 1;
	se->se_notseen = 0;

	spin_unlock_bh(&st->st_lock);

	/*
	 * If looking for a quick choice and nothing's
	 * been found check here.
	 */

	return 1;
#undef PICK1ST
#undef ISPROBE
}

static int 
ap_probe(struct ieee80211_scan_state *ss, struct ieee80211vap *vap)
{
	struct net_device *dev = vap->iv_dev;
	int x = 0;
	int i = 0;
	/*
	 * Send a broadcast probe request followed by
	 * any specified directed probe requests.
	 */
	for (x = 0; x < 2; x++) {
		ieee80211_send_probereq(vap->iv_bss,
					vap->iv_myaddr, dev->broadcast,
					dev->broadcast,
					"", 0,
					vap->iv_opt_ie, vap->iv_opt_ie_len);
		for (i = 0; i < ss->ss_nssid; i++)
			ieee80211_send_probereq(vap->iv_bss,
						vap->iv_myaddr, dev->broadcast,
						dev->broadcast,
						ss->ss_ssid[i].ssid,
						ss->ss_ssid[i].len,
						vap->iv_opt_ie, vap->iv_opt_ie_len);
	}
	return 0;
}

static struct ieee80211_channel *
find11gchannel(struct ieee80211com *ic, int i, int freq)
{
	struct ieee80211_channel *c;
	int j;

	/*
	 * The normal ordering in the channel list is b channel
	 * immediately followed by g so optimize the search for
	 * this.  We'll still do a full search just in case.
	 */
	for (j = i+1; j < ic->ic_nchans; j++) {
		c = &ic->ic_channels[j];
		if (c->ic_freq == freq && IEEE80211_IS_CHAN_ANYG(c))
			return c;
	}
	for (j = 0; j < i; j++) {
		c = &ic->ic_channels[j];
		if (c->ic_freq == freq && IEEE80211_IS_CHAN_ANYG(c))
			return c;
	}
	return NULL;
}
static const u_int chanflags[] = {
	IEEE80211_CHAN_B,	/* IEEE80211_MODE_AUTO */
	IEEE80211_CHAN_A,	/* IEEE80211_MODE_11A */
	IEEE80211_CHAN_B,	/* IEEE80211_MODE_11B */
	IEEE80211_CHAN_PUREG,	/* IEEE80211_MODE_11G */
	IEEE80211_CHAN_FHSS,	/* IEEE80211_MODE_FH */
	IEEE80211_CHAN_A,	/* IEEE80211_MODE_TURBO_A */ /* for turbo mode look for AP in normal channel */
	IEEE80211_CHAN_PUREG,	/* IEEE80211_MODE_TURBO_G */
	IEEE80211_CHAN_ST,	/* IEEE80211_MODE_TURBO_STATIC_A */
};

static void
add_channels(struct ieee80211com *ic,
	struct ieee80211_scan_state *ss,
	enum ieee80211_phymode mode, const u_int16_t freq[], int nfreq)
{
#define	N(a)	(sizeof(a) / sizeof(a[0]))
	struct ieee80211_channel *c, *cg;
	u_int modeflags;
	int i;

	KASSERT(mode < N(chanflags), ("Unexpected mode %u", mode));
	modeflags = chanflags[mode];
	for (i = 0; i < nfreq; i++) {
		c = ieee80211_find_channel(ic, freq[i], modeflags);
		if (c == NULL || isclr(ic->ic_chan_active, c->ic_ieee))
			continue;
		if (mode == IEEE80211_MODE_AUTO) {
			/*
			 * XXX special-case 11b/g channels so we select
			 *     the g channel if both are present.
			 */
			if (IEEE80211_IS_CHAN_B(c) &&
			    (cg = find11gchannel(ic, i, c->ic_freq)) != NULL)
				c = cg;
		}
		if (ss->ss_last >= IEEE80211_SCAN_MAX)
			break;
		ss->ss_chans[ss->ss_last++] = c;
	}
#undef N
}

static const u_int16_t rcl1[] =		/* 8 FCC channel: 52, 56, 60, 64, 36, 40, 44, 48 */
{ 5260, 5280, 5300, 5320, 5180, 5200, 5220, 5240 };
static const u_int16_t rcl2[] =		/* 4 MKK channels: 34, 38, 42, 46 */
{ 5170, 5190, 5210, 5230 };
static const u_int16_t rcl3[] =		/* 2.4Ghz ch: 1,6,11,7,13 */
{ 2412, 2437, 2462, 2442, 2472 };
static const u_int16_t rcl4[] =		/* 5 FCC channel: 149, 153, 161, 165 */
{ 5745, 5765, 5785, 5805, 5825 };
static const u_int16_t rcl7[] =		/* 11 ETSI channel: 100,104,108,112,116,120,124,128,132,136,140 */
{ 5500, 5520, 5540, 5560, 5580, 5600, 5620, 5640, 5660, 5680, 5700 };
static const u_int16_t rcl8[] =		/* 2.4Ghz ch: 2,3,4,5,8,9,10,12 */
{ 2417, 2422, 2427, 2432, 2447, 2452, 2457, 2467 };
static const u_int16_t rcl9[] =		/* 2.4Ghz ch: 14 */
{ 2484 };
static const u_int16_t rcl10[] =	/* Added Korean channels 2312-2372 */
{ 2312, 2317, 2322, 2327, 2332, 2337, 2342, 2347, 2352, 2357, 2362, 2367, 2372 };
static const u_int16_t rcl11[] =	/* Added Japan channels in 4.9/5.0 spectrum */
{ 5040, 5060, 5080, 4920, 4940, 4960, 4980 };
#ifdef ATH_TURBO_SCAN
static const u_int16_t rcl5[] =		/* 3 static turbo channels */
{ 5210, 5250, 5290 };
static const u_int16_t rcl6[] =		/* 2 static turbo channels */
{ 5760, 5800 };
static const u_int16_t rcl6x[] =		/* 4 FCC3 turbo channels */
{ 5540, 5580, 5620, 5660 };
static const u_int16_t rcl12[] =		/* 2.4Ghz Turbo channel 6 */
{ 2437 };
static const u_int16_t rcl13[] =		/* dynamic Turbo channels */
{ 5200, 5240, 5280, 5765, 5805 };
#endif /* ATH_TURBO_SCAN */

struct scanlist {
	u_int16_t	mode;
	u_int16_t	count;
	const u_int16_t	*list;
};

#define	IEEE80211_MODE_TURBO_STATIC_A	IEEE80211_MODE_MAX
#define	X(a)	.count = sizeof(a)/sizeof(a[0]), .list = a

static const struct scanlist staScanTable[] = {
	{ IEEE80211_MODE_11B,   		X(rcl3) },
	{ IEEE80211_MODE_11A,   		X(rcl1) },
	{ IEEE80211_MODE_11A,   		X(rcl2) },
	{ IEEE80211_MODE_11B,   		X(rcl8) },
	{ IEEE80211_MODE_11B,   		X(rcl9) },
	{ IEEE80211_MODE_11A,   		X(rcl4) },
#ifdef ATH_TURBO_SCAN
	{ IEEE80211_MODE_TURBO_STATIC_A,	X(rcl5) },
	{ IEEE80211_MODE_TURBO_STATIC_A,	X(rcl6) },
	{ IEEE80211_MODE_TURBO_A,		X(rcl6x) },
	{ IEEE80211_MODE_TURBO_A,		X(rcl13) },
#endif /* ATH_TURBO_SCAN */
	{ IEEE80211_MODE_11A,			X(rcl7) },
	{ IEEE80211_MODE_11B,			X(rcl10) },
	{ IEEE80211_MODE_11A,			X(rcl11) },
#ifdef ATH_TURBO_SCAN
	{ IEEE80211_MODE_TURBO_G,		X(rcl12) },
#endif /* ATH_TURBO_SCAN */
	{ .list = NULL }
};

static int
checktable(const struct scanlist *scan, const struct ieee80211_channel *c)
{
	int i;

	for (; scan->list != NULL; scan++) {
		for (i = 0; i < scan->count; i++)
			if (scan->list[i] == c->ic_freq) 
				return 1;
	}
	return 0;
}

/*
 * Start a station-mode scan by populating the channel list.
 */
static int
ap_start(struct ieee80211_scan_state *ss, struct ieee80211vap *vap)
{
#define	N(a)	(sizeof(a)/sizeof(a[0]))
	struct ieee80211com *ic = vap->iv_ic;
	struct ap_table *st = ss->ss_priv;
	const struct scanlist *scan;
	enum ieee80211_phymode mode;
	struct ieee80211_channel *c;
	int i;

	ss->ss_last = 0;
	/*
	 * Use the table of ordered channels to construct the list
	 * of channels for scanning.  Any channels in the ordered
	 * list not in the master list will be discarded.
	 */
	for (scan = staScanTable; scan->list != NULL; scan++) {
		mode = scan->mode;
		if (vap->iv_des_mode != IEEE80211_MODE_AUTO) {
			/*
			 * If a desired mode was specified, scan only 
			 * channels that satisfy that constraint.
			 */
			if (vap->iv_des_mode != mode) {
				/*
				 * The scan table marks 2.4Ghz channels as b
				 * so if the desired mode is 11g, then use
				 * the 11b channel list but upgrade the mode.
				 */
				if (vap->iv_des_mode != IEEE80211_MODE_11G ||
				    mode != IEEE80211_MODE_11B)
					continue;
				mode = IEEE80211_MODE_11G;	/* upgrade */
			}
		} else {
			/*
			 * This lets ieee80211_scan_add_channels
			 * upgrade an 11b channel to 11g if available.
			 */
			if (mode == IEEE80211_MODE_11B)
				mode = IEEE80211_MODE_AUTO;
		}
		/* XR does not operate on turbo channels */
		if ((vap->iv_flags & IEEE80211_F_XR) &&
		    (mode == IEEE80211_MODE_TURBO_A ||
		     mode == IEEE80211_MODE_TURBO_G))
			continue;
		/*
		 * Add the list of the channels; any that are not
		 * in the master channel list will be discarded.
		 */
		add_channels(ic, ss, mode, scan->list, scan->count);
	}

	/*
	 * Add the channels from the ic (from HAL) that are not present
	 * in the staScanTable.
	 */
	for (i = 0; i < ic->ic_nchans; i++) {
		c = &ic->ic_channels[i];
		/*
		 * scan dynamic turbo channels in normal mode.
		 */
		if (IEEE80211_IS_CHAN_DTURBO(c))
			continue;
		mode = ieee80211_chan2mode(c);
		if (vap->iv_des_mode != IEEE80211_MODE_AUTO) {
			/*
			 * If a desired mode was specified, scan only 
			 * channels that satisfy that constraint.
			 */
			if (vap->iv_des_mode != mode) 
				continue;
			
		} 
		if (!checktable(staScanTable, c))
			ss->ss_chans[ss->ss_last++] = c;
	}

	ss->ss_next = 0;
	/* XXX tunables */
	ss->ss_mindwell = msecs_to_jiffies(20);		/* 20ms */
	ss->ss_maxdwell = msecs_to_jiffies(200);	/* 200ms */

#ifdef IEEE80211_DEBUG
	if (ieee80211_msg_scan(vap)) {
		printf("%s: scan set ", vap->iv_dev->name);
		ieee80211_scan_dump_channels(ss);
		printf(" dwell min %ld max %ld\n",
			ss->ss_mindwell, ss->ss_maxdwell);
	}
#endif /* IEEE80211_DEBUG */

	st->st_newscan = 1;

	return 0;
#undef N
}

/*
 * Restart a bg scan.
 */
static int
ap_restart(struct ieee80211_scan_state *ss, struct ieee80211vap *vap)
{
	struct ap_table *st = ss->ss_priv;

	st->st_newscan = 1;
	return 0;
}

/*
 * Cancel an ongoing scan.
 */
static int
ap_cancel(struct ieee80211_scan_state *ss, struct ieee80211vap *vap)
{
	struct ap_table *st = ss->ss_priv;
	IEEE80211_CANCEL_TQUEUE(&st->as_actiontq);
	return 0;
}

/*
 * Pick a quiet channel to use for ap operation.
 */
static int
ap_end(struct ieee80211_scan_state *ss, struct ieee80211vap *vap,
       int (*action)(struct ieee80211vap *, const struct ieee80211_scan_entry *),
       u_int32_t flags)
{
	struct ap_table *st = ss->ss_priv;
	struct ieee80211com *ic = vap->iv_ic;
	int i, chan, bestchan, bestchanix;

	KASSERT(vap->iv_opmode == IEEE80211_M_HOSTAP,
		("wrong opmode %u", vap->iv_opmode));
	/* XXX select channel more intelligently, e.g. channel spread, power */
	bestchan = -1;
	bestchanix = 0;		/* NB: silence compiler */
	/* NB: use scan list order to preserve channel preference */
	for (i = 0; i < ss->ss_last; i++) {
		/*
		 * If the channel is unoccupied the max rssi
		 * should be zero; just take it.  Otherwise
		 * track the channel with the lowest rssi and
		 * use that when all channels appear occupied.
		 */
		/*
		 * Check for channel interference, and if found,
		 * skip the channel.  We assume that all channels
		 * will be checked so atleast one can be found
		 * suitable and will change.  IF this changes,
		 * then we must know when we "have to" change
		 * channels for radar and move off.
		 */

		if (ss->ss_chans[i]->ic_flags & IEEE80211_CHAN_RADAR)
			continue;
		if (flags & IEEE80211_SCAN_KEEPMODE) {
			if (ic->ic_curchan != NULL) {
				if ((ss->ss_chans[i]->ic_flags & IEEE80211_CHAN_ALLTURBO) != (ic->ic_curchan->ic_flags & IEEE80211_CHAN_ALLTURBO))
					continue;
			}
		}

		chan = ieee80211_chan2ieee(ic, ss->ss_chans[i]);

		IEEE80211_DPRINTF(vap, IEEE80211_MSG_SCAN,
			"%s: channel %u rssi %d bestchan %d bestchan rssi %d\n",
			__func__, chan, st->as_maxrssi[chan],
			bestchan, bestchan != -1 ? st->as_maxrssi[bestchan] : 0);

		if (st->as_maxrssi[chan] == 0) {
			bestchan = chan;
			bestchanix = i;
			/* XXX use other considerations */
			break;
		}
		if (bestchan == -1 ||
		    st->as_maxrssi[chan] < st->as_maxrssi[bestchan])
			bestchan = chan;
	}
	if (bestchan == -1) {
		/* no suitable channel, should not happen */
		IEEE80211_DPRINTF(vap, IEEE80211_MSG_SCAN,
			"%s: no suitable channel! (should not happen)\n", __func__);
		/* XXX print something? */
		return 0;			/* restart scan */
	} else {
		struct ieee80211_channel *c;
		struct ieee80211_scan_entry se;
		/* XXX notify all vap's? */
		/* if this is a dynamic turbo frequency , start with normal mode first */

		c = ss->ss_chans[bestchanix];
		if (IEEE80211_IS_CHAN_TURBO(c) && !IEEE80211_IS_CHAN_STURBO(c)) { 
			if ((c = ieee80211_find_channel(ic, c->ic_freq, 
				c->ic_flags & ~IEEE80211_CHAN_TURBO)) == NULL) {
				/* should never happen ?? */
				return 0;
			}
		}
		memset(&se, 0, sizeof(se));
		se.se_chan = c;

		st->as_action = ss->ss_ops->scan_default;
		if (action)
			st->as_action = action;
		st->as_selbss = se;
		
		/* 
		 * Must defer action to avoid possible recursive call through 80211
		 * state machine, which would result in recursive locking.
		 */
		IEEE80211_SCHEDULE_TQUEUE(&st->as_actiontq);
		return 1;
	}
}


/*
 * Age entries in the scan cache.
 */
static void
ap_age(struct ieee80211_scan_state *ss)
{
	struct ap_table *st = ss->ss_priv;
	struct ap_entry *se, *next;

	spin_lock(&st->st_lock);
	TAILQ_FOREACH_SAFE(se, &st->st_entry, se_list, next) {
		if (se->se_notseen > AP_PURGE_SCANS) {
			TAILQ_REMOVE(&st->st_entry, se, se_list);
			LIST_REMOVE(se, se_hash);
			if (se->base.se_wme_ie)
				FREE(se->base.se_wme_ie, M_80211_SCAN);
			if (se->base.se_wpa_ie)
				FREE(se->base.se_wpa_ie, M_80211_SCAN);
			if (se->base.se_rsn_ie)
				FREE(se->base.se_rsn_ie, M_80211_SCAN);
			if (se->base.se_ath_ie)
				FREE(se->base.se_ath_ie, M_80211_SCAN);
			if (se->base.se_meraki_ie)
				FREE(se->base.se_meraki_ie, M_80211_SCAN);
			

			FREE(se, M_80211_SCAN);
		}
	}
	spin_unlock(&st->st_lock);
}

/*
 * Iterate over the entries in the scan cache, invoking
 * the callback function on each one.
 */
static void
ap_iterate(struct ieee80211_scan_state *ss, 
	ieee80211_scan_iter_func *f, void *arg)
{
	struct ap_table *st = ss->ss_priv;
	struct ap_entry *se;
	u_int gen;

	spin_lock_bh(&st->st_scanlock);
	gen = st->st_scangen++;
restart:
	spin_lock(&st->st_lock);
	TAILQ_FOREACH(se, &st->st_entry, se_list) {
		if (se->se_scangen != gen) {
			se->se_scangen = gen;
			/* update public state */
			se->base.se_age = jiffies - se->se_lastupdate;
			spin_unlock(&st->st_lock);
			(*f)(arg, &se->base);
			goto restart;
		}
	}
	spin_unlock(&st->st_lock);

	spin_unlock_bh(&st->st_scanlock);
}

static void
ap_assoc_fail(struct ieee80211_scan_state *ss,
	const u_int8_t macaddr[IEEE80211_ADDR_LEN], int reason)
{
}

static void
ap_assoc_success(struct ieee80211_scan_state *ss,
	const u_int8_t macaddr[IEEE80211_ADDR_LEN])
{

}

/*
 * Default action to execute when a scan entry is found for ap
 * mode.  Return 1 on success, 0 on failure
 */
static int
ap_default_action(struct ieee80211vap *vap,
	const struct ieee80211_scan_entry *se)
{
	if (vap->iv_des_chan == IEEE80211_CHAN_ANYC) {
		ieee80211_create_ibss(vap, se->se_chan);
	} else {
		ieee80211_create_ibss(vap, vap->iv_des_chan);
	}
	return 1;
}

static void
action_tasklet(IEEE80211_TQUEUE_ARG data)
{
	struct ieee80211_scan_state *ss = (struct ieee80211_scan_state *)data;
	struct ap_table *st = (struct ap_table *)ss->ss_priv;
	struct ieee80211vap *vap = ss->ss_vap;

	(*ss->ss_ops->scan_default)(vap, &st->as_selbss);
}

static const struct ieee80211_scanner ap_default = {
	.scan_name		= "ap-ap",
	.scan_attach		= ap_attach,
	.scan_detach		= ap_detach,
	.scan_start		= ap_start,
	.scan_restart		= ap_restart,
	.scan_cancel		= ap_cancel,
	.scan_end		= ap_end,
	.scan_flush		= ap_flush,
	.scan_add		= ap_add,
	.scan_probe             = ap_probe,
	.scan_age		= ap_age,
	.scan_iterate		= ap_iterate,
	.scan_assoc_fail	= ap_assoc_fail,
	.scan_assoc_success	= ap_assoc_success,
	.scan_default		= ap_default_action, /* ieee80211_sta_join, */
};

/*
 * Module glue.
 */
MODULE_AUTHOR("John Bicket");
MODULE_DESCRIPTION("802.11 wireless support: default station scanner");
#ifdef MODULE_LICENSE
MODULE_LICENSE("Dual BSD/GPL");
#endif

static int __init
init_scanner_ap(void)
{
	ieee80211_scanner_register(IEEE80211_M_HOSTAP, &ap_default);
	return 0;
}
module_init(init_scanner_ap);

static void __exit
exit_scanner_ap(void)
{
	ieee80211_scanner_unregister_all(&ap_default);
}
module_exit(exit_scanner_ap);
