# Copyright (c) 1996 by the Student Information Processing Board
# 	  of the Massachusetts Institute of Technology
#  
# Permission to use, copy, modify, and distribute this software
# and its documentation for any purpose and without fee is
# hereby granted, provided that the above copyright notice
# appear in all copies and that both that copyright notice and
# this permission notice appear in supporting documentation,
# and that the names of M.I.T. and the M.I.T. S.I.P.B. not be
# used in advertising or publicity pertaining to distribution
# of the software without specific, written prior permission.
# M.I.T. and the M.I.T. S.I.P.B. make no representations about
# the suitability of this software for any purpose.  It is
# provided "as is" without express or implied warranty.
#
#	$Id: questions.sh,v 1.33 1996/06/01 18:39:14 ghudson Exp $
#	$Source: /afs/sipb/project/sipb-athena/src/install/arch/i386_nbsd1/RCS/questions.sh,v $

# /tmp/vars inputs:
#	on_mitnet	whether the machine is on MITnet
# Also relies on files created by hdinfo.

# /tmp/vars outputs
#	disk		disk device containing NetBSD partition
#	start		starting sector of NetBSD partition
#	size		ending sector of NetBSD partition
#	cylinders	number of cylinders on $disk
#	heads		number of tracks per cylinder on $disk
#	sectors		number of sectors per track on $disk
#	create		whether we need to create the partition
#	rootsize	sectors for root subpartition
#	swapsize	sectors for swap subpartition
#	cachesize	sectors for cache subpartition
#	u1size		sectors for u1 subpartition
#	u1cachesize	megabytes for cache if located in /u1 partition.
#	rootpw		encrypted root password
#	enable_toor	whether to enable the toor account
#	rootmail	where to forward root mail
#	subscribe	whether to subscribe to netbsd-announce
#	username	username, if subscribing to netbsd-announce
#	mail		whether to send configuration mail
#	mail_cc		where to cc configuration mail to
#	action		what to do after installing

. /tmp/vars

getresp() {
	read resp
	case "$resp" in
		"")
			resp="$1"
			;;
	esac
}

yesno() {
	while :; do
		read resp
		case "$resp" in
			"")
				resp="$1"
				break
				;;
			y*)
				resp="yes"
				break
				;;
			n*)
				resp="no"
				break
				;;
		esac
		echo -n "Please enter `y' or `n': [$1] "
	done
}

echo ""
echo "Before we do the actual installation, there are some questions to ask"
echo "about the way you want your system installed and configured.  Most of"
echo "the questions have reasonable defaults."
echo ""

# See if the user wants to install on an existing NetBSD partition.
disk=none
if [ -s /tmp/netbsdpart ]; then
	proto_disk=`awk ' { if ($3 > size) { disk = $1; size = $3; } }
			  END { print disk; }' /tmp/netbsdpart`
	echo "You have the following existing NetBSD partitions:"
	echo ""
	awk '{ print $1 ": " $3 / 2048 "MB"; }' /tmp/netbsdpart
	echo ""
	if [ -s /tmp/freespace ]; then
		echo "Type 'none' if you want to create a new partition."
	fi
	while :; do
		echo -n "Use which disk's partition? [$proto_disk] "
		getresp "$proto_disk"; disk="$resp"
		set -- `grep "^$disk" /tmp/netbsdpart`
		if [ "$1" = "$disk" -o \
		     "$disk" = none -a -s /tmp/freespace ]; then
			break
		fi
	done
	if [ "$disk" != none ]; then
		echo ""
		echo "The MBR says that this partition starts at sector $2"
		echo "and is $3 sectors long.  Please confirm:"
		echo -n "What sector does the NetBSD partition start at? [$2] "
		getresp "$2"; start="$resp"
		echo -n "How many sectors long is it? [$3] "
		getresp "$3"; size="$resp"
		create=no
	fi
fi

# If no existing NetBSD partition selected, find out where to create a new one.
if [ "$disk" = none ]; then
	proto_disk=`awk '{ if ($3 > size) { disk=$1; size=$3; }
			   END { print disk; }' /tmp/freespace`
	echo "We need to create a NetBSD partition to install onto.  You have"
	echo "the following areas of free space on your disks:"
	echo ""
	awk '{ print $1 ": " $3 / 2048 "MB at sector " $2 }' /tmp/freespace
	echo ""
	while :; do
		echo -n "Use free space on which disk? [$proto_disk] "
		getresp "$proto_disk"; disk="$resp"
		set -- `grep "^disk" /tmp/netbsdpart`
		if [ "$1" = "$disk" ]; then
			echo -n "You already have a NetBSD partition on that"
			echo " disk; you can't create another one."
			continue
		fi
		set -- `grep "^$disk" /tmp/freespace | sort -nr +3 | head -1`
		if [ "$1" = "$disk" ]; then
			break
		fi
	done
	echo -n "What starting sector? [$2] "
	getresp "$2"; start="$resp"
	echo -n "How many sectors long? [$3] "
	getresp "$3"; size="$resp"
	create=yes
fi

# Look up the geometry of the picked disk.
set -- `grep "^$disk" /tmp/geometries`
if [ "$1" != "$disk" ]; then
	echo "Geometry not found for ${disk}!  This shouldn't happen."
	exit 1
fi
cylinders="$2"
heads="$3"
sectors="$4"

# Get subpartition sizes for root, swap, afs cache, and /u1.
# Reserve 40MB or 97MB for root, swap, and cache depending on total size.
mbytes=`expr "$size" / 2048`
if [ "$mbytes" -lt 150 ]; then
	proto_root=15; proto_swap=14; proto_cache=11
else
	proto_root=20; proto_swap=32; proto_cache=45
fi
echo ""
echo "You have $mbytes megabytes in your NetBSD partition.  You will now be"
echo "asked to give the sizes of a root subpartition, a swap subpartition, an"
echo "optional AFS cache subpartition, and a /u1 subpartition (which will"
echo "contain most of the operating system and should be at least 50MB).  You"
echo "may leave space left over at the end for partitions you wish to create"
echo -n "later.  The default sizes for these partitions are ${proto_root}MB"
echo " for the root"
echo -n "subpartition, ${proto_swap}MB for the swap subpartition,"
echo " ${proto_cache}MB for the AFS cache"
echo "subpartition, and whatever is left for the /u1 subpartition."
echo ""
resp=0
while [ "$resp" -le 0 -o "$resp" -ge "$mbytes" ]; do
	echo -n "Number of megabytes for root subpartition? [$proto_root] "
	getresp "$proto_root"; rootsize=`expr "$resp" "*" 2048`
done
mbytes=`expr "$mbytes" - "$resp"`
resp=0
while [ "$resp" -le 0 -o "$resp" -ge "$mbytes" ]; do
	echo -n "Number of megabytes for swap subpartition? [$proto_swap] "
	getresp "$proto_swap"; swapsize=`expr "$resp" "*" 2048`
done
mbytes=`expr "$mbytes" - "$resp"`
resp=-1
while [ "$resp" -lt 0 -o "$resp" -ge "$mbytes" ]; do
	echo -n "Number of megabytes for cache subpartition (0 for none)?"
	echo -n " [$proto_swap] "
	getresp "$proto_cache"; cachesize=`expr "$resp" "*" 2048`
done
mbytes=`expr "$mbytes" - "$resp"`
resp=0
while [ "$resp" -le 0 -o "$resp" -ge "$mbytes" ]; do
	echo -n "Number of megabytes for /u1 subpartition? [$mbytes]"
	getresp "$proto_cache"; u1size=`expr "$resp" "*" 2048`
done
if [ "$u1size" = "$mbytes" ]; then
	# If we used all the megabytes, tack stray sectors onto /u1.
	u1size=`expr "$size" - "$rootsize" - "$swapsize" - "$cachesize"`
fi

if [ "$cachesize" -eq 0 ]; then
	echo ""
	echo "You have chosen not to create an AFS cache subpartition, so your"
	echo "AFS cache will reside in your /u1 subpartition.  We need to know"
	echo "how much space to use for the AFS cache."
	echo ""
	u1cachesize=0
	while [ "$u1cachesize" -eq 0 ]; do
		echo -n "How large should your cache be, in kilobytes? [5000] "
		getresp 5000
		u1cachesize="$resp"
	done
else
	u1cachesize=5000	# Shouldn't be needed, but just in case.
fi

echo ""
rootpw="`/srvd/install/readpw \
	'Please enter a password for the superuser account (root): '`"

echo ""
echo "The alternate super user account (toor) is like root, but uses /bin/sh"
echo "instead of /bin/csh as its shell.  You may enable the toor account from"
echo "the installation with the same password as root, if desired."
echo ""
echo -n "Would you like to enable the toor account? [yes] "
yesno yes
enable_toor="$resp"

echo ""
echo "Please enter an email address where mail to root@$host.$domain should"
echo "be forwarded to.  Just press enter if you want root's mail to be spooled"
echo "locally in /var/mail/root."
echo ""
echo -n "To what email address should mail to root be forwarded? "
read rootmail

echo ""
echo "SIPB maintains a low-traffic mailing list netbsd-announce@mit.edu for"
echo "announcements about the NetBSD Athena port and NetBSD at MIT in general."
echo "As a convenience, you may get Kerberos tickets and subscribe to"
echo "netbsd-announce from this script.  (You will be asked for your password"
echo "just after you finish answering these questions.)"
echo ""
echo -n "Would you like to subscribe to netbsd-announce? [yes] "
yesno yes; subscribe="$resp"

echo ""
echo "The SIPB would like to track NetBSD installations that used our script;"
echo "as such, we would like to receive automated mail with configuration"
echo "information about your system to help us better account for our"
echo "resources and to see how popular this script is."
echo ""
echo -n "May this script mail your configuration information to the SIPB? "
echo -n "[yes] "
yesno yes; mail="$resp"
if [ "$mail" = "yes" ]; then
	echo ""
	echo "If you would like the above configuration information sent to"
	echo "you as well, enter your username below.  Otherwise, just press"
	echo "return."
	echo ""
	echo -n "Where else would you like the mail sent? "
	read mail_cc
fi

echo ""
echo "Finally, what would you like to have happen when the install finishes?"
echo 'Enter "r" to have your machine reboot automatically, "h" to have your'
echo 'machine halt, or "s" to have your machine exit to a shell prompt.'
echo ""
echo -n "What should your machine do after installing? [reboot] "
action=""
while [ "$action" = "" ]; do
	getresp r
	case "$resp" in
		r*)	action=reboot	;;
		h*)	action=halt	;;
		s*)	action=exit	;;
		*)	echo -n 'Please enter "r", "h", or "s": [reboot] ' ;;
	esac
done

if [ -n "$rootmail" ]; then
	rootmail_disp="$rootmail"
else
	rootmail_disp="<not forwarded>"
fi

if [ "$mail" = yes ]; then
	mail_msg="yes (also to: $mail_cc)"
else
	mail_msg=no
fi

# Report $u1cachesize
cat << EOM
Okay, please confirm your answers to these questions:

Disk to install on:			$disk
Create a new partition:			$create
Starting sector of partition:		$start
Number of sectors in partition:		$size
Root subpartition size:			`expr $rootsize / 2048`MB
Swap subpartition size:			`expr $swapsize / 2048`MB
Cache subpartition size:		`expr $cachesize / 2048`MB
/u1 subpartition size:			`expr $u1size / 2048`MB
EOM
if [ "$cachesize" -eq 0 ]; then
	echo "Cache size on /u1			${u1cachesize}KB"
fi
cat << EOM
toor account enabled:			$enable_toor
Root mail forwards to:			$rootmail_disp
Subscribe to netbsd-announce:		$subscribe
Mail configuration to SIPB:		$mail_msg
Action after installation:		$action

EOM
echo -n "Do you accept these values? [yes] "

getresp yes
case "$resp" in
y)
	;;
n)
	echo "Okay, trying again."
	exec /srvd/install/questions
esac

if [ "$subscribe" = yes ]; then
	echo "To subscribe to the netbsd-announce mailing list, you need to"
	echo "get Kerberos tickets."
	echo ""
	username=
	while [ "$username" = "" ]; do
		echo -n "What is your username? "
		read username
	done

	trap "break" 2
	while [ ! -f "$KRBTKFILE" ]; do
		/etc/athena/gettime -s time > /dev/null
		kinit "$username"
	done
	trap 2

	if [ ! -f "$KRBTKFILE" ]; then
		echo "Since you didn't get Kerberos tickets, you will not be"
		echo "subscribed to netbsd-announce."
	fi
fi

cat >> /tmp/vars << EOM
disk="$disk"
start="$start"
size="$size"
cylinders="$cylinders"
heads="$heads"
sectors="$sectors"
create="$create"
rootsize="$rootsize"
swapsize="$swapsize"
cachesize="$cachesize"
u1size="$u1size"
u1cachesize="$u1cachesize"
rootpw="$rootpw"
enable_toor="$enable_toor"
rootmail="$rootmail"
subscribe="$subscribe"
username="$username"
mail="$mail"
mail_cc="$mail_cc"
action="$action"
EOM

if [ "$athena_inst" = "yes" ]; then
	echo ""
	echo "Please press return and read the following license statement"
	echo "about the use of the AFS on your system."
	read dummy
	more /MIT_COPYRIGHT
fi
