/**
 ** growbuf.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/

/**
 **  growbuf.c
 **
 ** v1.0  Fri May 17 11:02:00 1991  Randy Sargent.  created
 ** v1.1  Sun May 26 00:39:30 1991  Randy Sargent.  added growbuf_length
 ** v1.2  Fri May 31 23:11:30 1991  Randy Sargent.  added growbuf_insert, delete
 **/

#include CONFIG

#include "growbuf.h"

#define GROWBUF_ADD      10
/* If the following is rational fraction, do not parenthesize it! */
#define GROWBUF_FACTOR   3/2
#define NEW_GROWBUF_SIZE(old_size) ((old_size)*GROWBUF_FACTOR+GROWBUF_ADD)

/* 0 if successful */
#ifdef MALLOC_DEBUG
Int dmalloc_growbuf_init(Growbuf *g,const char *file,Int line)
{
    Int size= NEW_GROWBUF_SIZE(0);
    g->buf= debug_malloc(size,file,line);
    if (!g->buf) return 1;
    g->bufend= g->buf + size;
    g->bufptr= g->buf;
    return 0;
}
#else
Int growbuf_init(Growbuf *g)
{
    Int size= NEW_GROWBUF_SIZE(0);
    g->buf= malloc(size);
    if (!g->buf) return 1;
    g->bufend= g->buf + size;
    g->bufptr= g->buf;
    return 0;
}
#endif

void growbuf_term(Growbuf *g)
{
    free(g->buf);
    g->buf= g->bufptr= g->bufend= 0;
}

Int growbuf__grow(Growbuf *g)
{
    Int newsize= NEW_GROWBUF_SIZE(g->bufend - g->buf);
    char *newbuf;

    if(g->buf)
      newbuf= realloc(g->buf, newsize);
    else
      newbuf= malloc(newsize);
      
    if (!newbuf) return 1;
    g->bufptr = newbuf + (g->bufptr - g->buf);
    g->buf= newbuf;
    g->bufend= g->buf + newsize;
    return 0;
}
    
/* 0 if successful */
Int growbuf_add_char(Growbuf *g, Int c)
{
    if (g->bufptr == g->bufend) {
	if (growbuf__grow(g)) return 1;
    }
    *g->bufptr++= (char) c;
    return 0;
}

int growbuf_remove_char(Growbuf *g)
{
   /* return *(--(Char *)g->bufptr); */
   int ret;
   char *ptr= (char*) g->bufptr;
   ret= (int)*--ptr;
   g->bufptr= (char*) ptr;
   return ret;
}

int growbuf_last_char(Growbuf *g)
{
   int ret;
   char *ptr= (char*) g->bufptr;
   ret= (int)ptr[-1];
   return ret;
}

/* 0 if successful */
Int growbuf_add(Growbuf *g, void *data, Int len)
{
    while ((g->bufend - g->bufptr) < len) {
	if (growbuf__grow(g)) return 1;
    }
    memcpy(g->bufptr, data, len);
    g->bufptr += len;
    return 0;
}

/* 0 if successful */
Int growbuf_remove(Growbuf *g, void *data, Int len)
{
   char *ptr= g->bufptr;
   ptr -= len;
   memcpy(data, ptr, len);
   g->bufptr= ptr;
   return 0;
}

/* 0 if successful */
Int growbuf_last(Growbuf *g, void *data, Int len)
{
   char *ptr= g->bufptr;
   ptr -= len;
   memcpy(data, ptr, len);
   return 0;
}


Int growbuf_add_int(Growbuf *g, Int i)
{
    while ((g->bufend - g->bufptr) < sizeof(Int)) {
	if (growbuf__grow(g)) return 1;
    }
    {
         /*   *((Int*)g->bufptr)++= i; */
    	Int *ptr= (Int*) g->bufptr;
    	*ptr++ = i;
    	g->bufptr= (char*) ptr;
    }
    return 0;
}

Int growbuf_remove_int(Growbuf *g)
{
    /* return *(--(Int *)g->bufptr); */
    Int ret;
    Int *ptr= (Int*) g->bufptr;
   	ret= *--ptr;
  	g->bufptr= (char*) ptr;
  	return ret;
}

Int growbuf_last_int(Growbuf *g)
{
   Int ret;
   Int *ptr= (Int*) g->bufptr;
   ret= ptr[-1];
   return ret;
}

Int growbuf_add_short(Growbuf *g, short s)
{
    while ((g->bufend - g->bufptr) < sizeof(short)) {
	if (growbuf__grow(g)) return 1;
    }
    {
         /*   *((short*)g->bufptr)++= s; */
    	short *ptr= (short*) g->bufptr;
    	*ptr++ = s;
    	g->bufptr= (char*) ptr;
    }
    return 0;
}

Int growbuf_add_long(Growbuf *g, long l)
{
    while ((g->bufend - g->bufptr) < sizeof(long)) {
	if (growbuf__grow(g)) return 1;
    }
    {
         /*   *((long*)g->bufptr)++= l; */
    	long *ptr= (long*) g->bufptr;
    	*ptr++ = l;
    	g->bufptr= (char*) ptr;
    }
    return 0;
}

Int growbuf_add_double(Growbuf *g, double d)
{
    while ((g->bufend - g->bufptr) < sizeof(double)) {
	if (growbuf__grow(g)) return 1;
    }
    {
       double *ptr= (double*) g->bufptr;
       *ptr++ = d;
       g->bufptr= (char*) ptr;
    }
    return 0;
}

long growbuf_last_long(Growbuf *g)
{
    /* return ((long*)g->bufptr)[-1]; */
    long ret;
    long *ptr= (long*) g->bufptr;
   	ret= ptr[-1];
  	return ret;
}

long growbuf_remove_long(Growbuf *g)
{
    /* return *(--(long*)g->bufptr); */
    long ret;
    long *ptr= (long*) g->bufptr;
   	ret= *--ptr;
  	g->bufptr= (char*) ptr;
  	return ret;
}

Int growbuf_add_float(Growbuf *g, float f)
{
    while ((g->bufend - g->bufptr) < sizeof(float)) {
	if (growbuf__grow(g)) return 1;
    }
    
    {
         /*   *((float*)g->bufptr)++= f; */
    	float *ptr= (float*) g->bufptr;
    	*ptr++ = f;
    	g->bufptr= (char*) ptr;
    }
    return 0;
}

Int growbuf_add_ptr(Growbuf *g, void *p)
{
    while ((g->bufend - g->bufptr) < sizeof(void *)) {
	if (growbuf__grow(g)) return 1;
    }
    
    {
         /*   *((void**)g->bufptr)++= ptr; */
    	void **ptr= (void**) g->bufptr;
    	*ptr++ = p;
    	g->bufptr= (char*) ptr;
    }
    return 0;
}

void *growbuf_remove_ptr(Growbuf *g)
{
    /* return *(--(void **)g->bufptr); */
    void *ret;
    void **ptr= (void**) g->bufptr;
   	ret= *--ptr;
  	g->bufptr= (char*) ptr;
  	return ret;
    
}
void *growbuf_last_ptr(Growbuf *g)
{
   void *ret;
   void **ptr= (void **) g->bufptr;
   	ret= ptr[-1];
  	return ret;
}

char *growbuf_data(Growbuf *g)
{
    return g->buf;
}

void growbuf_clear(Growbuf *g)
{
    g->bufptr= g->buf;
}

/* Copies into smaller form (no empty space at end) */
/* 0 if successful */
Int growbuf_init_from_prototype(Growbuf *dest, Growbuf *src)
{
    Int size_src= src->bufptr - src->buf;
    dest->buf= malloc(size_src);
    if (!dest->buf) return 1;
    memcpy(dest->buf, src->buf, size_src);
    dest->bufend= dest->bufptr= dest->buf + size_src;
    
    return 0;
}

Int growbuf_copy(Growbuf *dest, Growbuf *src)
{
    Int max_size_dest= dest->bufend - dest->buf;
    Int size_src= src->bufptr - src->buf;

    if (size_src > max_size_dest) {
	char *newbuf= realloc(dest->buf, size_src);
	if (!newbuf) return 1;
	dest->buf= newbuf;
    }
    memcpy(dest->buf, src->buf, size_src);
    dest->bufend= dest->bufptr= dest->buf + size_src;

    return 0;
}

Int growbuf_trim(Growbuf *g)
{
    Int size= g->bufptr - g->buf;
    char *newbuf= realloc(g->buf, size);

    if (!newbuf) return 1;
    g->buf= newbuf;

    return 0;
}

/* 0 if successful */
Int growbuf_insert_char(Growbuf *g, Int pos, Int c)
{
    Int size= g->bufptr - g->buf;

    if (pos > size) return 2; /* out of bounds */

    if (g->bufptr == g->bufend) {
	if (growbuf__grow(g)) return 1; /* no more memory */
    }

    if (size - pos) memmove(g->buf + pos + 1, g->buf + pos, size - pos);

    g->buf[pos]= (char) c;
    g->bufptr++;
    return 0;
}

Int growbuf_insert(Growbuf *g, Int pos, void *data, Int len)
{
    Int size= g->bufptr - g->buf;

    if (pos > size) return 2; /* out of bounds */

    while (g->bufend - g->bufptr < len) {
	if (growbuf__grow(g)) return 1; /* no more memory */
    }

    if (size - pos) memmove(g->buf + pos + len, g->buf + pos, size - pos);

    memcpy(g->buf + pos, data, len);
    g->bufptr += len;
    return 0;
}


/* 0 if successful */
Int growbuf_delete_char(Growbuf *g, Int pos)
{
    Int size= g->bufptr - g->buf;

    if (pos >= size) return 2; /* out of bounds */

    if (size - pos - 1) memmove(g->buf + pos, g->buf + pos + 1, size - pos - 1);

    g->bufptr--;
    return 0;
}

Int growbuf_get_char(Growbuf *g, Int pos)
{
    Int size= g->bufptr - g->buf;

    if (pos >= size) return -1; /* out of bounds */

    return g->buf[pos];
}
    
Int growbuf_length(Growbuf *g)
{
    return GROWBUF_LENGTH(g);
}

/* 0 if successful.  1 if no instances found */
/* Assumes only Ints have been put in growbuf */

Int growbuf_search_delete_int(Growbuf *g, Int delete_me)
{
   Int *from, *to;
   Int found= 0;
   from= to= (Int*) (g->buf);
   
   while (from < (Int*) (g->bufptr)) {
      if (*from == delete_me) {
	 found++;
	 from++;
      }
      else {
	 *to++= *from++;
      }
   }
   /* ((void**)(g->bufptr)) -= found; */
   g->bufptr -= (found * sizeof(Int));
   return !found;
}

Int growbuf_search_delete_ptr(Growbuf *g, void *delete_me)
{
    void **from, **to;
    Int found= 0;
    from= to= (void**) (g->buf);

    while (from < (void**) (g->bufptr)) {
	if (*from == delete_me) {
	    found++;
	    from++;
	}
	else {
	    *to++= *from++;
	}
    }
    /* ((void**)(g->bufptr)) -= found; */
    g->bufptr -= (found * sizeof(void*));
    return !found;
}

Int growbuf_search_ptr(Growbuf *g, void *ptr)
{
    void **from;
    from= (void**) (g->buf);
    
    while (from < (void**) (g->bufptr)) {
       if (*from == ptr) {
	  return(1);
       }
       from++;
    }
    return(0);
}

Int growbuf_find_ptr(Growbuf *g, void *ptr)
{
   int i=0;
   void **from;
   from= (void**) (g->buf);
   
   while (from < (void**) (g->bufptr)) {
      if (*from == ptr) {
	 return(i);
      }
      from++;
      i++;
    }
    return(-1);
}


	    
	    
	
	    
	    
	    
