/**
 ** table.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/

/**
 ** table.c
 **
 ** Associative table, implemented as a sorted vector
 ** search time O(log N), insertion/deletion time O(N)
 **
 ** v1.0  Thu May 16 10:52:47 1991 Randy Sargent created
 ** v1.1  Fri Aug  9 1991 Randy Sargent
 **             added table_remove, table_get_key_and_value, table_display
 ** v1.2  Tue Oct 15 09:10:50 1991 Randy Sargent
 **             changed table_get_next to not get key or value if respective
 **             pointer is NULL.
 **/

#include CONFIG
#include "util.h"

#include "table.h"

#define TABLE_GROW_ADD      10
/* If the following is rational fraction, do not parenthesize it! */
#define TABLE_GROW_FACTOR   3/2
#define NEW_TABLE_SIZE(old_size) ((old_size)*TABLE_GROW_FACTOR+TABLE_GROW_ADD)

#define TABLE_KEY(t, index)   ( (t)->elems + (index)*(t)->elem_size )
#define TABLE_VALUE(t, index) (TABLE_KEY(t, index) + (t)->key_size)

Int table_init(Table *t, Int key_size, Int value_size, Int (*compare)(void*,void*))
{
    t->num_elems= 0;
    t->max_elems= NEW_TABLE_SIZE(0);
    t->key_size= key_size;
    t->elem_size= key_size + value_size;
    t->compare= compare;
    t->elems= malloc(t->elem_size * t->max_elems);
    if (t->elems) return 0; else return 1;
}

void table_display(Table *t, void (*print_key)(void*), void (*print_value)(void*))
{
    TableIter i;
    void *key, *value;
    table_rewind(t, &i);
    printf("Table <%lx>: %ld items\n", (long)t, t->num_elems);
    printf("Room for %ld items before enlargement is necessary\n", t->max_elems);
    if (print_key || print_value) {
	printf("Key: Value\n");
	while (table_get_next(t, &i, &key, &value)) {
	    if (print_key) (*print_key)(key);
	    printf(": ");
	    if (print_value) (*print_value)(value);
	    printf("\n");
	}
    }
}

void table_rewind(Table *t, TableIter *i)
{
    UNUSED(t);
    *i= 0;
}

Int table_get_next(Table *t, TableIter *i, void *key, void *value)
{
    if (*i >= t->num_elems) return 0;
    if (key) *(void**)key=   TABLE_KEY(t, *i);
    if (value) *(void**)value= TABLE_VALUE(t, *i);
    (*i)++;
    return 1;
}
    
Int table__elem(Table *t, void *key, Int *index)
{
    Int low= 0;
    Int high= t->num_elems - 1;

    while (low <= high) {
	Int check= (low + high) / 2;
	Int compare;

	compare= (t->compare) (key, TABLE_KEY(t, check));

	if (compare < 0) {
	    high = check - 1;
	}
	else if (compare > 0) {
	    low= check + 1;
	}
	else {
	    *index= check;
	    return 1;
	}
    }
    *index= low;
    return 0;
}

void *table_get(Table *t, void *key)
{
    Int index;
    if (table__elem(t, key, &index))
      return TABLE_VALUE(t, index);
    else
      return 0;
}

/* 0 if success */

/* get key and value less than or equal to input key */
Int table_get_key_and_value_lteq(Table *t, void *in_key, void *out_keyptr, void *out_valueptr)
{
    Int index;
    if (table__elem(t, in_key, &index)) {
	index++;
    }
    if (index > 0) {
	index--;
	*(void**)out_keyptr= TABLE_KEY(t, index);
	*(void**)out_valueptr= TABLE_VALUE(t, index);
	return 0;
    }
    else
      return 1;
}

/* get key and value greater than or equal to input key */
Int table_get_key_and_value_gteq(Table *t, void *in_key, void *out_keyptr, void *out_valueptr)
{
    Int index;
    table__elem(t, in_key, &index);
    if (index < t->num_elems) {
	index--;
	*(void**)out_keyptr= TABLE_KEY(t, index);
	*(void**)out_valueptr= TABLE_VALUE(t, index);
	return 0;
    }
    else
      return 1;
}

/* 0 if success */

Int table_get_key_and_value(Table *t, void *in_key, void *out_keyptr, void *out_valueptr)
{
    Int index;
    if (table__elem(t, in_key, &index)) {
	*(void**)out_keyptr= TABLE_KEY(t, index);
	*(void**)out_valueptr= TABLE_VALUE(t, index);
	return 0;
    }
    else
      return 1;
}

void table_term(Table *t)
{
    free(t->elems);
    t->elems= 0;
}

/* Sets table(key)= value.  If key is not present in table, inserts the binding.
   If key already has a binding, that binding is deleted and a new binding
   consisting of key,value is inserted.  Note that means the NEW key is written
   into the table (in case the compare function says two keys are "equal" which
   do not have identical representations.
   This function copies the key and the value into the table;  pointers to these
   copies will be returned by table_get */

Int table_set(Table *t, void *key, void *value)
{
    Int index;
    if (!table__elem(t, key, &index)) {
	/* Not found; must insert the element here */
	if (t->num_elems == t->max_elems) {
	    /* Must grow table */
	    t->max_elems= NEW_TABLE_SIZE(t->max_elems);
	    t->elems= realloc(t->elems, t->max_elems * t->elem_size);
	    if (!t->elems) return 1;
	}
	memmove(TABLE_KEY(t, index+1), TABLE_KEY(t, index),
		(t->num_elems - index) * t->elem_size);
	t->num_elems++;
    }
    memcpy(TABLE_KEY(t, index),   key,   t->key_size);
    memcpy(TABLE_VALUE(t, index), value, t->elem_size - t->key_size);
    return 0;
}

/* 0 if success */
Int table_remove(Table *t, void *key)
{
    Int index;
    if (!table__elem(t, key, &index)) {
	/* Not found */
	return 1;
    }
    memmove(TABLE_KEY(t, index), TABLE_KEY(t, index+1),
	    (t->num_elems - index - 1) * t->elem_size);
    t->num_elems--;
    /* TODO: could shrink space allocated to table when table usage %
       goes below a threshold */
    return 0;
}


Int table_strcmp(void *a, void *b)
{
    return strcmp(*(char**)a, *(char**)b);
}

Int table_stricmp(void *a, void *b)
{
    return stricmp(*(char**)a, *(char**)b);
}

Int table_intcmp(void *a, void *b)
{
    return (*(Int*)a) - (*(Int*)b);
}

Int table_doublecmp(void *a, void *b)
{
    if ((*(double*)a) > (*(double*)b)) return 1;
    if ((*(double*)a) < (*(double*)b)) return -1;
    return 0;
}

Int table_longcmp(void *a, void *b)
{
    if ((*(long*)a) < (*(long*)b)) return -1;
    if ((*(long*)a) > (*(long*)b)) return 1;
    return 0;
}

Int table_ptrcmp(void *a, void *b)
{
    if ((*(void**)a) < (*(void**)b)) return -1;
    if ((*(void**)a) > (*(void**)b)) return 1;
    return 0;
}

void table_print_int(void *a)
{
    printf("%ld", *(Int*)a);
}

void table_print_str(void *a)
{
    printf("%s", *(char**) a);
}
