#include CONFIG
#include <util.h>
#include <stream.h>
#include <stringlb.h>

#include <ctype.h>

#include "binrec.h"
#include "ihex.h"

Int ihex_debug= 0;

Int ihex_get_hex1(Stream *s)
{
    Int ch= STREAM_GETC(s);
    if ('0' <= ch && ch <= '9') return ch - '0';
    if ('a' <= ch && ch <= 'f') return ch - 'a' + 10;
    if ('A' <= ch && ch <= 'F') return ch - 'A' + 10;
    die(("Illegal hex digit in .hex file >%c<\n", (int)ch));
    return -1;
}

Int ihex_get_hex2(Stream *s)
{
    Int a= ihex_get_hex1(s);
    Int b= ihex_get_hex1(s);

    return (a<<4) + b;
}
    
Int ihex_get_hex4(Stream *s)
{
    Int a= ihex_get_hex2(s);
    Int b= ihex_get_hex2(s);

    return (a<<8) + b;
}

long ihex_get_hex8(Stream *s)
{
    long a= ihex_get_hex4(s);
    long b= ihex_get_hex4(s);

    return (a<<16) + b;
}

Binrec *read_contiguous_ihex_file(FILE *in)
{
    Binrec *ret;
    Stream s;
    stream_init_from_file(&s, in);
    ret=read_contiguous_ihex_stream(&s);
    stream_term(&s);
    return ret;
}

Binrec *read_contiguous_ihex_string(char *str)
{
    Binrec *ret;
    Stream s;
    stream_init_from_string(&s, str);
    ret=read_contiguous_ihex_stream(&s);
    stream_term(&s);
    return ret;
}

Binrec *read_contiguous_ihex_stream(Stream *s)
{
    Binrec *ret= binrec_create(0,0);
    if (ihex_debug) printf("Starting read contiguous\n");
    while (1) {
	Binrec *new= read_ihex_record_from_stream(s);
	Binrec *newret;
	if (!new) return ret;
	if (ihex_debug) printf("Got record\n");
	if (!binrec_can_concat(ret, new))
	  die((".hex file was not contiguous\n"));
	newret= binrec_concat(ret, new);
	binrec_destroy(ret);
	binrec_destroy(new);
	ret= newret;
    }
}

char *unparse_ihex_record(Binrec *rec)
{
    char *buf;
    long i;
    Int checksum= 0;
    buf= malloc((size_t) (rec->len * 2 + 30));
    buf[0]= 0;
    sprintf(buf + strlen(buf), ":%02lX%04lX00", rec->len + 3, rec->addr);
    for (i= 0; i< rec->len; i++)
      sprintf(buf + strlen(buf), "%02X", rec->data[i]);
    sprintf(buf + strlen(buf), "%02lX", checksum); /* sorry, this is bogus */
    sprintf(buf + strlen(buf), "\n");
    return buf;
}

int ihex_csum_long(long i)
{
    return
      ((i >> 0) & 0xff) +
	((i >> 8) & 0xff) +
	  ((i >> 16) & 0xff) +
	    ((i >> 24) & 0xff);
}

char *ihex_header()
{
   return string_copy("");
}

char *ihex_trailer()
{
   return string_copy(":00000001FF");
}

char *data_to_ihex(unsigned char *data, long addr, long len)
{
    char *buf;
    long i;
    Int checksum= 0;
    buf= malloc((size_t) (len * 2 + 30));
    buf[0]= 0;
    sprintf(buf + strlen(buf), ":%02lX%04lX00", len , addr);
    checksum += len + 5;
    checksum += ihex_csum_long(addr);
    for (i= 0; i< len; i++) {
       sprintf(buf + strlen(buf), "%02X", data[i]);
       checksum += data[i];
    }
    sprintf(buf + strlen(buf), "%02lX", (-checksum) & 0xff);
    sprintf(buf + strlen(buf), "\n");
    return buf;
}

Binrec *read_ihex_record_from_stream(Stream *s)
{
   Int c;
   /* Skip whitespace */
   while (1) {
      c= STREAM_GETC(s);
      if (!isspace(c)) break;
      if (ihex_debug) {printf("%ld,",c);}
      
   }
   if (c == EOF) {
      if (ihex_debug) {printf("EOF in read_ihex record\n");}
      return NULL;
   }
   if (c != ':') die(("Illegal line in .hex file"));
    
   {
      Binrec *ret;
      Int i, nbytes, addr,type;
      
      nbytes= ihex_get_hex2(s);
      addr=  ihex_get_hex4(s);
      type=  ihex_get_hex2(s);
      if(type!=0) {
	 if (ihex_debug) printf("got end of file marker\n");
	 return(NULL);
      }
      
      ret= binrec_create(addr, nbytes);
      
      for (i= 0; i< nbytes; i++) {
	 ret->data[i]= (char) ihex_get_hex2(s);
      }
      
      ihex_get_hex2(s);
      while (1) {
	 c= STREAM_GETC(s);
	 if (c == '\n' || c == '\r') {
	    nuke_cr(s);
	    break;
	 }
	 if (!isspace(c))
	   die(("Extra bytes at end of .hex record"));
      }
      
      if (ihex_debug) printf("got rec: addr:%lX len:%ld\n", addr, nbytes);
      return ret;
   }
   return 0;
}
	    
