/**
 ** tablen.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/

/**
 ** tablen.c
 **
 ** Associative tablen, implemented as a sorted vector
 ** search time O(log N), insertion/deletion time O(N)
 **
 ** adapted from tablen.c.  simpler calling interface,
 ** less overhead, but only ability to hold int-size objects
 **/


#include CONFIG

#include "util.h"

#include "tablen.h"

#define TABLEN_GROW_ADD      10
/* If the following is rational fraction, do not parenthesize it! */
#define TABLEN_GROW_FACTOR   3/2
#define NEW_TABLEN_SIZE(old_size) ((old_size)*TABLEN_GROW_FACTOR+TABLEN_GROW_ADD)

#define TABLEN_KEY(t, index)   ( (t)->elems + (index)*(t)->elem_size )
#define TABLEN_VALUE(t, index) (TABLEN_KEY(t, index) + (t)->key_size)

#define HPRIMES (sizeof(tablen_hashing_primes)/(sizeof(int)))
const int tablen_hashing_primes[]={3,5,7,11,13,17,19,23,29,31,37,41,
			      43,47,51,53,57,59};

int tablen_debug=0;

int tablen_hash2(int hash1)
{
   return(tablen_hashing_primes[hash1%HPRIMES]);
}

void tablen_clear(Tablen *t)
{
   t->num_elems= 0;
}

Int tablen_init(Tablen *t)
{
   t->num_elems= 0;
   t->max_elems= NEW_TABLEN_SIZE(0);
   t->elems= malloc(sizeof(Tablen_elem) * t->max_elems);
   if (t->elems) return 0; else return 1;
}

void tabless_display(Tablen *t)
{
   
void tablen_display(Tablen *t, void (*print_key)(void*), void (*print_value)(void*))
{
    TablenIter i;
    void *key, *value;
    tablen_rewind(t, &i);
    printf("Tablen <%lx>: %ld items\n", (long)t, t->num_elems);
    printf("Room for %ld items before enlargement is necessary\n", t->max_elems);
    if (print_key || print_value) {
	printf("Key: Value\n");
	while (tablen_get_next(t, &i, &key, &value)) {
	    if (print_key) (*print_key)(key);
	    printf(": ");
	    if (print_value) (*print_value)(value);
	    printf("\n");
	}
    }
}

void tablen_rewind(Tablen *t, TablenIter *i)
{
   UNUSED(t);
   *i= -1;
}

Int tablen_get_next(Tablen *t, TablenIter *i, void *key, void *value)
{
   (*i)++;
   if(t->type==hash_tab) {
      /* go until a not-empty cell is found */
      for(;(*i)<t->max_elems;(*i)++) {
	 if(!(t->is_empty(TABLEN_KEY(t,*i)) || t->is_deleted(TABLEN_KEY(t,*i))))
	   break;
      }
      return(0);			/* hit the end */
   }
   else {
      if (*i >= t->num_elems) return 0;
   }
   if (key) *(void**)key=   TABLEN_KEY(t, *i);
   if (value) *(void**)value= TABLEN_VALUE(t, *i);
   return 1;
}
    
inline Int tablen__elem(Tablen *t, void *key, Int *index)
{
   if(t->type==binsearch_tab) {
      return(tablen__binsearch_elem(t,key,index));
   }
   else if(t->type==hash_tab) {
      return(tablen__hash_elem(t,key,index));
   }
   return(1);
}
Int tablen__binsearch_elem(Tablen *t, void *key, Int *index)
{
    Int low= 0;
    Int high= t->num_elems - 1;

    while (low <= high) {
	Int check= (low + high) / 2;
	Int compare;

	compare= (t->compare) (key, TABLEN_KEY(t, check));

	if (compare < 0) {
	    high = check - 1;
	}
	else if (compare > 0) {
	    low= check + 1;
	}
	else {
	    *index= check;
	    return 1;
	}
    }
    *index= low;
    return 0;
}

Int tablen__hash_elem(Tablen *t, void *look, Int *index)
{
   int i,fdel=-1;
   int h,h1,h2;
   void *e;

   h1=t->hash(look);
   if(h1<0) h1=-h1;
   
   h2=tablen_hash2(h1);

   for(i=0,h=h1%t->max_elems,e=TABLEN_KEY(t,h);
       i<t->max_elems;
       i++,h=(h+h2)%t->max_elems,e=TABLEN_KEY(t,h)) {
      if(t->is_empty(e)) {			/* found an empty slot! */
	 if(fdel!=-1)
	   *index=fdel;
	 else
	   *index=h;
	 return(0);
      }
      else if(t->is_deleted(e))	{/* might be later, keep going */
	 if(fdel==-1) fdel=h;
	 continue;
      }
      else if(t->compare(e,look)==0){	/* found match */
	 *index=h;
	 return(1);
      }
   }
   ASSERT(0);
}

void *tablen_get(Tablen *t, void *key)
{
    Int index;
    if (tablen__elem(t, key, &index))
      return TABLEN_VALUE(t, index);
    else
      return 0;
}

/* 0 if success */

/* get key and value less than or equal to input key */
Int tablen_get_key_and_value_lteq(Tablen *t, void *in_key, void *out_keyptr, void *out_valueptr)
{
    Int index;
    if (tablen__elem(t, in_key, &index)) {
	index++;
    }
    if (index > 0) {
	index--;
	*(void**)out_keyptr= TABLEN_KEY(t, index);
	*(void**)out_valueptr= TABLEN_VALUE(t, index);
	return 0;
    }
    else
      return 1;
}

/* get key and value greater than or equal to input key */
Int tablen_get_key_and_value_gteq(Tablen *t, void *in_key, void *out_keyptr, void *out_valueptr)
{
    Int index;
    tablen__elem(t, in_key, &index);
    if (index < t->num_elems) {
	index--;
	*(void**)out_keyptr= TABLEN_KEY(t, index);
	*(void**)out_valueptr= TABLEN_VALUE(t, index);
	return 0;
    }
    else
      return 1;
}

/* 0 if success */

Int tablen_get_key_and_value(Tablen *t, void *in_key, void *out_keyptr, void *out_valueptr)
{
    Int index;
    if (tablen__elem(t, in_key, &index)) {
	*(void**)out_keyptr= TABLEN_KEY(t, index);
	*(void**)out_valueptr= TABLEN_VALUE(t, index);
	return 0;
    }
    else
      return 1;
}

void tablen_term(Tablen *t)
{
    free(t->elems);
    t->elems= 0;
}

/* Sets tablen(key)= value.  If key is not present in tablen, inserts the binding.
   If key already has a binding, that binding is deleted and a new binding
   consisting of key,value is inserted.  Note that means the NEW key is written
   into the tablen (in case the compare function says two keys are "equal" which
   do not have identical representations.
   This function copies the key and the value into the tablen;  pointers to these
   copies will be returned by tablen_get */

Int tablen_set(Tablen *t, void *key, void *value)
{
    Int index;
    if (!tablen__elem(t, key, &index)) {
	/* Not found; must insert the element here */
	if (t->num_elems == t->max_elems) {
	    /* Must grow tablen */
	    t->max_elems= NEW_TABLEN_SIZE(t->max_elems);
	    t->elems= realloc(t->elems, t->max_elems * sizeof(Tablen_elem));
	    if (!t->elems) return 1;
	}
	memmove(TABLEN_KEY(t, index+1), TABLEN_KEY(t, index),
		(t->num_elems - index) * sizeof(Tablen_elem));
	t->num_elems++;
    }
    memcpy(TABLEN_KEY(t, index),   key,   t->key_size);
    memcpy(TABLEN_VALUE(t, index), value, sizeof(Tablen_elem) - t->key_size);
    return 0;
}

Int tablen_hash_set(Tablen *t, void *key, void *value)
{
   Int index;

   if (t->num_elems > t->max_elems/2) {
      char *e,*old_elems=t->elems;
      int i,old_max=t->max_elems;
      
      /* Must grow tablen */
      t->max_elems= old_max*2;
      t->elems= malloc(t->max_elems * sizeof(Tablen_elem));
      if (!t->elems) return 1;
      tablen_clear(t);
      
      /* insert old elements into new tablen */
      for(i=0,e=old_elems;i<old_max;i++,e+=sizeof(Tablen_elem)) {
	 if(!t->is_empty(e) && !t->is_deleted(e)) {
	    tablen__elem(t, e, &index);
	    t->num_elems++;
	    memcpy(TABLEN_KEY(t, index),   e,   sizeof(Tablen_elem));
	 }
      }
      free(old_elems);		/* free old buffer */
   }
   
   if (!tablen__elem(t, key, &index)) {
      /* Not found; must insert the element here */
      t->num_elems++;
    }
   memcpy(TABLEN_KEY(t, index),   key,   t->key_size);
   memcpy(TABLEN_VALUE(t, index), value, sizeof(Tablen_elem) - t->key_size);
   return 0;
}

/* 0 if success */
Int tablen_remove(Tablen *t, void *key)
{
    Int index;
    if (!tablen__elem(t, key, &index)) {
	/* Not found */
	return 1;
    }
    memmove(TABLEN_KEY(t, index), TABLEN_KEY(t, index+1),
	    (t->num_elems - index - 1) * sizeof(Tablen_elem));
    t->num_elems--;
    /* TODO: could shrink space allocated to tablen when tablen usage %
       goes below a threshold */
    return 0;
}

Int tablen_hash_remove(Tablen *t, void *key)
{
   Int index;
   if (!tablen__elem(t, key, &index)) {
      /* Not found */
      return 1;
   }
   t->make_deleted(TABLEN_KEY(t, index));
   t->num_elems--;
   return 0;
}

Int tablen_strcmp(void *a, void *b)
{
    return strcmp(*(char**)a, *(char**)b);
}

Int tablen_stricmp(void *a, void *b)
{
    return stricmp(*(char**)a, *(char**)b);
}

Int tablen_intcmp(void *a, void *b)
{
    return (*(Int*)a) - (*(Int*)b);
}

Int tablen_doublecmp(void *a, void *b)
{
    if ((*(double*)a) > (*(double*)b)) return 1;
    if ((*(double*)a) < (*(double*)b)) return -1;
    return 0;
}

Int tablen_longcmp(void *a, void *b)
{
    if ((*(long*)a) < (*(long*)b)) return -1;
    if ((*(long*)a) > (*(long*)b)) return 1;
    return 0;
}

Int tablen_ptrcmp(void *a, void *b)
{
    if ((*(void**)a) < (*(void**)b)) return -1;
    if ((*(void**)a) > (*(void**)b)) return 1;
    return 0;
}

void tablen_print_int(void *a)
{
    printf("%ld", *(Int*)a);
}

void tablen_print_str(void *a)
{
    printf("%s", *(char**) a);
}

Int tablen_ptr_is_empty(void *a)
{
   return(*(void **)a==NULL);
}
void tablen_ptr_make_empty(void *a)
{
   *(void **)a=NULL;
}
Int tablen_ptr_is_deleted(void *a)
{
   return(*(void **)a==(void *)-1);
}
void tablen_ptr_make_deleted(void *a)
{
   *(void **)a=(void *)-1;
}
Int tablen_ptr_hash1(void *a)
{
   return(((Int)(*((void **)a)))>>2);
}
