/**
 ** xdraw.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/



#include CONFIG

#ifdef UNIX 

#include <util.h>
#include <growbuf.h>

#include <math.h>

#include "xdraw.h"

XColor black_color, white_color;

Display *display;
Int     screen;

Int win_debug= 0;

void setup_x(void)
{
    display= XOpenDisplay(NULL);
    if (!display) die(("Cannot open display\n"));
    screen= DefaultScreen(display);

    create_color(0,0,0, &black_color);
    create_color(100,100,100, &white_color);
}

#define MAX_COLOR_ENTRIES 100

XColor xdraw_color_table[MAX_COLOR_ENTRIES];
int xdraw_color_table_size;

int create_or_find_color(int r, int g, int b, XColor *c)
{
    int i;
    
    for (i= 0; i< xdraw_color_table_size; i++) {
	if ((xdraw_color_table[i].red / 655) == r &&
	    (xdraw_color_table[i].green / 655) == g &&
	    (xdraw_color_table[i].blue / 655) == b) {
	    copy_color(c, &xdraw_color_table[i]);
	    return 0;
	}
    }

    if (xdraw_color_table_size < MAX_COLOR_ENTRIES) {
	create_color(r,g,b, &xdraw_color_table[xdraw_color_table_size]);
	copy_color(c, &xdraw_color_table[xdraw_color_table_size]);
	xdraw_color_table_size++;
	return 0;
    }
    return 1;
}

void create_color(int r, int g, int b, XColor *c)
{
    c->red= (short) r * 655;
    c->green= (short) g * 655;
    c->blue= (short) b * 655;
    XAllocColorCells(display, DefaultColormap(display, screen), 0,
		     0, 0, &c->pixel, 1);
    /*printf("got color %ld\n", c->pixel);*/
    copy_color(c, c);
    XStoreColor(display, DefaultColormap(display, screen), c);
}

void copy_color(XColor *dest, XColor *src)
{
    dest->red=   src->red;
    dest->green= src->green;
    dest->blue=  src->blue;
    dest->flags= DoRed | DoGreen | DoBlue;
    dest->pixel= src->pixel;
}

void create_mutable_color(Int r, Int g, Int b, XColor *c)
{
    c->red= (short) r * 655;
    c->green= (short) g * 655;
    c->blue= (short) b * 655;
    XAllocColorCells(display, DefaultColormap(display, screen), 0,
		     0, 0, &c->pixel, 1);
    /*printf("got color %ld\n", c->pixel);*/
    copy_color(c, c);
}

#if 0
NOPROTO void main(Int argc, char ** argv)  /*__ no proto */
{
    Win *layer1, *layer2, *w;
    XImage *layer1_im, *layer2_im;
    XImage *im;
    Int x;

    setup_x();
    w= make_window(xdim,ydim,0);

    win_scale(w, 1.0);

    drawit(w);
    saber_stop();
}

#endif


void draw_pixel(Win *w, Int x, Int y)
{
    win_newpath(w);
    win_moveto(w, x, y);
    win_lineto(w, x, y);
    win_stroke(w);
}
    
void draw_line(Win *w, Int x, Int y, Int xx, Int yy)
{
    win_newpath(w);
    win_moveto(w, x, y);
    win_lineto(w, xx, yy);
    win_stroke(w);
}
    
void draw_box(Win *w, Int x1, Int y1, Int x2, Int y2, Int x3, Int y3, Int x4, Int y4)
{
    win_setcolor(w, &black_color);

    win_newpath(w);
    win_moveto(w, x1, y1);
    win_lineto(w, x2, y2);
    win_lineto(w, x3, y3);
    win_lineto(w, x4, y4);
    win_lineto(w, x1, y1);
    win_fill(w);

    win_setcolor(w, &white_color);

    win_newpath(w);
    win_moveto(w, x1, y1);
    win_lineto(w, x2, y2);
    win_lineto(w, x3, y3);
    win_lineto(w, x4, y4);
    win_lineto(w, x1, y1);
    win_stroke(w);
}

XImage *get_image(Win *w)
{
    return
      XGetImage(w->display, w->w, 0, 0, w->width, w->height, AllPlanes, ZPixmap);
}

void put_image(Win *w, XImage *im)
{
    XPutImage(w->display, w->w, w->gc, im, 0, 0, 0, 0, w->width, w->height);
}

void copy_win(Win *dest, Win *src)
{
    assert(dest->display== src->display);
    XCopyArea(src->display, src->w, dest->w, dest->gc, 0, 0, src->width, src->height, 0, 0);
}
    
Win *make_window(Int width, Int height, Int pixmap)
{
    XGCValues values;
    Win *win= malloc(sizeof(Win));

    win->arc= 0;
    win_scale(win, 1.0);
    growbuf_init(&win->points);
    win->display= display;
    win->screen= screen;
    win->disable= 0;

    if (pixmap) {
	win->w= XCreatePixmap(display, RootWindow(display, screen),
			      width, height,
			      DefaultDepthOfScreen(XScreenOfDisplay(display, screen)));
    }
    else {
	win->w= XCreateSimpleWindow(display, RootWindow(display, screen),
				    0,0,width,height,1,
				    WhitePixel(display,screen),
				    BlackPixel(display,screen));
	XMapWindow(display, win->w);
	XSync(display, 0);
	XSelectInput(display, win->w,
		     ExposureMask | KeyPressMask | ButtonPressMask |
		     ButtonReleaseMask | StructureNotifyMask |
		     ButtonMotionMask);

	/*    XSetGraphicsExposures(display, win->gc, False); */

    }

    win->gc= XCreateGC(display, win->w, 0, &values);
    win->width= width;
    win->height= height;
    win->pixmap= pixmap;

    win_setcolor(win, &black_color);
    
    XFillRectangle(display, win->w, win->gc, 0,0,width, height);

    win_setlinewidth(win, 1);
    win_setcolor(win, &white_color);
    win_flush(win);
    return win;
}

Int win_compute_scale(Win *w, Int x)
{
    return (Int) floor(.5 + w->scale * (double)x);
}

void win_clear(Win *w)
{
    XFillRectangle(display, w->w, w->gc, 0,0,w->width, w->height);
}

XColor *win_getcolor(Win *w)
{
   return w->xcolor;
}

void win_setcolor(Win *w, XColor *c)
{
   w->xcolor= c;
   XSetForeground(w->display, w->gc, c->pixel);
}

void win_scale(Win *w, double scale)
{
    w->scale= scale;
}

void win_flush(Win *w)
{
    XSync(w->display, 0);
}
    
void win_setlinewidth(Win *w, Int x)
{
    Int width= win_compute_scale(w, x);
    w->disable= !x;
    if (x && !width) width= 1;
    XSetLineAttributes(w->display, w->gc, width, LineSolid, CapRound, JoinRound);
    if (win_debug) printf("line width now %d\n", w->gc->values.line_width);
}

void win_setlinecap(Win *w, Int x)
{
    switch (x) {
      case 1:
	break;
      default:
	unimplemented(("line cap other than 1\n"));
    }
}

void draw_points(Win *w, void *points, Int npoints)
{
    XDrawPoints(w->display, w->w, w->gc, points, npoints, CoordModeOrigin);
}

void win_stroke(Win *w)
{
    Int lines= growbuf_length(&w->points);
    Int arc= w->arc;

    if (lines && arc) die(("Both lines and arc in stroke\n"));

    if (!w->disable) {
	if (lines) {
	    XDrawLines(w->display, w->w, w->gc,
		       (XPoint*) growbuf_data(&w->points),
		       growbuf_length(&w->points) / sizeof(XPoint),
		       CoordModeOrigin);
	    if (win_debug) printf("drew %ld points\n", growbuf_length(&w->points) / sizeof(XPoint));
	}
	else if (arc) {
	    XDrawArc(w->display, w->w, w->gc,
		     w->arc_x, w->arc_y,
		     w->arc_diam, w->arc_diam,
		     w->arc_beg, w->arc_span);
	}
	else {
	    die(("No path in stroke\n"));
	}
    }
}

void win_fill(Win *w)
{
    Int lines= growbuf_length(&w->points);
    Int arc= w->arc;

    if (lines && arc) die(("Both lines and arc in fill\n"));

    if (lines) {
	XFillPolygon(w->display, w->w, w->gc,
		     (XPoint*) growbuf_data(&w->points),
		     growbuf_length(&w->points) / sizeof(XPoint),
		     Nonconvex, CoordModeOrigin);
    }
    else if (arc) {
	XFillArc(w->display, w->w, w->gc,
		 w->arc_x, w->arc_y,
		 w->arc_diam, w->arc_diam,
		 w->arc_beg, w->arc_span);
    }
    else {
	die(("No path in fill\n"));
    }
}
		 
void win_newpath(Win *w)
{
    growbuf_clear(&w->points);
    w->arc= 0;
}

void win_moveto(Win *w, Int x, Int y)
{
    XPoint xpoint;
    xpoint.x= win_compute_scale(w, x);
    xpoint.y= win_compute_scale(w, y);
    if (growbuf_length(&w->points)) {
	die(("Didn't execute newpath before moveto\n"));
    }
    growbuf_add(&w->points, &xpoint, sizeof(XPoint));
    if (win_debug) printf("moveto %d %d\n", (Int)xpoint.x, (Int)xpoint.y);
}

void win_lineto(Win *w, Int x, Int y)
{
    XPoint xpoint;
    xpoint.x= win_compute_scale(w, x);
    xpoint.y= win_compute_scale(w, y);
    growbuf_add(&w->points, &xpoint, sizeof(XPoint));
    if (win_debug) printf("lineto %d %d\n", (Int)xpoint.x, (Int)xpoint.y);
}    

void win_arc(Win *w, Int x, Int y, Int diam, Int beg, Int span)
{
    if (w->arc) die(("Multiple arcs in path"));
    w->arc= 1;
    w->arc_x= win_compute_scale(w, x - (diam / 2));
    w->arc_y= win_compute_scale(w, y - (diam / 2));
    w->arc_diam= win_compute_scale(w, diam);
    if (w->arc_diam < 2) w->arc_diam= 2;
    w->arc_beg= beg*64;
    w->arc_span= span*64;
}

int win_file_descriptor(Win *w)
{
   return ConnectionNumber(w->display);
}

int win_get_mouse_state(Win *w, int *x, int *y, int *buttons)
{
   Window root, child;
   int root_x, root_y, child_x, child_y;
   
   XQueryPointer(w->display, w->w, &root, &child,
		 &root_x, &root_y, &child_x, &child_y, buttons);
   *x= child_x;
   *y= child_y;
   return 0;
}

   
int win_get_event(Win *w)
{
   XEvent event;
   int got_event= 0;
   
   while (1) {
      /*printf("get_event\n");*/
      if (!XCheckMaskEvent(w->display, -1, &event)) return got_event;
      got_event= 1;
      /*printf("more ...");*/
   }
}

#endif /* #ifdef UNIX */

