/**
 ** listcore.c
 **
 ** Copyright 1990, 1991 by Randy Sargent.
 **
 ** The author hereby grants to MIT permission to use this software.
 ** The author also grants to MIT permission to distribute this software
 ** to schools for non-commercial educational use only.
 **
 ** The author hereby grants to other individuals or organizations
 ** permission to use this software for non-commercial
 ** educational use only.  This software may not be distributed to others
 ** except by MIT, under the conditions above.
 **
 ** Other than these cases, no part of this software may be used or
 ** distributed without written permission of the author.
 **
 ** Neither the author nor MIT make any representations about the 
 ** suitability of this software for any purpose.  It is provided 
 ** "as is" without express or implied warranty.
 **
 ** Randy Sargent
 ** Research Specialist
 ** MIT Media Lab
 ** 20 Ames St.  E15-301
 ** Cambridge, MA  02139
 ** E-mail:  rsargent@athena.mit.edu
 **
 **/

/**
 **  listcore.c
 **
 ** v1.0  Sun Jun  2 14:01:06 1991  Randy Sargent.  created
 ** v1.1  Mon Aug 12 11:53:04 1991  Randy Sargent.  added ListConstruct
 **/

#include CONFIG

#include "util.h"
#include "runtype.h"
#include "stringlb.h"
#include "listcore.h"

List list_the_empty_list_storage= {
#if RUN_TIME_TYPING
    runtype_list,
#endif    
    &list_the_empty_list_storage,
    &list_the_empty_list_storage
#ifdef LIST_BAGGAGE_INIT_NIL
      , LIST_BAGGAGE_INIT_NIL
#endif	
  };
List *list_the_empty_list= &list_the_empty_list_storage;
List *Nil= &list_the_empty_list_storage;

List list_end_of_list_storage= {
#if RUN_TIME_TYPING
    runtype_list,
#endif    
    &list_the_empty_list_storage,
    &list_the_empty_list_storage
#ifdef LIST_BAGGAGE_INIT_NIL
      , LIST_BAGGAGE_INIT_NIL
#endif	
  };
    
List *Eol= &list_end_of_list_storage;

#ifndef MALLOC
#define MALLOC malloc
#endif

#ifndef FREE
#define FREE(ptr, size) free(ptr)
#endif

Int list_init_module(void)
{
#if RUN_TIME_TYPING    
    runtype_register_service(runtype_list, service_display,
			     (Service_func) list_display);
    runtype_register_service(runtype_list, service_print,
			     (Service_func) list_print);
    runtype_register_service(runtype_list, service_sprint,
			     (Service_func) list_sprint);
    runtype_register_service(runtype_list, service_gc,
			     (Service_func) list_gc);
#endif    
    return 0;
}

List *list_cons(void *car, void *cdr)
{
    List *ret= MALLOC(sizeof(List));
    if (ret) {
#ifdef LIST_BAGGAGE_INIT
	LIST_BAGGAGE_INIT
#endif
#if RUN_TIME_TYPING
	ret->runtype= runtype_list;
#endif    
	ret->car= car;
	ret->cdr= cdr;
    }
    return ret;
}

List *list_reverse(List *input)
{
    List *output= Nil;
    while (input != Nil) {
	output= list_cons(CAR(input), output);
	input= CDR(input);
    }
    return output;
}

			  
void list_push(List **l, void *item)
{
    *l= list_cons(item, *l);
}

void *list_pop(List **l)
{
    if (*l != Nil) {
	void *ret= CAR(*l);
	*l= CDR(*l);
	return ret;
    }
    else return Eol;
}

/* 0 if successful.  1 if item not present.
   will delete multiple occurences of item.
   will free cons cells that are taken out of list */

Int list_delete(List **l, void *item)
{
    Int not_found= 1;
    while (*l != Nil) {
	if (CAR(*l) == item) {
	    List *to_free= *l;
	    *l= CDR(*l);
	    FREE(to_free, sizeof(List));
	    not_found= 0;
	}
	else {
	    l= (List**) &CDR(*l);
	}
    }
    return not_found;
}

void list_free_cell(List *l)
{
    FREE(l, sizeof(List));
}

/* Functions for incremental construction of a list from front to back */
/* (back to front is trivial with use of list_cons) */

void list_start(ListConstruct *lc)
{
    lc->car= lc;
    lc->cdr= Nil;
}

void list_add_to_end(ListConstruct *lc, void *item)
{
    List *newcons= list_cons(item, Nil);
    if (!newcons) lc->cdr= 0;
    if (lc->car) ((List*)lc->car)->cdr=  newcons;
    lc->car= newcons;
}

List *list_end(ListConstruct *lc)
{
    return lc->cdr;
}

List *list_end_set_cdr(ListConstruct *lc, void *cdr)
{
    if (lc->car) ((List*)lc->car)->cdr= cdr;
    return lc->cdr;
}

List *list(void *first, ...)
{
    ListConstruct lc;
    va_list args;
    
    if (first == Eol) return Nil;
    list_start(&lc);
    list_add_to_end(&lc, first);
    va_start(args, first);
    while (1) {
	void *arg;
	arg= va_arg(args, void*);
	if (arg == Eol) break;
	list_add_to_end(&lc, arg);
    }
    return list_end(&lc);
}
    
void list_set_car(List *l, void *car)
{
    if (l != Nil) l->car= car;
}

void list_set_cdr(List *l, void *cdr)
{
    if (l != Nil) l->cdr= cdr;
}

/* May die if given something other than a proper association list */

List *list_assoc(List *l, void *obj)
{
    while (l != Nil) {
	if (CAR((List*)CAR(l)) == obj) return CAR(l);
	l= CDR(l);
    }
    return Nil;
}

/* May die if given something other than a proper list */
Int list_length(List *l)
{
    Int len= 0;
    while (l != Nil) {
	len++;
	l= CDR(l);
    }
    return len;
}
      
#if RUN_TIME_TYPING

NOPROTO void list_display(void *o) 
{
    List *l= (List*) o;
    proc_printf("[list ");
    while (l != Nil) {
	display(CAR(l));
	l= CDR(l);
    }
    proc_printf("] ");
}

NOPROTO void list_print(void *o) 
{
    List *l= (List*) o;
    proc_printf("[");
    while (l != Nil) {
	print(CAR(l));
	l= CDR(l);
	if (l == Nil) break;
	proc_printf(" ");
    }
    proc_printf("]");
}

NOPROTO char *list_sprint(void *o) 
{
    char *ret;
    List *l= (List*) o;
    ret= string_copy("[");
    while (l != Nil) {
	ret= string_append(free_soon(ret), free_soon(sprint(CAR(l))));
	l= CDR(l);
	if (l == Nil) break;
	ret= string_append(free_soon(ret), " ");
    }
    ret= string_append(free_soon(ret), "]");
    return ret;
}

NOPROTO void list_gc(void *o) 
{
    runtype_gc(CAR((List*)o));
    runtype_gc(CDR((List*)o));
#ifdef LIST_BAGGAGE_GC
    LIST_BAGGAGE_GC
#endif
}
    
#else

NOPROTO void list_display(List *l, void (*displayfunc) (void*))
{
    proc_printf("[List ");
    while (l != Nil) {
	(*displayfunc)(CAR(l));
	l= CDR(l);
    }
    proc_printf("] ");
}

NOPROTO void list_print(List *l, void (*printfunc) (void*)) 
{
    proc_printf("[");
    while (l != Nil) {
	(*printfunc)(CAR(l));
	l= CDR(l);
	if (l == Nil) break;
	proc_printf(" ");
    }
    proc_printf("]");
}

#endif
