/* ==== specific.c =======================================================
 * Copyright (c) 1994 by Chris Provenzano, proven@mit.edu
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *  This product includes software developed by Chris Provenzano.
 * 4. The name of Chris Provenzano may not be used to endorse or promote 
 *	  products derived from this software without specific prior written
 *	  permission.
 *
 * THIS SOFTWARE IS PROVIDED BY CHRIS PROVENZANO ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL CHRIS PROVENZANO BE LIABLE FOR ANY 
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR 
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT 
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF 
 * SUCH DAMAGE.
 *
 * Description : Pthread thread specific data management.
 *
 *  1.20 94/03/30 proven
 *      -Started coding this file.
 */

#ifndef lint
static const char rcsid[] = "specific.c,v 1.50.2.1 1994/08/24 06:11:18 proven Exp";
#endif

#include <pthread.h>

static struct pthread_key key_table[PTHREAD_DATAKEYS_MAX];
static semaphore key_lock = SEMAPHORE_CLEAR;

/* ==========================================================================
 * pthread_key_create()
 */
int pthread_key_create(pthread_key_t *key, void (*destructor)(void *))
{
	semaphore *lock, *klock;

	lock = &(key_lock);
	while (SEMAPHORE_TEST_AND_SET(lock)) {
		pthread_yield();
	}
	for ((*key) = 0; (*key) < PTHREAD_DATAKEYS_MAX; (*key)++) {
		if (key_table[(*key)].count == 0) {
			key_table[(*key)].destructor = destructor;
			key_table[(*key)].count++;

			klock = & (key_table[(*key)].lock);
			SEMAPHORE_RESET(klock);
			SEMAPHORE_RESET(lock);
			return(OK);
		}
	}
	SEMAPHORE_RESET(lock);
	return(EAGAIN);
}

/* ==========================================================================
 * pthread_key_destroy()
 */
int pthread_key_destroy(pthread_key_t key)
{
	semaphore *lock;
	int ret;

	if (key < PTHREAD_DATAKEYS_MAX) {
		lock = &(key_table[key].lock);
		while(SEMAPHORE_TEST_AND_SET(lock)) {
			pthread_yield();
		}
		switch (key_table[key].count) {
		case 1:
			key_table[key].destructor = NULL;
			key_table[key].count = 0;
		case 0:
			ret = OK;
			break;
		default:
			ret = EBUSY;
		}
		SEMAPHORE_RESET(lock);
	} else {
		ret = EINVAL;
	}
	return(ret);
}

/* ==========================================================================
 * pthread_cleanupspecific()
 */
void pthread_cleanupspecific(void) 
{
	semaphore *lock;
	int key;

	lock = &(key_lock);
	while (SEMAPHORE_TEST_AND_SET(lock)) {
		pthread_yield();
	}
	for (key = 0; key < PTHREAD_DATAKEYS_MAX; key++) {
		if ((key_table[key].count > 1) && (pthread_run->specific_data[key])) {
			if (key_table[key].destructor) {
				key_table[key].destructor(pthread_run->specific_data[key]);
			}
			key_table[key].count--;
		}
	}
	free(pthread_run->specific_data);
	SEMAPHORE_RESET(lock);
}

static inline const void ** pthread_key_allocate_data(void)
{
	const void ** new_data;
	if (new_data = (const void**)malloc(sizeof(void *) * PTHREAD_DATAKEYS_MAX)) {
		memset((void *)new_data, 0, sizeof(void *) * PTHREAD_DATAKEYS_MAX);
	}
	return(new_data);
}

/* ==========================================================================
 * pthread_setspecific()
 */
int pthread_setspecific(pthread_key_t key, const void * value)
{
	semaphore *lock;
	int ret;

	if ((pthread_run->specific_data) ||
	  (pthread_run->specific_data = pthread_key_allocate_data())) {
		if ((key < PTHREAD_DATAKEYS_MAX) && (key_table)) {
			lock = &(key_table[key].lock);
			while(SEMAPHORE_TEST_AND_SET(lock)) {
				pthread_yield();
			}
			if (key_table[key].count) {
				if (pthread_run->specific_data[key] == NULL) {
					key_table[key].count++;
				} else if (value == NULL) {
					key_table[key].count--;
				}
				pthread_run->specific_data[key] = value;
				ret = OK;
			} else {
				ret = EINVAL;
			}
		} else {
			ret = EINVAL;
		}
		SEMAPHORE_RESET(lock);
	} else {
		ret = ENOMEM;
	}
	return(ret);
}

/* ==========================================================================
 * pthread_getspecific()
 */
void * pthread_getspecific(pthread_key_t key)
{
	semaphore *lock;
	void *ret;

	if ((pthread_run->specific_data) && (key < PTHREAD_DATAKEYS_MAX)
      && (key_table)) {
		lock = &(key_table[key].lock);
		while(SEMAPHORE_TEST_AND_SET(lock)) {
			pthread_yield();
		}
		if (key_table[key].count) {
			ret = (void *)pthread_run->specific_data[key];
		} else {
			ret = NULL;
		}
		SEMAPHORE_RESET(lock);
	} else {
		ret = NULL;
	}
	return(ret);
}
