/*
 * KCLL -  Ken and Chee's Limey Lisp  
 * All Rights to the code and any products created with this code is 
 * hereby granted. 
 *
 * I.E. You can do whatever the hell you want with this code. 
 * The only restriction is that this copyright notice not be modified.
 */

#include "vector.h"
#include "num.h"
#include "cont.h"
#include "stream.h"

LLTag llvector_t;

static LLVector *make_vector()
{
  LLVector *v;

  v = (LLVector *) llmake_obj(llvector_t); 
  v->size = 0; 
  v->vector = (LLObj **) malloc(0);
  return (v);
}

static void resize_vector(v, size) 
     LLVector *v;
     long size; 
{
  int n;

  v->vector = (LLObj **) realloc (v->vector, sizeof(LLObj *) * size);
  for (n = v->size; n < size; n++)
    v->vector[n] = NIL;		/* Fill up new space with NIL's */
  v->size = size;
}

LLVector *llcmake_vector(size) 
int size; 
{ 
  LLVector *v; 
  v = make_vector() ; 
  resize_vector(v, size); 
  return (v);
}

void llmake_vector() 
{
  long size;
  LLVector *v;

  size = llnum_to_long(llnextcheckedarg(llnum_t)); 
  lllastarg();
  if (size < 0)
    llerror(LLARGS_OUT_OF_RANGE);
  v = make_vector(); 
  resize_vector(v, size);
  llpusharg(v);
}
 
void llresize_vector() 
{
  long size;
  LLVector *v;

  v = (LLVector *) llnextcheckedarg(llvector_t);
  size = llnum_to_long(llnextcheckedarg(llnum_t));
  if (size < 0) llerror(LLARGS_OUT_OF_RANGE);
  lllastarg(); 
  resize_vector(v, size);
  llpusharg(v);
}

void llvector_length()
{
  LLVector *v; 

  v = (LLVector *) llnextcheckedarg(llvector_t); 
  lllastarg();
  llpusharg(lllong_to_num(llcvector_length(v)));
}
 
void llvector_ref() 
{
  LLVector *v; 
  long ref;
  
  v = (LLVector *) llnextcheckedarg(llvector_t);
  ref = llnum_to_long(llnextcheckedarg(llnum_t));
  lllastarg();
  if (ref >= v->size) llerror(LLARGS_OUT_OF_RANGE);
  llpusharg(v->vector[ref]);
}

void llvector_setbang()
{
  LLVector *v; 
  LLObj *obj; 
  long ref;

  v = (LLVector *) llnextcheckedarg(llvector_t);
  ref  = llnum_to_long(llnextcheckedarg(llnum_t));
  obj = llnextarg();
  lllastarg();
  if (ref >= v->size) 
    llerror(LLARGS_OUT_OF_RANGE);
  v->vector[ref] = obj;
  llpusharg(obj);
}

static void print_vector (vector, stream) 
     LLVector *vector; 
     LLStream *stream; 
{
  long i; 
  LLObj **scan;

  llstrprintf(stream, "#(");
  for (i = vector->size, scan = vector->vector; 
       i; 
       scan++, i--) {
    llprint_obj (*scan, stream);
    if (i > 1) llstrprintf(stream, " ");
  }
  llstrprintf(stream, ")");
}

static void touch_vector (vector)
     LLVector *vector;
{
  long i; 
  LLObj **scan;

  for (i = vector->size, scan = vector->vector; 
       i; 
       scan++, i--) 
    lltouch_obj (*scan);
}

void llvector() 
{
  LLVector *v; 
  LLObj **obj;
  v = llcmake_vector(llargc); 
  obj = v->vector;
  while(llargc)  {
    *(obj++) = llnextarg(); 
  }
  llpusharg(v);
}
     
void llinit_vector() 
{
  llvector_t = lladd_obj_td (sizeof(LLVector), "Vector", 0, touch_vector,
			     print_vector, 0 ); 
  llregister_cfunc(llvector, "vector"); 
  llregister_cfunc(llvector_length, "vector-length");
  llregister_cfunc(llmake_vector, "make-vector"); 
  llregister_cfunc(llresize_vector, "resize-vector");
  llregister_cfunc(llvector_ref, "vector-ref");
  llregister_cfunc(llvector_setbang, "vector-set!");
}


