#   toba -- build a Java program using Toba
#
#   Copyright 1996, 1997 Arizona Board of Regents; see COPYRIGHT file for 
#   details.
#
#
#   usage:  toba [options] file...
#
#   options:
#
#	-A string	add string to args passed to cc
#	-B machine      bootstrap by running the interpreter on a remote machine
#	-C		produce C files but don't compile or link them
#	-c		compile to .o files but do not link
#	-d flags	set debugging flags for Toba
#	-g		pass -g (enable debugging) to cc
#	-I path		set include path for C compiler
#			(note: Java API is always in path)
#	-i		retain intermediate files upon exit
#	-J		use interpreted version of Toba
#	-K		run javac to produce .class files but go no further
#       -L ldfilter     set loader filter program
#	-m mainclass	generate a main from mainclass
#	-O		pass -O (enable optimization) to cc and javac
#	-o filename	write executable to filename instead of a.out
#	-P packagedir	create a package in packagedir
#	-R              re-translate classes we found in tobaclasspath
#	-r		recursively pull in all dependencies
#	-S		produce .s files but don't assemble or link them
#	-v 		("verbose") list commands executed
#	-w		suppress warnings from javac
#	-X		use directly executable version of Toba (default)
#
#
#   file argument interpretation:
#
#	file.java	Java file to compile, translate, and link
#	file.class	class file to translate and link
#	file.c		C file to compile and link
#	file.o		object file to link
#	file.a		library to link
#	file.so		library to link
#	-lxxx		library to link (must follow a "normal" file)
#	-Lxxx		library directory to search (must follow "normal" file)
#	classname	Java class from $CLASSPATH to translate and link

# return the full path of an executable
whereis() {
   # get the path
   ret=`type $* 2>/dev/null | sed -e 's#[()]##g' -e 's#^.* ##g'`

   # make it absolute
   case $ret in
      /*) ;;
      *)  ret=`pwd`/$ret;;
   esac

   # eliminate . and .. components
   echo $ret | sed -e 's#/\./#/#g'	\
		   -e 's#/[^/][^./]*/\.\./#/#g'
}

# run toba with the Java interpreter
# CPATH and TOP should be computed before this is called
itoba()
{
    # $OSTYPE is a bash builtin variable and bash
    # is the /bin/sh that comes with cygwin
    ICPATH=$CPATH:.:$TOP
    if [ "X$OSTYPE" = "Xwin32" ] ; then
        # win32's interpreter needs semicolon separators
        ICPATH=`echo $ICPATH | sed -e 's/:/;/g'`
	TOBAPATH=`echo $TOBAPATH | sed -e 's/:/;/g'`
    fi

    if [ "X$BOOTSTRAP" != "X" ]; then
       $BOOTSTRAP "cd $PWD; CLASSPATH=$ICPATH $JAVA -Dtoba.class.path=$TOBAPATH toba.translator.Trans $@"
    else
       CLASSPATH=$ICPATH $JAVA -Dtoba.class.path=$TOBAPATH toba.translator.Trans "$@"
    fi
}

ijavac() {
    if [ "X$BOOTSTRAP" != "X" ]; then
       $BOOTSTRAP "cd $PWD; CLASSPATH=$CLASSPATH $JAVAC $@"
    else
       CLASSPATH=$CLASSPATH $JAVAC "$@"
    fi
}

#  test if a file is newer than another
is_newer() {
    # use find to see if the file is older, ignoring all errors
    res=`find "$1" -newer "$2" 2>/dev/null || true`
    if [ "$res" = "$1" ] ; then
        return 0     # true
    else
        return 1     # false
    fi
}

#  execute command, possibly echoing to stdout
trace() {
    trap 'rm -f $TMP; exit 1' 1 2 15	# seems to be needed
    # sed protects dollar signs from expansion; tr protects sed from long lines
    CMD=`echo "$@" | tr ' ' '\012' | sed 's/\\$/\\\\$/g' | tr '\012' ' '`
    $ECHO + "$CMD"
    eval "$CMD" || exit 1
}

#  remove files; disabled by -i option
remove() {
    trap 'rm -f $TMP; exit 1' 1 2 15	# seems to be needed
    if [ $# -gt 0 ]; then
	$RMTRACE rm -f "$@"
    fi
}
RMTRACE=trace



# deduce current directory
PROG=`whereis $0`
TOP=`echo $PROG | sed 's#/[^/]*/[^/]*$##'`


# paths for interpreted and executed javac and toba
IJAVAC=ijavac
ITOBA=itoba
XJAVAC=$TOP/bin/xjavac
XTOBA=$TOP/bin/xtoba
JAVA=java

# default option values
TOBA=$XTOBA
MYJAVAC=$XJAVAC
NOTOBA=
NOCOMPILE=
NOLINK=
XFILE=a.out
PACKAGENAME=
INCLUDES=
ECHO=:
JFLAGS=
CFLAGS="$CTOBA -w"
TFLAGS=
LDFLAGS=
LDFILTER=

#  process command options

USAGE="usage: $0 [-cCgiJKOrSvwX] [-A arg] [-d flags] [-IP dir] [-m main] [-o file] file..."

# NB: -B disabled per checked in version
while getopts A:Ccd:gI:iJKL:m:Oo:P:rRSvwX c; do
    case $c in
	A)  CFLAGS="$CFLAGS $OPTARG";;
	B)  BOOTSTRAP=rsh $OPTARG;;
	C)  NOCOMPILE=exit;;
	c)  NOLINK=exit;;
	d)  TFLAGS="$TFLAGS -d$OPTARG";;
	g)  CFLAGS="$CFLAGS -g";;
	I)  INCLUDES="$INCLUDES -I$OPTARG";;
	i)  RMTRACE=:;;
	J)  TOBA=$ITOBA; MYJAVAC=$IJAVAC;;
	K)  NOTOBA=exit;;
        L)  LDFILTER="${OPTARG}";;
        m)  TFLAGS="$TFLAGS -m $OPTARG";;
	O)  CFLAGS="$CFLAGS -O";  JFLAGS="$JFLAGS -O";;
	o)  XFILE="$OPTARG";;
        P)  PACKAGENAME="$OPTARG"
            TFLAGS="$TFLAGS -P $PACKAGENAME";;
        R)  TFLAGS="$TFLAGS -R";;
        r)  TFLAGS="$TFLAGS -r";;
	S)  NOLINK=exit;  CFLAGS="$CFLAGS -S";;
	v)  ECHO=echo;;
	w)  JFLAGS="$JFLAGS -nowarn";;
	X)  TOBA=$XTOBA; MYJAVAC=$XJAVAC;;
	\?) echo $USAGE; exit 1;;
    esac
done


#  process file names

shift `expr $OPTIND - 1`

if [ $# = 0 ]; then
    echo $USAGE
    exit 1
fi

JFILES=
KFILES=
CFILES=
LFILES=

for f in $*; do
    case $f in
	*.java)		JFILES="$JFILES $f";;
	*.class)	KFILES="$KFILES $f";;
	*.c)		CFILES="$CFILES $f";;
	*.h)		HFILES="$HFILES $f";;
	*.o)		OFILES="$OFILES $f";;
	*.a)		LFILES="$LFILES $f";;
	*.so)		LFILES="$LFILES $f";;
        -L*)		LFILES="$LFILES $f";;
	-l*)		LFILES="$LFILES $f";;
	-Wl,*)		LFILES="$LFILES $f";;
	*)		KFILES="$KFILES $f";;
    esac
done


# make sure SUN, RUNTIME, TOBA and API are in the toba path
# Note: the ordering (and the fact that API is listed twice) is
# important when linking statically.
PKGS=$TOP/packages
STDTOBAPATH=$PKGS/BISS:$PKGS/TOBA:$PKGS/SUN:$PKGS/API:$PKGS/RUNTIME:$PKGS/API
if [ "x$TOBAPATH" != "x" ] ; then
    trace TOBAPATH=$TOBAPATH:$STDTOBAPATH
else
    trace TOBAPATH=$STDTOBAPATH
fi
trace export TOBAPATH

# explicit or implicit CLASSPATH, before we add to it
CPATH="${CLASSPATH:-.}"

# set JAVA_HOME in the same way javac would
PRG=`whereis ${JAVA}`
trace J_HOME=`dirname $PRG`/..
if [ -z "${JAVA_HOME}" ] ; then
   JAVA_HOME=${J_HOME}
   trace export JAVA_HOME
fi

# Process TOBAPATH to get library and include directories
OLDIFS=$IFS
IFS=:
for component in $TOBAPATH ; do
    BASE=`basename $component`
    if [ -f $component/lib$BASE* ] ; then
        EXTRAFLAGS=`cat $component/extraflags`
        LDFLAGS="$LDFLAGS -L$component -l$BASE $EXTRAFLAGS"
        INCLUDES="$INCLUDES -I$component/include"
        if [ "x$RPOPT" != "x" ] ; then
            RPOPT="$RPOPT$component:"
        fi
    fi
done
if [ "x$RPOPT" != "x" ] ; then
    LDFLAGS="$LDFLAGS $RPOPT$component"
fi
IFS=$OLDIFS


# exit on any errors beyond here
set -e

# touch timestamp file; arrange deletion on exit
TMP=.tmp.toba.$$
trap 'rm -f $TMP' 0 1 2 15
touch $TMP
# is_newer needs at least one second delta to work properly. ugh
sleep 1


# Make package structure
if [ "x$PACKAGENAME" != "x" ] ; then
    test -d $PACKAGENAME         || trace mkdir $PACKAGENAME
    test -d $PACKAGENAME/include || trace mkdir $PACKAGENAME/include
    echo $LFILES > $PACKAGENAME/extraflags
fi


# compile Java files to bytecode
INT1=
if [ "x$JFILES" != "x" ]; then
    export JAVA_HOME  
trace CLASSPATH=$CPATH:$TOBAPATH
trace export CLASSPATH

    # compute a list of directories java src is in, and compile the src
    JDIRS=`(echo .; for f in $JFILES ; do dirname $f; done) | sort | uniq`
    trace $MYJAVAC $JFLAGS $JFILES

    # find all new class files in the src directories
    INT1=
    for d in $JDIRS; do
        for f in $d/*.class; do
            if is_newer "$f" $TMP; then
                INT1="$INT1 $f"
            fi
        done
    done
    KFILES="$KFILES $INT1"
fi

# exit if -S given
$NOTOBA


# translate bytecode to C code
INT2=
if [ "x$KFILES" != "x" ]; then
    trace CLASSPATH="$CPATH:$TOP"
    trace export CLASSPATH
    trace $TOBA $TFLAGS $KFILES
    remove $INT1

    INT2=
    for f in *.c; do
        if is_newer "$f" $TMP ; then
            INT2="$INT2 $f"
        fi
    done     
    if [ "x$INT2" != "x" ] ; then
        CFILES="$CFILES $INT2"
    fi

    INT2h=
    for f in *.h; do
        if is_newer "$f" $TMP ; then
            INT2h="$INT2h $f"
        fi
    done     
    if [ "x$INT2h" != "x" ] ; then
        HFILES="$HFILES $INT2h"
    fi
fi

# preserve headers when rolling packages
if [ "x$PACKAGENAME" != "x" ] ; then
    if [ "x$HFILES" != "x" ] ; then
        trace cp $HFILES $PACKAGENAME/include
    fi
fi 

# exit if -C given
$NOCOMPILE


# compile .c files
INT3=
if [ "x$CFILES" != "x" ]; then
    trace $CC $CFLAGS -c $INCLUDES $CFILES
    remove $INT2
    remove $INT2h

    # following "tr" segments prevent sed from getting lines too long
    INT3=`echo "$CFILES " | tr ' ' '\012' | sed 's/\.c$/.o/g'`
    OFILES="$OFILES $INT3"
fi

# exit if -c given
$NOLINK


# link .o files together
if [ "x$OFILES" != "x" ]; then
    if [ "x$PACKAGENAME" != "x" ] ; then
        # Build a library
        LIB=lib`basename $PACKAGENAME`
        if [ $LTARG = "shared" ] ; then
	    # A bug on irix prevents compiling directly to the destination dir
            trace $CC $SLOPT $CFLAGS -o $LIB.so $OFILES $LFILES
	    trace mv $LIB.so $PACKAGENAME
        else
            trace ar rs $PACKAGENAME/$LIB.a $OFILES
        fi

        # The interpreter does not fully honor the umask.
        # Fix up permissions if we used the interpreter.
	if [ $TOBA = $ITOBA ] ; then
	    # This will set all bits not masked off by the umask
            find $PACKAGENAME -newer $TMP | xargs chmod +rwx
	fi
    else
        # Build a binary
        trace $LDFILTER $CC $CFLAGS -o $XFILE $OFILES $LDFLAGS $LFILES
    fi
    remove $INT3 
fi

