/*  runtime.c -- general runtime support routines  */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>

#include "toba.h"
#include "runtime.h"
#include "../gc/gc.h"
#include "sthreads.h"
#include "md.h"

#include "java_io_PrintStream.h"
#include "java_lang_Error.h"
#include "java_lang_NullPointerException.h"
#include "java_lang_Object.h"
#include "java_lang_String.h"
#include "java_lang_System.h"
#include "java_lang_Thread.h"
#include "java_lang_ThreadDeath.h"
#include "java_util_Hashtable.h"

#ifdef OPTION_JIT
/* Need this to register ClassRef structures so they're visible jit
 * for runtime linking. */
#include "toba_classfile_ClassRef.h"
#endif /* OPTION_JIT */

/* Set this nonzero after basic initialization done */
static int done_init = 0;

static void report(Object o);
static void walk_classes(Class c);
#ifdef OPTION_JIT
static void register_crefs(Class c);
#endif /* OPTION_JIT */
static void intern_strings(Class c);
static Object arglist(int argc, char *argv[]);

struct start_info {
    Class c;
    void (*main)();
    int argc;
    char **argv;
};

/* VM initialization */
void vminit() 
{
    md_init();				/* machine-specific initialization */
    initstructs();			/* initialize structures */
    monitor_package_init();		/* initialize monitor package */
    finalization_init();		/* initialize finalization routines */
    intern_string_init();               /* initialize string interning */
    sthread_init();			/* initialize threads package */
    meminit();				/* initialize memory system.
					 * Must happen after threads startup */
    loader_init();			/* initialize dynamic loading support */
}

/* VM initialization continues here after the call the sthread_launch */
void real_start(struct start_info *i) 
{
    /* Init the VM and API */
    vminit();
    walk_classes(i->c);			/* initialize API support */
#ifdef OPTION_JIT
    /* We need to register all the static ClassRef structures, so we don't
     * end up with multiple class structures associated with interfaces. */
    walk_classes (&cl_toba_classfile_ClassRef.C);
    initclass (&cl_toba_classfile_ClassRef.C);
    register_crefs(&cl_toba_classfile_ClassRef.C);
    register_crefs(i->c);
#endif /* OPTION_JIT */
    
    /* Done with all basic initialization; say so, so we can change
     * behavior in walk_classes and initclass. */
    done_init = 1;

    /* this thread runs main */
    start_thread(i->main, arglist(i->argc, i->argv));

    sthread_exit();                     /* main thread exits */
    terminate(0);			/* do final termination */
    return;				/* shouldn't get here */
}

/* Do any necessary initialization and call a 
 * routine to launch the primary thread. */

void vmstart(struct start_info *i) 
{
    /* Launching the initial thread may not return, 
     * but we need a place to wait for the main thread to
     * exit, so we launch through real_start */
    extern void membootstrap (void); /* alloc.c */
    membootstrap();			/* initial bootstrapping of the GC */
    sthread_launch(real_start, i);

    /*NOTREACHED*/
}

/*  start(c, main, argc, argv) -- initialize and start the inital thread
 *  running, which will call main() */

int start(Class c, Void main(Object o), int argc, char *argv[])
{ 
    /* Start is called only once, so this is safe */
    static struct start_info info;

    info.c = c;
    info.main = main;
    info.argc = argc;
    info.argv = argv;
   
    vmstart(&info);

    return 1;				/* shouldn't get here */
}



/*  walk_classes(c) -- perform load-time setup of all reachable classes  */

static void
walk_classes(Class c)
{
    int i;

    if (c == 0)                         /* If there aren't any classes */
	return;				/* to deal with here, leave. */

    if (c->flags & IS_SETUP)		/* if already processed */
    	return;		
    c->flags |= IS_SETUP;		/* mark so we don't revisit */

    /* Array classes do not have strlist entries.  I don't like segfaults,
     * so we won't try this on them.  (Strictly speaking, we shouldn't get
     * here with an array class, but the interpreter did pass one at
     * one point.) */
    if (! (IS_ARRAY & c->flags)) {
       intern_strings(c);			/* "intern" all strings */
       c->name = (Object *)c->strlist[0];	/* install interned class name */
#ifndef NDEBUG
    } else {
       fprintf (stderr, "INTERNAL WARNING: Unexpected array class in walk_classes.\n"
                "This should not happen and has not been debugged.");
#endif /* NDEBUG */
    }

    /* If there's a routine to initialize static final fields, call it.
     * This may call intern_string. */
    if (NULL != c->sfiinitializer) {
       c->sfiinitializer ();
    }

    if (! done_init) {
       /* All classes that are walked/registered during the basic
        * initialization phase are considered to have arrived through the
        * system class loader.  After that point, we can't register classes
        * here because we don't know where they came from; hopefully, they'll
        * be registered when they're loaded. */
       register_class (c);
    }

    for(i = 0; i < c->nsupers; i++)
	walk_classes(c->supers[i]);
    for(i = 0; i < c->ninters; i++)
	walk_classes(c->inters[i]);
    for(i = 0; i < c->nothers; i++)
	walk_classes(c->others[i]);

}

#ifdef OPTION_JIT
/* One thing we _do_ have to do is register the classref, so when we
 * lookup classes by name, e.g. when building interface lists, we get
 * the pre-compiled ones where necessary.  This calls into
 * toba.classfile.ClassRef.  Naturally, dependencies mean it can't be done
 * during the initial walk_classes, when things aren't set up yet. */
static void
register_crefs (Class c)
{
    int i;

    if (c == 0)                         /* If there aren't any classes */
	return;				/* to deal with here, leave. */

    if (c->flags & IS_SETUP2)		/* if already processed */
    	return;		
    c->flags |= IS_SETUP2;		/* mark so we don't revisit */

    if (NULL != c->clref) {
       RegisterClassRef_C_2Ejuw (c->clref);
    }

    for(i = 0; i < c->nsupers; i++)
	register_crefs(c->supers[i]);
    for(i = 0; i < c->ninters; i++)
	register_crefs(c->inters[i]);
    for(i = 0; i < c->nothers; i++)
	register_crefs(c->others[i]);

}
#endif /* OPTION_JIT */

/*  intern_strings(c) -- "canonicalize" all string constants of a class  */
static void
intern_strings(Class c)
{
    const void **p;
    const Char *start, *end;
    Object s;

    start = c->strpool;				/* start of first string */

    /* The class file loader does this work on its own,
     * so we don't need to in that case. */
    if (start == 0)
	return;

    for (p = c->strlist; *p != NULL; p++) {	/* for each slot */
	end = *p;				/* end of string */
	s = arraystring(start, end - start);	/* make into interned String */
	*p = s;					/* save in slot */
    	start = end;				/* update pointer for next str*/
    }
}



/*
 *  start_thread (func, arg) -- start a thread
 *
 *  At this point the thread should still be marked as unkillable.
 */
void
start_thread(Void (*startfunc)(Object o), Object arg)
{
    struct mythread *thr = mythread();
    struct in_java_lang_Thread *throbj;

    /* set up a default exception handler */
    thr->jmpbuf = thr->default_jmpbuf;
    thr->exception_count = 0;
    if (setjmp(thr->jmpbuf)) {		/* exception catcher of last resort */
	sthread_got_exception();
	if (thr->exception == NULL)
	    thr->exception = construct(&cl_java_lang_NullPointerException.C);
	if (instanceof(thr->exception, &cl_java_lang_ThreadDeath.C, 0)) 
	    return;			 /* ignore java.lang.ThreadDeath */
	if (thr->exception_count++ == 0) {  /* try using Java API just once */
	    report(thr->exception);	/* try reporting via API once */
            return;
	} else {
	    fprintf(stderr,"\nuncaught exception\n");
            return;
	}
    }

    /* once we have an exception handler we are killable */
    sthread_dontkill_end(thr);
    startfunc(arg);

    /* the thread is finished, upcall to his exit method */
    throbj = sthread_current()->obj;
    throbj->class->M.exit__5xUOu.f(throbj);
}

/*  report(o) -- report uncaught exception o  */

static void report(Object o)
{
    struct cl_java_lang_Object * clg;
    Object s;

    assert (NULL != o);
    clg = ((struct in_java_lang_Object *)o)->class;
    s = clg->M.toString__4d9OF.f(o);
    /* If this is a null pointer exception, and we know where it may have
     * been thrown, say so. */
    if ((&cl_java_lang_NullPointerException == (void *) clg) &&
        (NULL != npe_file_name)) {
       fprintf(stderr,"\nuncaught exception: %s (maybe at %s/%d)\n", cstring(s),
               npe_file_name, npe_file_line);
    } else {
       fprintf(stderr,"\nuncaught exception: %s\n", cstring(s));
    }
}



/*  arglist(argc, argv) -- construct Java arglist  */

static Object arglist(int argc, char *argv[])
{
    Object o, *a;
    int i;
    
    o = anewarray(&cl_java_lang_String.C, argc - 1);	/* alc String array */
    a = ((struct aarray *)o)->data;
    for (i = 1; i < argc; i++)		/* skip argv[0] */
	a[i - 1] = javastring(argv[i]);
    return o;
}



/*  terminate(status) -- perform any needed exit tasks, and exit  */

extern void exit (int);
void terminate(int status)
{
    memexit();				/* run finalizers */
    exit(status);			/* exit */
}



/*  fatal(mesg) -- immediate fatal error exit  */

void fatal(char *mesg)
{
    fprintf(stderr, "%s\n", mesg);
    exit(1);
}



/*  unimpl(name) -- immediate exit from unimplemented method  */

void unimpl(char *name)
{
    fprintf(stderr, "unimplemented method: %s\n", name);
    exit(1);
}

/*
 *  initclass(myclass) -- initialize class
 *
 *  Elsewhere, myclass->needinit is just checked for zero/nonzero as a boolean.
 *  In here, only, we distinguish between
 *	1  initialization needed
 *	2  initialization in progress
 */
void initclass(Class myclass)
{
    jmp_buf newbuf;
    struct mythread *thr;
    void *oldbuf;
    volatile int monitor_held;

    if (!myclass->needinit)		/* exit quickly if already initialized*/
	return;

    /* If needinit is negative, we never resolved this class. */
    if (0 > myclass->needinit) {
       throwInternalError ("initclass on unresolved class %s: %d\n",
                cstring (myclass->name), myclass->needinit);
    }

    thr = mythread();
    oldbuf = thr->jmpbuf;
    monitor_held = 0;
    if (setjmp(newbuf)) {		/* catch exceptions during init: */
        sthread_got_exception();
        if (monitor_held) {
	    myclass->needinit = 1;	/* mark as uninitialized again */
	    exitclass(myclass, thr, 0, &monitor_held);
        }
	thr->jmpbuf = oldbuf;		/* restore init caller's jump buffer */
	if (instanceof(thr->exception, &cl_java_lang_Error.C, 0))
	    athrow(thr->exception);	/* rethrow error */
	else				/* or mutate other exception */
	    throwExceptionInInitializerError(thr->exception);
    }
    thr->jmpbuf = newbuf;		/* register exception handler */

    /* synchronize vs. other threads */
    enterclass(myclass, thr, 1, &monitor_held);	
    if (!myclass->needinit) {		/* check again under lock */
	exitclass(myclass, thr, 0, &monitor_held);
	thr->jmpbuf = oldbuf;
	return;
    }

    if (myclass->needinit > 1) {
        /* if init is already in progress, must be same thread; just return */
	exitclass(myclass, thr, 0, &monitor_held);
	thr->jmpbuf = oldbuf;
	return;
    }

    myclass->needinit++;		/* indicate initialization in progress*/

    if (myclass->nsupers > 1)		/* if there is a superclass */
	initclass(myclass->supers[1]);	/* initialize it */

    /* ensure proper setup in case class was not reachable from main class */
    /* (this can happen if it's called only from native code) */
    if (!(myclass->flags & IS_SETUP)) {	/* if not yet setup & registered */
    	walk_classes(myclass);		/* intern strings & register class */
    }
#ifdef OPTION_JIT
    if ((!(myclass->flags & IS_SETUP2)) &&
        done_init) {
        /* After initialization, we know the ClassRef class has been
         * initialized, so it's ok to register class references.  During
         * init, we register references in a separate step. */
        register_crefs (myclass); /* register classrefs */
    }
#endif /* OPTION_JIT */

    if (myclass->initializer)		/* if there is a <clinit> */
	myclass->initializer();		/* call it */

    myclass->needinit = 0;		/* indicate initialization complete */
    thr->jmpbuf = oldbuf;		/* restore previous exception handler */

    /* release lock */
    exitclass(myclass, thr, 0, &monitor_held);
}



/*
 * cstring(S) -- return a C string corresponding to the java string.
 *
 * characters are truncated to 8 bits; result string is in allocated memory.
 */
char *cstring(const struct in_java_lang_String *str)
{
    struct carray *value = (struct carray *)str->value;
    int end, i, j;
    char *cstr;

    cstr = (char *)allocate(str->count + 1);

    end = str->offset + str->count;
    j = 0;
    for(i = str->offset; i < end; i++) {
	cstr[j++] = value->data[i] & 0xff;
    }
    cstr[j] = '\0';

    return cstr;
}



/* javastring(s) -- return a java String class for the ascii C string  */

struct in_java_lang_String *javastring(const char *cstr)
{
    struct in_java_lang_String *str;
    struct carray *value;
    int len, i;

    len = strlen(cstr);
    value = anewarray(&cl_char, len);
    for(i = 0; i < len; i++) {
	value->data[i] = cstr[i] & 0xff;
    }

    str = (struct in_java_lang_String *)construct(&cl_java_lang_String.C);
    str->value = value;
    str->offset = 0;
    str->count = len;

    return intern_string (str);
}

/* arraystring(a, n) -- return a java String class from array of n Chars  */

struct in_java_lang_String *arraystring(const Char *a, int n)
{
    struct in_java_lang_String *str;
    struct carray *value;

    value = anewarray(&cl_char, n);
    memcpy(value->data, a, n * sizeof(Char));

    str = (struct in_java_lang_String *)construct(&cl_java_lang_String.C);
    str->value = value;
    str->offset = 0;
    str->count = n;

    return intern_string (str);
}

/*  println(o) == System.out.println(o)  (call from dbx to print an object)  */

void println(Object o)
{
    initclass(&cl_java_lang_System.C);
    println_O_n6qWU(cl_java_lang_System.V.out, o);
}

