/* Template to use to replace I/O system calls with a wrapper that
 * keeps blocking I/O from blocking all of the user-level threads */

/* This template uses 4 macros to define the system call:
 * SYSCALLNAME - the name of the system call
 * ARGDECL     - the declaration of the arguments to the call
 * ARGCALL     - the actual arguments to the call
 * RETTYPE     - the return type of the system call
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <errno.h>
#include <stdio.h>

extern long syscallWrapIOPending;

/* fd is assumed to be an open file descriptor */
RETTYPE
__wrap_SYSCALLNAME(ARGDECL) 
{

    int flags = fdescGetFlags(fd);

    if (flags < 0) {
	if ( !fdescIsOpen(fd) ) {
	    /* Only for the standard IO files. */
	    setupFileDesc(fd, 0);
	    flags = 0;
	} else {
	    errno = -flags;
	    return -1;
	}
    }

    /* If the user wants non-blocking IO, then just do the 
     * system call. */
    if ((flags & O_NDELAY) != 0)
    {
	return __real_SYSCALLNAME(ARGCALL);
    } else {
	int res;
 
        syscallWrapIOPending++;

	/* Fake blocking I/O using non-blocking I/O so 
	 * that only the current thread blocks */
        while (1) {
	    res = __real_SYSCALLNAME(ARGCALL);
	    if ((res == -1) && 
		((errno == EWOULDBLOCK) || 
                 (errno == EINPROGRESS) ||
		 (errno == EALREADY)) ) {
		fdescWait(fd);
	    } else {
	        break;
	    }
        }

        syscallWrapIOPending--;

        return res;
    }
}
