
#include <signal.h>
#include <sys/time.h>

#include "scout_thread.h"
#include "scout_semaphore.h"
#include "scout_thread_toba.h"

extern void *GC_malloc(int n);

#define handlerresulttype void
typedef handlerresulttype (*handlertype)();

struct threadTimeout {
    struct threadTimeout *next;
    int go_at_tick;
    Thread thread;
};

static struct threadTimeout timeQueue;
static struct Semaphore mutex;
static struct itimerval ClockInterval;

volatile unsigned long timeNow;

static void
set_clock (void)
{
    setitimer (ITIMER_REAL, &ClockInterval, NULL);
}
  
static void
threadSetClock(long interval)
{
    ClockInterval.it_value.tv_sec = interval / 1000000;
    ClockInterval.it_value.tv_usec = interval % 1000000;
    ClockInterval.it_interval.tv_sec = interval / 1000000;
    ClockInterval.it_interval.tv_usec = interval % 1000000;
    set_clock();
}

static int
definehandler(sig, handler)
     int sig;
     handlertype handler;
{
    struct sigaction act;
    
    if (handler == SIG_IGN) {
        sigemptyset (&act.sa_mask);
	act.sa_handler = SIG_IGN;
	act.sa_flags = 0;
	sigaction(sig, &act, NULL);
    } else {
	act.sa_handler = handler;
	sigfillset(&act.sa_mask);
	act.sa_flags = SA_RESTART;
	sigaction (sig, &act, NULL);
    }
    return 0;
}

static void
clock_tick (int sig)
{
    struct threadTimeout *n, *t;

    timeNow++;

    /* semaphoreWait (&mutex); */
    for (n = timeQueue.next; n != NULL; ) {
	if (n->go_at_tick <= timeNow) {
	    threadAsyncWakeup (n->thread);
	    t = n;
	    n = n->next;
	    t->next = NULL;  /* for garbage collector */
	} else {
	    break;
	}
    }
    timeQueue.next = n;

#ifdef USER_LEVEL
    syscallWrapIOCheck();
#endif

    /* semaphoreSignal (&mutex); */
}

static void
init_clock( interval )
     long        interval;
{
    struct sigaction act;

    /* definehandler(SIGPIPE, SIG_IGN);
     * definehandler(SIGINT, (handlertype) sig_int_handler);
     * definehandler(SIGIO,  (handlertype) dispatch);
     * definehandler(SIGURG,  (handlertype) dispatch);
     */

    definehandler (SIGALRM, clock_tick);

    threadSetClock(interval);
    return;
}

void
threadTimerReset(void)
{
    init_clock(_TIMER_GRANULARITY * 1000);
}

void 
threadTimerInit (void)
{
    timeNow = 0;
    semaphoreInit (&mutex, 1);
    threadTimerReset();
}

static void
timedSuspend(int ticks)
{
    struct threadTimeout *t, *n, *p;

    t = GC_malloc (sizeof(*t));
    t->go_at_tick = timeNow + ticks;
    t->thread = tSelf;

    enable_alarm (0);

    for (p = &timeQueue, n = p->next; ; p = n, n = n->next) {
	if (n == NULL || n->go_at_tick >= t->go_at_tick) {
	    p->next = t;
	    t->next = n;
	    break;
	}
    }

    enable_alarm (1);

    threadSuspend();
}

void
threadSleep (long millis)
{
    int ticks;

    /* Sleep of 0 is a yield() */
    if (millis == 0)
	threadYield();
#if 1
    else if (millis < _TIMER_GRANULARITY) {
	struct timeval t;
	t.tv_sec = 0;
	t.tv_usec = millis * 1000;
        (void)select(0, 0, 0, 0, &t);
    }
#endif
    else { 
        ticks = (millis>>_TIMER_SHIFT) + 1;
        timedSuspend(ticks);
    }
}





