/*****************************************************************************
 *
 * Microchip DeviceNet Stack (Multicast Polled Messaging Connection Object Source)
 *
 *****************************************************************************
 * FileName:        conn5.c
 * Dependencies:    
 * Processor:       PIC18F with CAN
 * Compiler:       	C18 02.20.00 or higher
 * Linker:          MPLINK 03.40.00 or higher
 * Company:         Microchip Technology Incorporated
 *
 * Software License Agreement
 *
 * The software supplied herewith by Microchip Technology Incorporated
 * (the "Company") is intended and supplied to you, the Company's
 * customer, for use solely and exclusively with products manufactured
 * by the Company. 
 *
 * The software is owned by the Company and/or its supplier, and is 
 * protected under applicable copyright laws. All rights are reserved. 
 * Any use in violation of the foregoing restrictions may subject the 
 * user to criminal sanctions under applicable laws, as well as to 
 * civil liability for the breach of the terms and conditions of this 
 * license.
 *
 * THIS SOFTWARE IS PROVIDED IN AN "AS IS" CONDITION. NO WARRANTIES, 
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED 
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A 
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE COMPANY SHALL NOT, 
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR 
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 *
 * This file contains Multicast Polled messaging support for the Connection Object 
 * described in Section 5-4 and Chapter 7 of Volume 1 of the DeviceNet 
 * specification. Additional support is in the UsrConn.c file.
 * 
 *
 *
 * Author               Date        Comment
 *~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 * Ross Fosler			04/28/03	...	
 * 
 *****************************************************************************/

#include	"dnet.def"			// Global definitions file

#include 	"typedefs.h"

#if SUPPORT_MULTICAST_POLL

#include	"conn.h"			// Connection prototypes and symbols

#include	"services.h"		// Service codes
#include	"errors.h"			// Error codes
#include	"class.h"			// Class codes

#include	"route.h"			// Global symbols defined by the router
#include	"dnet.h"			// DeviceNet prototypes and symbols
#include	"frag.h"			// Fragmentation control
#include	"CAN.h"


#pragma	udata
/*********************************************************************
 * Connection related variables
 ********************************************************************/
CONN 	uConn5;

//unsigned char	uConn5RxBuffer[CONN_MULTICAST_RX_SIZE];
//unsigned char	uConn5TxBuffer[CONN_MULTICAST_TX_SIZE];



/*********************************************************************
 * Function:        unsigned char _Conn5Create(void)
 *
 * PreCondition:    The CAN (or other) driver must be prepared to 
 *					accept some filter settings.
 *
 * Input:       	none	
 *                  
 * Output:      	unsigned char handle to the connection	
 *
 * Side Effects:    none
 *
 * Overview:        This function creates a connection 
 *					in the predefined set and returns a 
 *					handle to the connection. The connection is 
 *					placed into a 'configuring' state. 
 *					
 *					This function will call out to the users code
 *					for additional resource allocation. If the 
 *					user returns NULL then the allocation will not
 *					occur.
 *
 * Note:            This function is not called directly by 
 *					application code.
 ********************************************************************/
unsigned char _Conn5Create()
{
	if (UsrConnCreateEvent(5))
	{
		//Initialize the connection attributes
		uConn5.attrib.state = _STATE_CONFIGURING;
		uConn5.attrib.transportClass.byte = 0x83;
		uConn5.attrib.produced_cid.bytes.MSB = (uDNet.MACID >> 3) | (0xC << 3);
		uConn5.attrib.produced_cid.bytes.LSB = uDNet.MACID << 5;
		uConn5.attrib.expected_packet_rate.word = 0;
		
		_existentFlags.bits.multi = 1;
					
		// Setup the pointer and other info for the receiving side
		uConn5.rx.lenMax = uConn5.attrib.consumed_con_size.bytes.LSB;
		uConn5.rx.fragState = 0;
		uConn5.rx.oldFrag = 0;
		
		// Setup the pointer and other info for the transmitting side
		uConn5.tx.lenMax = uConn2.attrib.produced_con_size.bytes.LSB;
		uConn5.tx.fragState = 0;
		uConn5.tx.oldFrag = 0;				
					
		return (5);
	}
	else return(0);
}


/*********************************************************************
 * Function:        unsigned char _Conn5Close()
 *
 * PreCondition:    The connection should have already been open.
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char instance that closed
 *
 * Side Effects:    none
 *
 * Overview:        Closes the specified connection. Users code is 
 *					notified to release any used resources.
 *
 * Note:            none
 ********************************************************************/
unsigned char _Conn5Close()
{
	// Transition to the non-existent state
	uConn5.attrib.state = _STATE_NON_EXISTENT;
	
	_establishFlags.bits.multi = 0;
	_existentFlags.bits.multi = 0;
	
	// Issue a request to the driver to stop receiving the message
	CANClrFilter(uConn5.attrib.consumed_cid.word);

	UsrConnCloseEvent(5);

	return(5);
}




/*********************************************************************
 * Function:        void _Conn5TimerEvent(void)
 *
 * PreCondition:    None
 *
 * Input:       	None	
 *                  
 * Output:      	None
 *
 * Side Effects:    None
 *
 * Overview:        Update timer and process any timer events. If the 
 *					timer overflows then the state of the connection
 *					is changed.
 *
 * Note:            None
 ********************************************************************/
void _Conn5TimerEvent(void)
{	
	// Process the watchdog if the packet rate is other than 0
	if (uConn5.attrib.expected_packet_rate.word)
	{
		// Adjust the time
		uConn5.timer.word -= TICK_RESOLUTION;

		// If the wdt expires then change the state of the connection
		if (uConn5.timer.word == 0) 
		{
			uConn5.attrib.state = _STATE_TIMED_OUT;
			_establishFlags.bits.multi = 0;
		}
	}   
}




/*********************************************************************
 * Function:        void _Conn5RxEvent(void)
 *
 * PreCondition:    none
 *
 * Input:       	none
 *                  
 * Output:      	none
 *
 * Side Effects:    none
 *
 * Overview:        Process received data for this connection.
 *
 * Note:            This event occures when data has been received
 *					for this connection instance.
 ********************************************************************/
void _Conn5RxEvent(void)
{
	#if CONN_MULTICAST_RX_FRAG	
	// If fragmented
	if (uConn5.attrib.consumed_con_size.word > 8)
	{
		// Process the fragment
		_mFragNoAckConsume(uConn5.rx);

		// Test the status of the process	
		if (uConn5.rx.fragFlags.bits.b0)
		{
			// Reset the fragment finish flag
			uConn5.rx.fragFlags.bits.b0 = 0;
		
			// Reset the connection wdt
			uConn5.timer.word = uConn5.attrib.expected_packet_rate.word << 2;

			// Indicate message has been received
			_rxFlag.bits.multi = 1;
			
			// Notify the application
			UsrConnRxDataEvent(5);
		}
	}
	else
	#endif

	// else process a non-fragmented message
	{ 
		uConn5.rx.len = CANGetRxCnt();

		// Copy the message to the connection buffer
		CANGetRxDataTyp0(uConn5.rx.pMsg);
		//*((_MSG_DATA *)(uConn5.rx.pMsg)) = *((_MSG_DATA *)CANGetRxDataPtr());
		    
		// Reset the connection wdt
		uConn5.timer.word = uConn5.attrib.expected_packet_rate.word << 2;	
	
		// Indicate message has been received (located in conn.c)	
		_rxFlag.bits.multi = 1;
		
		// Notify the application
		UsrConnRxDataEvent(5);
	}	 

	// Release the hardware to continue receiving
	CANRead();	
}




/*********************************************************************
 * Function:        void _Conn5TxOpenEvent(void)
 *
 * PreCondition:    A transmit request must have been made.
 *
 * Input:       	none	
 *                  
 * Output:      	none
 *
 * Side Effects:    none
 *
 * Overview:        Process open transmit que event
 *
 * Note:            This event occurs when the buffer is available 
 *					for this connection instance to transmit. A 
 *					transmit request must have been made to enter 
 *					this function.
 ********************************************************************/
void _Conn5TxOpenEvent(void)
{
	// Set the produced CID
   	CANPutTxCID(uConn5.attrib.produced_cid.word);

	#if CONN_MULTICAST_TX_FRAG		
	// If fragmented
	if (uConn5.attrib.produced_con_size.word > 8)
	{
		// Process the fragment
		_mFragNoAckProduce(uConn5.tx);		
		
		// Request the hardware to queue the message to send
		CANSend(5);
		
		// Test the status of the process		
		if (uConn5.tx.fragFlags.bits.b0)
		{			
			//Clear the transmit flag to open access to the write buffer
			_txFlag.bits.multi = 0;
		}
	}
	else
	#endif

	// else process a non-fragmented message
	{
		// Copy the message to the hardware buffer
		CANPutTxDataTyp0(uConn5.tx.pMsg);
		//*((_MSG_DATA *)CANGetTxDataPtr()) = *((_MSG_DATA *)(uConn5.tx.pMsg));

		// Set the length of the message
		CANPutTxCnt(uConn5.tx.len);

		// Request the hardware to queue the message to send
		CANSend(5);

		//Clear the transmit flag to open access to the write buffer
		_txFlag.bits.multi = 0;
	}
}


/*********************************************************************
 * Function:        void _Conn5TxEvent(void)
 *
 * PreCondition:    Data must have been queued to transmit.
 *
 * Input:       	none	
 *                  
 * Output:      	none
 *
 * Side Effects:    none
 *
 * Overview:        Process data for this connection.
 *
 * Note:            This event occurs when the buffer has successfully
 *					placed the requested data on the bus.
 ********************************************************************/
void _Conn5TxEvent(void)
{
	#if CONN_MULTICAST_TX_FRAG		
	// If fragmented
	if (uConn5.attrib.produced_con_size.word > 8)
	{
		// Test the status of the process		
		if (uConn5.tx.fragFlags.bits.b0)
		{
			// Reset finish flag to be captured later
			uConn5.tx.fragFlags.bits.b0 = 0;
			
			// Set flag indicating data has been placed on the bus
 			_txFinFlags.bits.multi = 1; 	
 			
 			// Notify the application
 			UsrConnTxDataEvent(5);
		}
	}
	else
	#endif
	{
		// Set flag indicating data has been placed on the bus
 		_txFinFlags.bits.multi = 1; 
 		
 		// Notify the application
 		UsrConnTxDataEvent(5);
 	}
}





/*********************************************************************
 * Function:        unsigned char _Conn5ExplicitEvent(void)
 *
 * PreCondition:    none
 *
 * Input:       	none	
 *                  
 * Output:      	unsigned char success of the request
 *
 * Side Effects:    none
 *
 * Overview:        Handle explicit messaging for this instance
 *
 * Note:            None
 ********************************************************************/
unsigned char _Conn5ExplicitEvent(void)
{
	switch(mRouteGetServiceID())
   	{
   		case SRVS_GET_ATTRIB_SINGLE:
   			return (_Conn5GetAttrib());
   		case SRVS_SET_ATTRIB_SINGLE:
   			return (_Conn5SetAttrib());
   	   	default:
   			mRoutePutError(ERR_SERVICE_NOT_SUPPORTED);
   			break;
   	}
		
	return (1);
}


/*********************************************************************
 * Function:        unsigned char _Conn5GetAttrib()
 *
 * PreCondition:    none
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char success of the request
 *
 * Side Effects:    none
 *
 * Overview:        Handle explicit messaging
 *
 * Note:            None
 ********************************************************************/
unsigned char _Conn5GetAttrib(void)
{
	UINT	work;
	USINT 	i;

	switch (mRouteGetAttributeID())
	{
		case	_ATTRIB_STATE:
			mRoutePutByte(uConn5.attrib.state);
			break;
		case	_ATTRIB_INSTANCE_TYPE:
			mRoutePutByte(1);
			break;
		case	_ATTRIB_CLASS_TRIGGER:
			mRoutePutByte(uConn5.attrib.transportClass.byte);
			break;
		case 	_ATTRIB_PRODUCED_CID:
			work.word = (uConn5.attrib.produced_cid.word >> 5);
			mRoutePutByte(work.bytes.LSB);
			mRoutePutByte(work.bytes.MSB);
			break;
		case	_ATTRIB_CONSUMED_CID:
			work.word = (uConn5.attrib.consumed_cid.word >> 5);
			mRoutePutByte(work.bytes.LSB);
			mRoutePutByte(work.bytes.MSB);
			break;	
		case 	_ATTRIB_INITIAL_COMM_CHAR:
			mRoutePutByte(0x01);
			break;
		case	_ATTRIB_PRODUCED_CONN_SIZE:
			mRoutePutByte(uConn5.attrib.produced_con_size.bytes.LSB);
			mRoutePutByte(uConn5.attrib.produced_con_size.bytes.MSB);
			break;
		case	_ATTRIB_CONSUMED_CONN_SIZE:
			mRoutePutByte(uConn5.attrib.consumed_con_size.bytes.LSB);
			mRoutePutByte(uConn5.attrib.consumed_con_size.bytes.MSB);
			break;
		case	_ATTRIB_EXPECTED_RATE:
			mRoutePutByte(uConn5.attrib.expected_packet_rate.bytes.LSB);
			mRoutePutByte(uConn5.attrib.expected_packet_rate.bytes.MSB);
			break;
		case	_ATTRIB_WDT_ACTION:
			mRoutePutByte(0x00);
			break;
		case	_ATTRIB_PRODUCED_CONN_PATH_LEN:
			mRoutePutByte(uConn5.attrib.produced_path_len.bytes.LSB);
			mRoutePutByte(uConn5.attrib.produced_path_len.bytes.MSB);
			break;
		case	_ATTRIB_PRODUCED_CONN_PATH:
			if (mRouteOutLen() > uConn5.attrib.produced_path_len.bytes.LSB)  
			{
				for (i = 0; i < (uConn5.attrib.produced_path_len.bytes.LSB); i++)
				{
					mRoutePutByte(uConn5.attrib.produced_path[i]);
				}
			}
			else
			{
   				mRoutePutError(ERR_REPLY_TOO_LARGE);
			}
			break;
		case	_ATTRIB_CONSUMED_CONN_PATH_LEN:
			mRoutePutByte(uConn5.attrib.consumed_path_len.bytes.LSB);
			mRoutePutByte(uConn5.attrib.consumed_path_len.bytes.MSB);
			break;
		case	_ATTRIB_CONSUMED_CONN_PATH:
			if (mRouteOutLen() > uConn5.attrib.consumed_path_len.bytes.LSB)  
			{
				for (i = 0; i < (uConn5.attrib.consumed_path_len.bytes.LSB); i++)
				{
					mRoutePutByte(uConn5.attrib.consumed_path[i]);
				}
			}
			else
			{
				mRoutePutError(ERR_REPLY_TOO_LARGE);
			}
			break;
		default:
			mRoutePutError(ERR_ATTRIB_NOT_SUPPORTED);
			break;
	}
	return(1);
}




/*********************************************************************
 * Function:        unsigned char _Conn5SetAttrib(void)
 *
 * PreCondition:    none
 *
 * Input:       	none
 *                  
 * Output:      	unsigned char success of the request
 *
 * Side Effects:    none
 *
 * Overview:        Handle explicit messaging
 *
 * Note:            None
 ********************************************************************/
unsigned char _Conn5SetAttrib(void)
{
	UINT 	work;

	// Ignore the first byte (it is actually the attribute ID) 
	mRouteGetByte();

	switch (mRouteGetAttributeID())
	{	
		case	_ATTRIB_EXPECTED_RATE:
			// Continue only if there is sufficient data
			if (mRouteTestValidInputDataLen(2))
			{
				if (uConn5.attrib.state == _STATE_CONFIGURING) 
				{
					uConn5.attrib.state = _STATE_ESTABLISHED;
					_establishFlags.bits.multi = 1;
				}

				// Read in the requested packet rate
				uConn5.attrib.expected_packet_rate.bytes.LSB = mRouteGetByte();
				uConn5.attrib.expected_packet_rate.bytes.MSB = mRouteGetByte();
	
				// Get the ls bits
				work.word = uConn5.attrib.expected_packet_rate.bytes.LSB & (TICK_RESOLUTION - 1);
	
				// Remove the ls bits from desired resolution
				uConn5.attrib.expected_packet_rate.bytes.LSB &= (~(TICK_RESOLUTION - 1));
				
				// Round up if necessary
				if (work.word) uConn5.attrib.expected_packet_rate.word += (TICK_RESOLUTION);
	
				// Return the value actually used
				mRoutePutByte(uConn5.attrib.expected_packet_rate.bytes.LSB);
				mRoutePutByte(uConn5.attrib.expected_packet_rate.bytes.MSB);
	
				// Set the timer 4x (section 5-4.4.2)
				uConn5.timer.word = uConn5.attrib.expected_packet_rate.word << 2; 
			}	
			break;
		case	_ATTRIB_CLASS_TRIGGER:
			if (uConn5.attrib.state == _STATE_CONFIGURING)
			{
				if (mRouteTestValidInputDataLen(1))
				{ 
					// To be handled by the app
					UsrConnSetAttribEvent(5);
				}
			}
			else
			{
				mRoutePutError(ERR_OBJECT_STATE_CONFLICT);
			}
			break;
		case	_ATTRIB_PRODUCED_CONN_PATH:
			if (uConn5.attrib.state == _STATE_CONFIGURING)
			{
				if (mRouteTestValidInputDataLen(uConn4.attrib.produced_path_len.bytes.LSB))
				{
					// To be handled by the app
					UsrConnSetAttribEvent(5);
				}
			}
			else
			{
				mRoutePutError(ERR_OBJECT_STATE_CONFLICT);
			}
			break;
		case	_ATTRIB_CONSUMED_CONN_PATH:
			if (uConn5.attrib.state == _STATE_CONFIGURING)
			{
				if (mRouteTestValidInputDataLen(uConn5.attrib.consumed_path_len.bytes.LSB))
				{
					// To be handled by the app
					UsrConnSetAttribEvent(5);
				}
			}
			else
			{
				mRoutePutError(ERR_OBJECT_STATE_CONFLICT);
			}
			break;
		case	_ATTRIB_PRODUCED_CONN_SIZE:
			if (uConn5.attrib.state == _STATE_CONFIGURING)
			{
				if (mRouteTestValidInputDataLen(2))
				{
					// To be handled by the app
					UsrConnSetAttribEvent(5);
				}
			}
			else
			{
				mRoutePutError(ERR_OBJECT_STATE_CONFLICT);
			}
			break;
		case	_ATTRIB_CONSUMED_CID:
			if (mRouteTestValidInputDataLen(2))
			{
				work.bytes.LSB = mRouteGetByte();
				work.bytes.MSB = mRouteGetByte();
				
				work.word = work.word << 5;
				
				if (((work.bytes.MSB & 0xC0) == 0x80) && 
					((work.bytes.MSB & 0x3F) < 64) &&
					((work.bytes.LSB & 0xE0) == 0))
				{
					uConn5.attrib.consumed_cid.word = work.word;
					CANSetFilter(work.word);
				}
				else
				{
					mRoutePutError(ERR_INVALID_ATTRIB_VALUE);
				}
			}
			break;
		case	_ATTRIB_WDT_ACTION:
		case	_ATTRIB_CONSUMED_CONN_SIZE:
		case	_ATTRIB_STATE:
		case	_ATTRIB_INSTANCE_TYPE:
		case 	_ATTRIB_PRODUCED_CID:
		case 	_ATTRIB_INITIAL_COMM_CHAR:
		case	_ATTRIB_PRODUCED_CONN_PATH_LEN:
		case	_ATTRIB_CONSUMED_CONN_PATH_LEN:
			mRoutePutError(ERR_ATTRIB_NOT_SETTABLE);
			break;
			
		default:
			mRoutePutError(ERR_ATTRIB_NOT_SUPPORTED);
			break;
	}
	return(1);
}



#endif	// SUPPORT_MULTICAST_POLL
