/*
 * Copyright (c) 1985, 1988 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *	  must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.	IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#if defined(LIBC_SCCS) && !defined(lint)
/*static char *sccsid = "from: @(#)gethostbyname.c	6.45 (Berkeley) 2/24/91";*/
static char *rcsid = "$Id: gethostbyname.c,v 1.8 1995/03/27 10:35:36 ghudson Exp $";
#endif /* LIBC_SCCS and not lint */

#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <netdb.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <resolv.h>
#include "res_internal.h"

static struct hostent *fake_hostent(const char *hostname, struct in_addr addr,
									struct hostent_answer *result);
static struct hostent *file_find(const char *name, struct res_data *data,
								 struct hostent_answer *result);

struct hostent *gethostbyname(const char *hostname)
{
	struct res_data *data = _res_init();

	if (!data)
		return NULL;
	if (!data->host_answer) {
		data->host_answer = malloc(sizeof(struct hostent_answer));
		if (!data->host_answer) {
			data->errval = NO_RECOVERY;
			return NULL;
		}
	}
	return gethostbyname_r(hostname, data->host_answer);
}

struct hostent *gethostbyname_r(const char *hostname,
								struct hostent_answer *result)
{
	struct res_data *data = _res_init();
	struct in_addr addr;
	querybuf buf;
	const char *p;
	int n;
	
	if (!data)
		return NULL;
	
	/* Check for all-numeric hostname with no trailing dot. */
	if (isdigit(hostname[0])) {
		p = hostname;
		while (*p && (isdigit(*p) || *p == '.'))
			p++;
		if (!*p && p[-1] != '.') {
			/* Looks like an IP address; convert it. */
			if (inet_aton(hostname, &addr) == -1) {
				data->errval = HOST_NOT_FOUND;
				return NULL;
			}
			return fake_hostent(hostname, addr, result);
		}
	}
	
	/* Do the search. */
	n = res_search(hostname, C_IN, T_A, buf.buf, sizeof(buf));
	if (n >= 0)
		return _res_parse_answer(&buf, n, 0, result, data);
	else if (errno == ECONNREFUSED)
		return file_find(hostname, data, result);
	else
		return NULL;
}

static struct hostent *fake_hostent(const char *hostname, struct in_addr addr,
									struct hostent_answer *result)
{
	strncpy(result->name, hostname, BUFSIZ - 1);
	result->name[BUFSIZ - 1] = 0;
	result->host.h_name = result->name;
	
	result->host_addr = addr;
	result->h_addr_ptrs[0] = (char *) &result->host_addr;
	result->h_addr_ptrs[1] = NULL;
	result->host.h_addr_list = result->h_addr_ptrs;
	result->host.h_length = sizeof(unsigned long);
	
	result->host_aliases[0] = NULL;
	result->host.h_aliases = result->host_aliases;
	
	result->host.h_addrtype = AF_INET;
	
	return &result->host;
}

static struct hostent *file_find(const char *name, struct res_data *data,
								 struct hostent_answer *result)
{
	struct hostent *entry;
	char **alias;
	FILE *fp = NULL;
	
	sethostent_r(&fp);
	while (1) {
		
		/* Get a host entry friom the file. */
		entry = gethostent_r(result, &fp);
		if (!entry)
			break;
		
		/* Check the name and aliases. */
		if (strcasecmp(entry->h_name, name) == 0)
			break;
		for (alias = entry->h_aliases; *alias; alias++) {
			if (strcasecmp(alias, name) == 0)
				break;
		}
		if (*alias)
			break;
	}
	endhostent_r(&fp);
	
	if (!entry)
		data->errval = HOST_NOT_FOUND;
	return entry;
}

