/*
 * KCLL -  Ken and Chee's Limey Lisp  
 * All Rights to the code and any products created with this code is 
 * hereby granted. 
 *
 * I.E. You can do whatever the hell you want with this code. 
 * The only restriction is that this copyright notice not be modified.
 */
 
#include <varargs.h>
#include "stream.h"
#include "string.h"
#include "num.h"
#include "cont.h"

/* Functions:

   LLStream *make_stream_to_file (file) 
   int llread_stream (stream)
   llunread_stream (stream, char)
   llwrite_stream (stream, char)
   llstrprintf (stream, format, ...)
   llinit_stream ()
*/   

LLTag llstream_t;
LLStream *s_stdin;
LLStream *s_stdout;
LLStream *s_stderr;

/* Create and return a stream object, which will provide access to the
   given file */
LLStream *llmake_stream_to_file (file)
     FILE *file;
{
  LLStream *out;

  out = (LLStream *) llmake_obj(llstream_t);
  out->type = LLFILE;
  out->data.file = file;
  out->buffered_chars = 0;
  return out;
}

LLStream *llmake_stream_from_chars(str)
     char *str; 
{
  LLStream *out; 
  out = (LLStream *) llmake_obj(llstream_t);
  out->type = LLCSTRING;
  out->data.string = str;
  out->buffered_chars = 0;
  return(out);
}

int llread_stream (stream)
     LLStream *stream;
{
  if(stream->buffered_chars) {
    return ((int)stream->look_ahead[--(stream->buffered_chars)]);
  }
  switch (stream->type) { 
  case LLFILE:
    return fgetc(stream->data.file); 
  case LLCSTRING:
    return *(stream->data.string++);
  default: 
    return(NULL);
  }
}

void llunread_stream (stream, chr)
     LLStream *stream;
     unsigned char chr;
{
  stream->look_ahead[stream->buffered_chars++] = chr;
}

void llwrite_stream (stream, chr)
     LLStream *stream;
     char chr;
{
  switch (stream->type) {
  case LLFILE:
    fputc (chr, stream->data.file);
    break; 
  case LLCSTRING:
    llerror(LLREAD_ONLY_STREAM);
    break; 
  }
}

void free_stream (stream)
     LLStream *stream;
{
  switch(stream->type) {
  case LLFILE:
    fclose(stream->data.file);
    break;
  }
}

char *llcfgets(stream) 
     LLStream *stream; 
{
  char buf[10240];
  if (stream->type != LLFILE) llerror(LLNOT_A_FILE);
  return(fgets(buf, 10240, stream->data.file));
}

void llread_chars()
{
  LLStream *stream; 
  char *str;
  int size;
  stream = (LLStream *)llnextcheckedarg(llstream_t);
  lllastarg();
  str = llcfgets(stream); 
  size = strlen(str); 
  if(size && str[size-1] == '\n') size--;
  llpusharg(llchars_to_string(str, size));
}


void llstrprintf (va_alist) va_dcl
{
  va_list ap;
  LLStream *stream;
  char *format;
  
  va_start(ap);
  stream = va_arg(ap, LLStream *);
  format = va_arg(ap, char *);
  switch(stream->type) {
  case LLFILE:
    /*_doprnt (format, ap,stream->data.file);*/
    vfprintf(stream->data.file, format, ap);
    break; 
  case LLCSTRING:
    llerror(LLREAD_ONLY_STREAM);
    break;
  }
}

void llcstrflush(stream)
LLStream *stream; 
{
  switch(stream->type) {
  case LLFILE:
    fflush(stream->data.file);
    break; 
  }
}

void llcfopen(type) 
     LLFileType type; 
{
  LLString *string; 
  FILE *fptr;
  char *file, *opentype; 
  
  string = (LLString *)llnextcheckedarg(llstring_t);
  lllastarg();
  file = (char *) malloc(sizeof(char) * string->length + 1);
  bzero(file, string->length + 1);
  bcopy(string->text, file, string->length);
  switch(type) {
  case LLINPUTFILE:	opentype = "rb"; break; 
  case LLOUTPUTFILE:	opentype = "wb"; break;   
  case LLAPPENDFILE:	opentype = "ab"; break; 
  case LLUPDATEFILE:	opentype = "r+"; break; 
  }
  if ((fptr = fopen(file, opentype)) == NULL) {
    free(file);
    llerror(LLCANT_OPEN);
  }
  free(file);
  llpusharg(llmake_stream_to_file(fptr));
}

void llclose_port() 
{
  LLStream *stream; 
  stream = (LLStream *) llnextcheckedarg(llstream_t); 
  lllastarg();
  if(stream->type != LLFILE) llerror(LLNOT_A_FILE);
  fclose(stream->data.file); 
  llpusharg(T); 
}

void llopen_input_file() { llcfopen(LLINPUTFILE); }
void llopen_output_file() { llcfopen(LLOUTPUTFILE); }
void llopen_append_file() { llcfopen(LLAPPENDFILE); }
void llopen_update_file() { llcfopen(LLUPDATEFILE); }

void llget_file_position() 
{
  LLStream *stream; 
  stream = (LLStream *) llnextcheckedarg(llstream_t);
  if(stream->type != LLFILE) llerror(LLNOT_A_FILE);
  lllastarg(); 
  llpusharg(lllong_to_num(ftell(stream->data.file)));
}
void llset_file_position() 
{
  LLStream *stream; 
  LLNum *offset, *from;
  stream = (LLStream *) llnextcheckedarg(llstream_t);
  offset = (LLNum *) llnextcheckedarg(llnum_t);
  from = (LLNum *) llnextcheckedarg(llnum_t);
  if(stream->type != LLFILE) llerror(LLNOT_A_FILE);
  lllastarg(); 
  fseek(stream->data.file, llnum_to_long(offset), (int)llnum_to_double(from));
  llpusharg(T);
}
void llcurrent_input_port ()
{
  lllastarg(); 
  llpusharg(s_stdin);
}
void llcurrent_output_port()
{
  lllastarg(); 
  llpusharg(s_stdout);
}

void llprint()
{
  LLString *string; 
  int n;
  char *c;
  while(llmoreargs()) {
    if(llobj_tag(llpeekarg()) == llstring_t) {
      string = (LLString *)llnextarg();
      c = string->text; 
      n = string->length;
      while(n--) {
	llwrite_stream(s_stdout, *c++);
      }
    } else 
      llprint_obj(llnextarg(), s_stdout);
    
  }
  llpusharg(T);
}

/* Initialize the stream type descriptor */
void llinit_stream ()
{
  llstream_t = lladd_obj_td (sizeof(LLStream), "Stream",
			     free_stream, 0, 0, 0);
  s_stdin = llmake_stream_to_file(stdin);
  s_stdout = llmake_stream_to_file(stdout);
  s_stderr = llmake_stream_to_file(stderr);
  set_env(tlge, llcstring_to_sym("stdin"), (LLObj *)s_stdin);
  set_env(tlge, llcstring_to_sym("stdout"), (LLObj *)s_stdout);
  set_env(tlge, llcstring_to_sym("stderr"), (LLObj *)s_stderr);
  llregister_cfunc(llopen_input_file, "open-input-file");
  llregister_cfunc(llopen_output_file, "open-output-file");
  llregister_cfunc(llopen_append_file, "open-append-file");
  llregister_cfunc(llopen_update_file, "open-update-file");
  llregister_cfunc(llclose_port, "close-port");
  llregister_cfunc(llget_file_position, "get-file-position");
  llregister_cfunc(llset_file_position, "set-file-position");
  llregister_cfunc(llcurrent_input_port, "current-input-port");
  llregister_cfunc(llcurrent_output_port, "current-output-port");
  llregister_cfunc(llread_chars, "read-string");
  llregister_cfunc(llprint, "print");
}

