#include <iostream>
#include <string>
#include "SDL.h"
#include "SDL_thread.h"
#include "rt.h"
#include "ConfigFile.h"

#define RT_THREADDONE SDL_USEREVENT

int SCREEN_WIDTH, SCREEN_HEIGHT;
bool Interactive;
int Threads;

bool Quit = false;	//used to signal the renderer threads to quit

using std::cout;
using std::string;
using std::endl;

Intersection ISection(Ray & r)
{
	double Dist = 1000000;
	double tmp;
	Obj3D * o = NULL;
	Intersection is;
	
	for(int i=0;i<NUM_OBJECTS;i++)
	{
		tmp = Objects[i]->ISection(r);
		if(tmp > 0 && tmp < Dist)
		{
			Dist = tmp;
			o = Objects[i];
		}
	}

	if(o != NULL)
	{
		is.Dist = Dist;
		is.Point = r.Start + r.Dir * Dist;
		is.Obj = o;
	}
	else
	{
		is.Dist = -1;
		is.Point.Set(0,0,0);
		is.Obj = NULL;
		
	}
	return is;
}

bool isectcomp(const Intersection & i1, const Intersection & i2)
{
	return (i1.Dist < i2.Dist);
}

Color4 Shadow(const Vector3D<> & Point, Ray ray, double d2l)
{
	Color4 out(1,1,1);
	Color4 c;
	
	Intersection is;
	Vector3D <> Normal;
	double tmp;
	
	std::vector<Intersection> isects;
	isects.reserve(NUM_OBJECTS);
	isects.clear();
	
	for(int i=0;i<NUM_OBJECTS;i++)
	{
		tmp = Objects[i]->ISection(ray);
		if(tmp > 0 && tmp < d2l)
		{
			is.Dist = tmp;
			is.Obj = Objects[i];
			is.Point = ray.Start + ray.Dir*is.Dist;
			is.Color = is.Obj->Surf.ColorAt(is.Point,is.Obj->NormalAt(is.Point));
			if(is.Color.C[3] == 1.0)
				return Color4(0,0,0);
			isects.push_back(is);
		}
	}
	sort(isects.begin(),isects.end(),isectcomp);
	while(!isects.empty())
	{
		is = isects.back();
		isects.pop_back();
		is.Point = ray.Start + ray.Dir*is.Dist;
		out = (out * (1-c.C[3]) + is.Color * is.Color.C[3]) * (1-is.Color.C[3]);
	}
	return out;
/*	do
	{
		isect = ISection(ray);
		d2l2 -= (isect.Dist*isect.Dist) + 0.001; 
		if(isect.Dist == -1 || d2l2 < 0)
			return out;
		Normal = isect.Obj->NormalAt(isect.Point);
		c = isect.Obj->Surf.ColorAt(isect.Point,Normal);
		out = 
		if(out.C[3] == 0.00)
			return out;
	}while(1);*/
}

Color4 GetColor(Ray & r,int bounce = 0)
{
	Color4 c;
	Color4 light = Ambient;
	Color4 color;
	Color4 tmp;

	double li, d2l, d2i, ca, spec;
	
	Ray ToLight, Out;
	
	Intersection i = ISection(r);
	
	if(i.Obj != NULL)
	{
		Vector3D<> ISect = r.Start + r.Dir * i.Dist;
 		Vector3D<> Normal = i.Obj->NormalAt(ISect);
			
		ISect += Normal * .0002;		//Kludge to make sure the intersection point is on right side of surface

		for(int l=0;l<NUM_LIGHTS;l++)
		{
			ToLight.Start = ISect;
   			Lights[l]->VectorFrom(ISect,ToLight.Dir,d2l);

			tmp = Shadow(ISect,ToLight,d2l) * Lights[l]->IntensityAt(ISect);

//			if(tmp.C[3] > 0)
//			{
				li = ToLight.Dir.dot(Normal);
				light += tmp * li;

				if(i.Obj->Surf.Specular && !bounce)		//don't calc specular for reflection bounces
				{
					Vector3D<> Reflect = (Normal*(2*li) - ToLight.Dir).Norm();
					ca = Reflect.dot(-r.Dir);
					if(ca < 0) ca = 0;
					spec = pow(ca,i.Obj->Surf.Gloss) * i.Obj->Surf.Specular;
					//spec = pow((ToLight.Dir-r.Dir).Norm().dot(Normal),i.Obj->Surf.Gloss);

					c += tmp * spec;
				}
//			}
		}

		color = i.Obj->Surf.ColorAt(ISect,Normal);
		c += color * light * i.Obj->Surf.Diffuse;

		if(bounce < MAX_BOUNCES)
		{
			if(i.Obj->Surf.Reflect > 0.0)
			{
				Out.Start = ISect;
				Out.Dir = (r.Dir - Normal*(2*r.Dir.dot(Normal))).Norm();
				tmp = GetColor(Out,bounce+1);
				
				c += tmp * i.Obj->Surf.Reflect;
			}
			
			if(color.C[3] < 1.0)
			{
				double a = color.C[3];
				c *= a;
	
				Out.Start = ISect-Normal*.0005;
				Out.Dir = r.Dir;
				tmp = GetColor(Out,bounce+1);
	
				c += tmp*(1.0-a);
			}
		}
		
		if(c.C[0] > 1) c.C[0] = 1;
		if(c.C[1] > 1) c.C[1] = 1;
		if(c.C[2] > 1) c.C[2] = 1;
	}
	
	return c;
}

struct RTJob
{
	SDL_Surface * Surf;
	unsigned short FirstLine, LastLine;
};

int Raytrace(void * J)
{
	RTJob * job = (RTJob*)J;

	SDL_Surface * surf = job->Surf;
	
	Uint32 * dest = (Uint32*)surf->pixels + job->FirstLine*(surf->pitch/4);
	Color4 col;
	Ray r;
	int x,y;
	double sw = AA*SCREEN_WIDTH, sh = AA*SCREEN_HEIGHT;
	double sx = 8./sw, sy = 8./sh;
	double scale = 255./(AA*AA);
	int i,j;
	for(y=job->FirstLine*AA;y<job->LastLine*AA;y+=AA)
	{
		for(x=0;x<sw;x+=AA)
		{
			col.C[0] = col.C[1] = col.C[2] = 0;
			for(i=0;i<AA;i++)
			{
				for(j=0;j<AA;j++)
				{
					r.Start.Set(double((x+i)-sw/2) * sx,
								double(sh/2-(y+j)) * sy,0);
					r.Dir = (r.Start - Vector3D<>(0,0,10));
					r.Dir.Normalize();
					col += GetColor(r);
				}
			}
	
			*dest = SDL_MapRGB(surf->format,(Uint8)(col.C[0]*scale),
											(Uint8)(col.C[1]*scale),
											(Uint8)(col.C[2]*scale));
			dest++;
		}
		//Checking this after every line, rather then every pixel, will
		//make it slower to quit, but should provide a render speed-up compared to
		//a per-pixel check (I have no idea if the difference would actually
		//be significant, though)
		if(Interactive)
			SDL_Flip(surf);
		if(Quit)
			return 0;
		dest += (surf->pitch/4) - SCREEN_WIDTH;
	}
	SDL_Flip(surf);
	if(!Quit)		//if we finished, and weren't forced to quit...
	{
		SDL_Event ThreadDone;
		ThreadDone.type = RT_THREADDONE;
		SDL_PushEvent(&ThreadDone); //push out an event saying we're done
 	}
	return 0;
}

int main(int argc, char ** argv)
{
	RTJob * R;
	SDL_Thread ** T;
	SDL_Event e;
	int i;
	long t;
	int ThreadsDone = 0;
	
	char * CfgFN;
	if(argc > 1)
		CfgFN = argv[1];
	else
		CfgFN = "Tracer.cfg";
		
	ConfigFile Cfg(CfgFN);
		
	try
	{
		Assert(SDL_Init(SDL_INIT_VIDEO) >= 0,(std::string)"Unable to init SDL:"+SDL_GetError());
		atexit(SDL_Quit);
		
  		SCREEN_WIDTH = (int)Cfg.GetDouble("Tracer.Screen.Width",640);
		SCREEN_HEIGHT = (int)Cfg.GetDouble("Tracer.Screen.Height",480);
		cout << "(" << SCREEN_WIDTH << "," << SCREEN_HEIGHT << ")" << endl;
		MAX_BOUNCES = (int)Cfg.GetDouble("Tracer.Bounces",3);
		Interactive = Cfg.GetString("Tracer.Interactive") == "on";
		AA = (short)Cfg.GetDouble("Tracer.AA",1);

		SDL_Surface * screen;
		screen = SDL_SetVideoMode(SCREEN_WIDTH,SCREEN_HEIGHT,32,SDL_DOUBLEBUF);
		Assert(screen,std::string("Unable to open video: ") + SDL_GetError());

		t = SDL_GetTicks();
		std::string scene = Cfg.GetString("Tracer.Scene","Scene.xml");
		SDL_WM_SetCaption(("Tracing: <" + scene + ">").c_str(),NULL);
		LoadScene(scene.c_str());
		cout << "Scene Load took " << ((float)SDL_GetTicks()-t)/1000 << " seconds." << endl;
		t = SDL_GetTicks();

		Threads = (int)Cfg.GetDouble("Tracer.Threads",1);

		R = new RTJob[Threads];
		T = new SDL_Thread*[Threads];
		
		if(SDL_MUSTLOCK(screen))
			SDL_LockSurface(screen);

		for(i=0;i<Threads;i++)
		{
			R[i].Surf = screen;
			R[i].FirstLine = int(i*(double(SCREEN_HEIGHT)/Threads));
			R[i].LastLine = int((i+1)*(double(SCREEN_HEIGHT)/Threads));
			T[i] = SDL_CreateThread(Raytrace,(void*)(R+i));
			DEBUG("Created rendering thread.");
		}

		while(!Quit && Interactive)
		{
			if(SDL_PollEvent(&e))
			{
				if(e.type == SDL_QUIT)
					Quit = true;
				else if(e.type == SDL_KEYDOWN)
				{
					if(e.key.keysym.sym == SDLK_ESCAPE)	//quit on escape
						Quit = true;
					else if(e.key.keysym.sym == 'q')			//q sets us to quit on finish
						Interactive = false;
					else if(e.key.keysym.sym == 's')
						SDL_SaveBMP(screen,"Raytrace.bmp");
				}
				else if(e.type == RT_THREADDONE)
				{
					if(++ThreadsDone == Threads)		//act only if all threads are done
					{
						double time = ((float)SDL_GetTicks()-t)/1000;
						cout << "Raytrace took " << time << " seconds." << endl;
					 	char title[256];
					 	sprintf(title,"%s : %.2f seconds",scene.c_str(),time);
					 	SDL_WM_SetCaption(title,title);
					 	
					 	SDL_SaveBMP(screen,"Raytrace.bmp");
					}
				}
			}
		}
		if(SDL_MUSTLOCK(screen))
			SDL_UnlockSurface(screen);
	}
	catch(const Exception & e)
	{
		cout << "Exception: " << e.Message.c_str() << endl;
		exit(-1);
	}
	catch(...)
	{
		cout << "Unknown exceptional condition encountered." << endl;
		exit(-2);
	}

	for(i=0;i<Threads;i++)
		SDL_WaitThread(T[i],NULL);
	delete[] T;
	delete[] R;

	FreeScene();

	return 0;
}
