%{
#include "S.h"
#include "y.tab.h"
#include "options.h"

typedef struct S_token {
	int ttype;
	char *value;
	struct S_token *next;
} S_token;

typedef struct s_source {
	struct s_vector *ptree;
	FILE *f;
	int lineno;  /* lex stuff */
	char *sptr;  /* lex stuff */
	char *sbuf;  /* lex stuff */
	struct s_source *next;
	int ltoken; /* last token */
	int line_pos;
	int last_got;
} s_source;

#define NULL_TOKEN	(S_token *)NULL
#define NULL_SOURCE	(s_source *)NULL
#define MAXSTRING	1024
#define SMALL_STRING	16
#define LINE_ESTIMATE	64
#define MAXPAR		25
#define END_STAT	0

#undef input
#define input()		(Is_newline == EOF ? 0 : yytchar)
#define Is_newline	(Next_char == '\n' ? (yylineno++, yytchar) : yytchar)
#define Next_char	(yytchar = yysptr > yysbuf ? U(*--yysptr) : Do_getc)
#define Do_getc		(Do_prompt, last_got = getc(yyin))
#define Do_prompt	(Should_prompt ? (Prompt(Which_prompt), 0) : 0)
#define Should_prompt	(do_prompt && line_pos == 0 && last_got == '\n')
#define Which_prompt	(contn ? cont_prompt : S_prompt)

static void Prompt(), yyflush(), pushpar(), push_token();
static void make_comment(), makename(), makestring();
static int end_token(), poppar(), digit(), make_number();

static s_source *cur_source;
static char *last_name, *error_msg = NULL_STRING;
static char *s_lex_; /* C pointer to value of token */
static int last_got = '\n', doing_parse = TRUE, do_balance = TRUE, read_list;
static int line_pos = 0; /* # tokens read since last new line (for prompting) */
static int pstack[MAXPAR], parlev = 0, ltoken = END_STAT, contn = 0;
static int continue_pos, do_prompt, comment_ok = FALSE;
static S_token *token_list, *last_token;
static vector *S_comment;

int cur_interact = 1, doing_list = 0;
int S_lex_debug = FALSE; /* print tokens: delete? see set_data() & S_lex() */
extern long zstakz_[];
extern char zcstkz_[];
extern YYSTYPE yylval;

static void 
Prompt(text)
char *text;
{
	fflush(stdout); fflush(stderr); fputs(text, stderr); fflush(stderr);
}

int 
S_lex()
{
	static int kept_type;
	static char *kept_value;
	int ttype;

	if(ltoken == CONTINUE) { /* kept a token from last time */
		ltoken = ttype = kept_type; s_lex_ = kept_value;
		goto post_process;
	}
	if(doing_list && read_list>=doing_list) return(END_STAT);
	switch(ltoken) {
	case QUESTION:
	case END_OF_FILE:
		comment_ok = FALSE; ltoken=END_STAT; line_pos = 0;
		if(S_lex_debug){
			if(doing_list)fprintf(stderr,"Item %ld, ",read_list+1);
			fprintf(stderr,"token  type: END_STAT, value: %s\n",do_unlex(s_lex_));
		}
		return(END_STAT);
	/* tokens that can be followed by a comment */
	case RPAR:
		comment_ok = !continue_pos; /* after if(), for(), function() */
		break;
	case NULL:
	case SEMI:
	case COMMA:
	case FUN_DEF:
	case LBRACE:
		comment_ok = TRUE;
		break;
	default: comment_ok = FALSE; /* save comments for later */
	}
	do {
		if(S_comment && comment_ok) ttype = COMMENT;
		else {
			ttype = yylex();
			push_token(ttype);
		}
	} while(ttype == CONTINUE || (ttype == COMMENT && !comment_ok));
	continue_pos = 0; /* cancel forced continue: see action for \n */
	if(ttype==NULL) ttype = ltoken; /* NULL == END_STAT (not a great choice) so
					for now, ltoken used to distinguish */
post_process: /* a token (yylex() or kept) */
	/*
	 * line_pos is set to zero by the various tokens that can
	 * result from new lines; in this case leave it alone, else
	 * increment it.  (line_pos used by Prompt and pop_par)
	 * This switch also pops parens, to check for balance
	 */
	switch(ttype) {
	case LIST_SEP:
		if(doing_list && ltoken != LIST_SEP)
			{read_list++; ltoken =LIST_SEP;}
	case SEMI: 
	case END_STAT:
	case COMMENT:
		if(line_pos)line_pos++; break;
	case RPAR:
	case RBRACK:
	case DOUBLE_RBRACK:
	case RBRACE:
		ttype = poppar(ttype);
	default: line_pos++;
	}
	switch (ttype) { /* generate the token value */
	case LITERAL:
		break; /* value handled by action */
	case COMMENT:
		yylval.vector_val = alc1(COMMENT,S_comment);
		S_comment = NULL;
		break;
	case RPAR:
	case END_STAT:
	/* wrapup any comments preceding these tokens */
		if(S_comment) { /* save this token, return the comment */
			kept_value = s_lex_;
			kept_type = ttype;
			if(ttype==RPAR)pushpar(ttype);
			yylval.vector_val = alc1(COMMENT,S_comment);
			S_comment = NULL;
			ttype = LEFT_COMMENT;
			ltoken = CONTINUE;
		}
		else yylval.char_val = s_lex_;
		break;
	default:
		yylval.char_val  =  s_lex_ ;
	}
	if(S_lex_debug)
		fprintf(stderr,"token type \"%s\", value: %s\n",
	  	token_name(ttype), do_unlex(s_lex_));
	return(ttype);
}

static void 
yyflush()
{
	int c;
	char *buf = S_alloc(80L,sizeof(char)), *p = buf, *overflow = buf+79;

	if(cur_interact && ltoken != END_STAT && ltoken != CONTINUE)
		do {
			c=input(); if(p<overflow)*p++=c;
		} while(c != EOF && c != '\n');
	*p = '\0'; s_lex_ = buf;
	push_token(UNKNOWN);
	contn = FALSE;
}

/* was the previous token ok to end a statement? */
static int 
end_token()
{
	switch(ltoken) {
	case NAME:
	case QUESTION:
	case STRING:
	case LITERAL:
	case RPAR:
	case RBRACK:
	case DOUBLE_RBRACK:
	case RBRACE:
	case BREAK:
	case NEXT:
	case SYSTEM:
	case QUIT:
		return(1);
	default: return(0);
	}
}

%}
alpha [a-zA-Z.]
alphanum [a-zA-Z0-9.?]
digit [0-9]
integer {digit}+
pm [+\-]
exp ([dDeE]{pm}?{integer})
float (({integer}{exp})|({integer}\.{digit}*{exp}?)|(\.{integer}{exp}?))
numeric ({integer}|{float})
complex ({numeric}i)
%%
"for"	{  
	makename(yytext,yyleng+1);
	pushpar(CONTINUE);
	ltoken = FOR; return(FOR);}
"in"	{  
	makename(yytext,yyleng+1);
	ltoken = IN; return(IN);}
"if"	{  
	makename(yytext,yyleng+1);
	pushpar(CONTINUE);
	ltoken = IF; return(IF);}
"else"	{  
	makename(yytext,yyleng+1);
	ltoken = ELSE; return(ELSE);}
"repeat"	{  
	makename(yytext,yyleng+1);
	ltoken = REPEAT; return(REPEAT);}
"break"	{  
	makename(yytext,yyleng+1);
	ltoken = BREAK; return(BREAK);}
"next"	{  
	makename(yytext,yyleng+1);
	ltoken = NEXT; return(NEXT);}
"while"	{  
	makename(yytext,yyleng+1);
	pushpar(CONTINUE);
	ltoken = WHILE; return(WHILE);}
"function" {
	makename(yytext,yyleng+1);
	pushpar(CONTINUE);
	ltoken = FUN_DEF; return(FUN_DEF);}
"return" {
	makename(yytext,yyleng+1);
	ltoken = RETURN; return(RETURN);}
{numeric}	{
	ltoken = make_number(DOUBLE); return(ltoken);}
{complex}	{
	ltoken = make_number(COMPLEX); return(ltoken);}
{alpha}{alphanum}*	{ 
	makename(yytext,yyleng+1); last_name = s_lex_;
	ltoken = NAME; return(NAME);}

#.*$	{  
	makename(yytext,yyleng+1);
	make_comment();
	/*leave ltoken alone */ return(COMMENT);
	}

\"	|
"'"	{ 
	makestring();
	ltoken = STRING; return(STRING); }


"+"	|
"-"	{ 
	makename(yytext,yyleng+1); last_name = s_lex_;
	ltoken = ADDOP; return(ADDOP); }
&	|
"|"	{
	char c = input();
	if(c==yytext[0]){yytext[1]=c;yytext[2]='\0';yyleng=2;}
	else unput(c);
	makename(yytext,yyleng+1); last_name = s_lex_;
	ltoken = ANDOR; return(ANDOR); }
":"	{ 
	makename(yytext,yyleng+1); last_name = s_lex_;
	ltoken = COLON; return(COLON); }
","	{ 
	makename(yytext,yyleng+1);
	ltoken = COMMA; return(COMMA); }
"$"	{ 
	makename(yytext,yyleng+1); last_name = s_lex_;
	ltoken = DOLLAR; return(DOLLAR); }
"="	{ 
	makename(yytext,yyleng+1);
	ltoken = EQUAL; return(EQUAL); }
"<-"	|
"_"	{ 
	makename(yytext,yyleng+1); last_name = s_lex_;
	ltoken = LARROW; return(LARROW); }
"<<-"	{
	makename(yytext,yyleng+1); last_name = s_lex_;
	ltoken = DBLEARROW; return(DBLEARROW); }
"{"	{ pushpar(RBRACE); 
	makename(yytext,yyleng+1);
	ltoken = LBRACE; return(LBRACE); }
"["	{
	char c = input(); int partoken;
	ltoken = LBRACK; partoken=RBRACK;
	if(c==yytext[0]){
		yytext[1]=c;yytext[2]='\0';yyleng=2;
		ltoken = DOUBLE_LBRACK; partoken=DOUBLE_RBRACK;
	}
	else unput(c);
	pushpar(partoken); 
	makename(yytext,yyleng+1); last_name = s_lex_;
	return(ltoken); }
"~"	{
	makename(yytext,yyleng+1); last_name = s_lex_;
	ltoken = SIMILAR; return(SIMILAR); }
"!="	|
"<"	|
"<="	|
"=="	|
">"	|
">="	{ 
	makename(yytext,yyleng+1); last_name = s_lex_;
	ltoken = LOGOP; return(LOGOP); }
"("	{ pushpar(RPAR); 
	makename(yytext,yyleng+1);
	ltoken = LPAR; return(LPAR); }
"*"	|
"/"	{ 
	makename(yytext,yyleng+1); last_name = s_lex_;
	ltoken = MULOP; return(MULOP); }
"!"	{
	if(ltoken) {
		makename(yytext,yyleng+1); last_name = s_lex_;
		ltoken = NOT; return(NOT);
	  }
	else {
		makestring();
		ltoken = SYSTEM; return(SYSTEM);
	  }
	}
"%"[^%\n]*%	{ 
	makename(yytext,yyleng+1); last_name = s_lex_;
	if(yytext[yyleng-1]=='\n') {
		fprintf(stderr,"Invalid operator: \"%s\"\n",yytext);
		return(UNKNOWN);
	}
	else {ltoken = SPOP; return(SPOP); } }
"?"{alphanum}*	{ 
	makename(yytext,yyleng+1); last_name = s_lex_;
	ltoken = QUESTION; return(QUESTION);
	}
"->"	{ 
	makename(yytext,yyleng+1); last_name = s_lex_;
	ltoken = RARROW; return(RARROW); }
"}"	{
	makename(yytext,yyleng+1);
	ltoken = (RBRACE) ; return(RBRACE); }
"]"	{
	char c = input();
	ltoken = RBRACK;
	if(c==yytext[0] && pstack[parlev-1] == DOUBLE_RBRACK){
		yytext[1]=c;yytext[2]='\0';yyleng=2;
		ltoken = DOUBLE_RBRACK;
	}
	else unput(c);
	makename(yytext,yyleng+1); last_name = s_lex_;
	return(ltoken); }
")"	{ 
	makename(yytext,yyleng+1);
	ltoken = (RPAR) ; return(RPAR); }
";"	{ 
	makename(yytext,yyleng+1);
	while(parlev>0 && pstack[ parlev-1]==CONTINUE)parlev--; /* see note at \n */
	ltoken = parlev==0 ? END_STAT : SEMI;
	return(ltoken);
	}
"**"	|
"^"	{ 
	makename("^",2); last_name = s_lex_;
	ltoken = UARROW; return(UARROW); }
[ \t]*	{ makename(yytext,yyleng+1);
	if(doing_list && parlev==0){return(LIST_SEP);}
 	return(CONTINUE); } /* ignore white space */
\n	{
	int l = line_pos;
	makename(yytext,yyleng+1);
	line_pos = 0; contn = ltoken!=NULL;
	if(doing_list && parlev==0){return(LIST_SEP);}
 	if(continue_pos && ltoken == RPAR && continue_pos  == l){
		ltoken=CONTINUE;
		return(CONTINUE);
	}
		/* above continues after certain tokens (for, if, while, function)
		 * in the context:  <token>( stuff )\n other stuff
		 * when the other stuff starts, S_lex will cancel continue_pos
		 */
	else if( parlev == 0 && end_token()) { /* end of statement */
		contn = FALSE;
		ltoken = END_STAT; return(END_STAT);
		}
	else if( parlev > 0 && pstack[ parlev-1 ] == RBRACE && end_token()) { /* compound */
		makename(";",2);
		ltoken = SEMI; return(SEMI);
		}
	else return(CONTINUE);
	}
.	{
	makename(yytext,yyleng+1);
	ltoken = UNKNOWN; return(UNKNOWN); }
%%

static void 
pushpar(partype)
int partype;
{
	if(do_balance && parlev < MAXPAR)
		pstack[parlev] = partype;
	parlev++;
}

/* poppar -- check for matching correct brace, bracket or paren  */
static int 
poppar(partype)
{
	if(!do_balance)return(partype);
	if(--parlev < MAXPAR) if(parlev < 0 || pstack[parlev] != partype) {
		int what = pstack[parlev];
		error_msg = parlev<0 ? "No opening parenthesis"
		  : (what!=CONTINUE ? encs1("Unbalanced parentheses, expected \"%s\"",
			token_name(pstack[parlev]))
		  : "Error in loop or condition");
		parlev = 0; return(UNBALANCED);
		}
	if(pstack[parlev-1] == CONTINUE) {
		continue_pos = line_pos+1;
		while( parlev>0 && pstack[parlev-1]==CONTINUE) parlev--;
		}
	return(partype);
}

/*makestring -- read a string token */
static void 
makestring()
{
	int c,quote = yytext[0], prompt; long limit,current;

	limit=SMALL_STRING; current = 0; contn = 1;
	if(quote=='!')quote='\n'; /* ! escape for SYSTEM token */
	s_lex_ = S_alloc((long)SMALL_STRING, sizeof(char) );
	while((c=input()) && c != quote) {
		prompt = do_prompt && c =='\n';
		if(c == '\\')  { /* escaped characters */
			if((c=input()) == 0) break;
			if(digit(c)) {
				int i, cc;
				unput(c);
				MEANINGFUL(cc);
				for(i = c = 0; i < 3 && digit(cc=input()); i++)
					c = 8*c + (cc - '0');
				if(cc == 0) break;
				if(i < 3) unput(cc);
			}
			else switch(c) {
			case 'n': c='\n'; break;
			case 't': c='\t'; break;
			case 'b': c='\b'; break;
			case 'r': c='\r'; break;
			}
		}
		*(s_lex_ + current++) = c;
		if(current >= limit) {
			s_lex_ = S_realloc(s_lex_, limit * 2,limit ,sizeof(char));
			limit *= 2;
		}
		if(prompt)Prompt("Continue string: ");
	}
	if(quote == '\n')unput('\n');
	*(s_lex_ + current) = '\0';
	yytext[0]='\0';
}

static int 
digit(c)
int c;
{ return( c>= '0' && c<= '9'); }

static char *cur_s_name;
static FILE *fprev = stdin;
static int is_text;

/* the source pushing code is needlessly general; only the parsing from stdin
/* will resume after allocation; otherwise, the parsing only comes from the
/* parse()  function, which always  eats either  text or a whole file */
int 
push_source(file,text)
char *file; int text;
{
	extern int yylineno; extern char *yysptr, yysbuf[];
	FILE *f;
	s_source *s; int n;
	if(file==NULL)f=stdin;
	else if((f=fopen(file,"r"))==NULL)
		Recover(encs1("can't open source file \"%s\"",file),NULL_ENTRY);
	is_text = text;
	if(f!=yyin) { 
		s = PERMALLOC(1, s_source);
		/* this structure lasts possibly over expressions, so it is
		/* explicitly freed by pop_source */
		s->next = cur_source;
		s->ptree = S_ptree;
		s->f = yyin;
		s->lineno = yylineno;
		s->sptr = yysptr;
		s->ltoken = ltoken;
		s->line_pos = line_pos;
		s->last_got = last_got;
		if(yysptr>yysbuf){ /* copy the save buffer */
			s->sbuf = PERMALLOC(n=yysptr-yysbuf+1, char);
			strncpy(s->sbuf,yysbuf,n);
		}
		cur_source = s; cur_s_name = file;
		fprev = yyin; yyin = f; yylineno = 0;
	}
	Slexinit();
	return(TRUE);
}

int 
pop_source()
{
	extern int yylineno; int i; extern char *yysptr, yysbuf[];
	s_source *s;
	if(yyin != fprev && cur_source) {
		s = cur_source;
		i = (s->next!=NULL);
		fclose(yyin);
		cur_source = s->next;
		S_ptree = s->ptree;
		yyin = s->f;
		yylineno = s->lineno;
		yysptr = s->sptr;
		ltoken = s->ltoken;
		line_pos = s->line_pos;
		last_got = s->last_got;
		if(yysptr>yysbuf)strcpy(yysbuf,s->sbuf);
		if(s->sbuf) free(s->sbuf);
		free((char *)s);
	}
	else { /* basic values */
		i = FALSE;
		cur_source = NULL_SOURCE;
		S_ptree = NULL_ENTRY;
		yyin = stdin;
		yylineno = 0;
		yysptr = yysbuf;
		ltoken = 0;
		line_pos = 0;
		last_got = '\n';
	}
	Slexinit();
	return(i);
}

/* initialize at beginning of statement, including after error */
void 
Slexinit()
{
	cur_interact = yyin==stdin && isatty(2); /* read stdin, errors to tty */
	do_prompt = (isatty(fileno(yyin)) || (S_echo_on && yyin==stdin)) ||
		getenv("ALWAYS_PROMPT");
	last_token = token_list = NULL_TOKEN;
	error_msg = NULL;
	ltoken = NULL;
	contn = FALSE;  do_balance = TRUE;
	doing_list = read_list = parlev = continue_pos = 0;
}


int 
yywrap()
{
	ltoken=END_OF_FILE;
	return(1);
}

static void 
push_token(ttype)
int ttype;
{
	S_token *new;
	char *s;
	/* initial white space is ignored */
	if(ttype==CONTINUE && last_token==NULL_TOKEN)return;
	new  = (S_token *)S_alloc(1L,sizeof(S_token));
	if(last_token){
		last_token->next = new;
		last_token = new;
	}
	else token_list = last_token = new;
	new->ttype = ttype;
	switch(ttype) {
	case STRING:
		s = do_unlex(s_lex_);
		break;
	case SYSTEM:
		s = S_alloc(strlen(s_lex_)+2L, sizeof(char));
		*s = '!'; strcpy(s+1, s_lex_);
		break;
	default:
		s = s_lex_;
	}
	new->value = s;
	if(S_echo_on && yyin==stdin){
		fputs(s,stdout);
		if(*s == ';'  && ttype == END_STAT) putc('\n',stdout);
		if(*s == '\n' || ttype == END_STAT) fflush(stdout);
	}
}


#ifdef YYDEBUG
/* yacc calls yyerror with 2 args if YYDEBUG is set */

yyerror(string, arg)
char *string, *arg;
{
	fprintf(stderr, string, arg);
}
#else

#define MAX_COUNT 60


yyerror(string)
char *string;
{
	long count,new_count,total; char *what;
	S_token *p;
	if(string) fputs("Syntax error: ",stderr);
	what = token_name(ltoken);
	if(ltoken != UNKNOWN  && !name_eq(what,s_lex_))
		what = encs2("%s (%s)",what,do_unlex(s_lex_));
	else if(ltoken!=UNKNOWN)what = do_unlex(s_lex_);
	else what = encs1("unknown expression (%s)",do_unlex(s_lex_));
	if(error_msg)
		fprintf(stderr,"%s, before %s ",error_msg,what);
	else if(string) fprintf(stderr,"%s used illegally ",what);
	/* else, called just for the output of the current partial expression*/
	yyflush();
	count = 0;
	if(cur_interact || is_text) {
		fputs("at this point:\n",stderr);
		p = token_list; total = 0;
		while(p) { total += strlen(p->value); p = p->next;}
		p = token_list; total -= MAX_COUNT; /* the tail of the expr */
		while(p){
			if((count<MAX_COUNT||count>total) &&p->next)
				fputs(p->value,stderr);
			new_count = count + strlen(p->value);
			if(count<MAX_COUNT && new_count >=MAX_COUNT
			  && new_count<=total) fputs(" ...\n\t",stderr);
			count = new_count;
			p = p->next;
		}
	}
	else if(yyin!=stdin) fprintf(stderr,"at line %d, file %s",yylineno+1,cur_s_name);
	else fprintf(stderr,"at line %d of standard input",yylineno+1);
	fputs("\n",stderr);
	if(audit_file) {
		p = token_list;
		while(p) { fputs(p->value,audit_file); p = p->next; }
		fputs("#~error: statement with syntax error\n",audit_file);
	}
	token_list = (S_token *)NULL;
	last_signal = SIGSERROR;
	line_pos=0; Slexinit();
}
#endif

static void 
make_comment()
{
	vector *p = S_comment;
	if(p) {
		if(p->length>=p->nalloc) {
			p->value.Char = (char **)S_realloc((char *)p->value.Char,2*p->length,p->length,sizeof(char *));
			p->nalloc = 2*p->length;
		}
	}
	else p = S_comment = alcvec(CHAR, 0L);
	*(p->value.Char+(p->length++)) = s_lex_;
	return;
}

static void makename(p,l) 
char *p; int l;
{
	s_lex_ = S_alloc((long)l,sizeof(char));
	strcpy(s_lex_,p);
}

static int 
make_number(mode)
int mode;
{
	char *ptr; long n; vector *temp;
	makename(yytext,yyleng+1);
	for(ptr=yytext; *ptr != '\0'; ptr++)
		if(*ptr == 'd' || *ptr == 'D')*ptr = 'e';
	temp = yylval.vector_val = alcvec(mode,1L);
	switch(mode) {
	case DOUBLE:
		sscanf(yytext,"%lf",temp->value.Double);
		break;
	case COMPLEX:
		n = sscanf(yytext,"%lf%lf",&(temp->value.Complex->re),
					   &(temp->value.Complex->im));
		if(n == 1) {
			temp->value.Complex->im = temp->value.Complex->re;
			temp->value.Complex->re = 0;
		}
		break;
	}
	temp->x.frame = cons_frame; /*  protection */
	return(LITERAL);
}

vector *
get_token(ent,arglist)
vector *ent, *arglist;
{
	int mode; vector *value;
	UNUSED(ent); UNUSED(arglist);
	do_balance = FALSE; mode = S_lex(); do_balance = TRUE;
	switch(mode) {
	case LITERAL:
	case COMMENT:
		value = copy_data(yylval.vector_val,NULL_ENTRY);
		break;
	case NAME:
		value = alc_name(c_s_cpy(s_lex_));
		break;
	case STRING:
		value = alcvec(CHAR,1L);
		value->value.Char[0] = c_s_cpy(s_lex_);
		break;
	case SPOP: case ADDOP: case MULOP: case ANDOR: case COLON:
	case DOLLAR: case LOGOP: case NOT: case LBRACK: case DOUBLE_LBRACK:
		value = alcvec(FUN_CALL,1L);
		value->value.tree[0] = alc_name(c_s_cpy(s_lex_));
		break;
	default:
		value = alcvec(mode,min_length(mode));
	}
	return(value);
}

vector *
get_record(ent,arglist)
vector *ent, *arglist;
{
	vector *value, *rec_sep, *item, *current,**p;
	int named, pattern, done; long n, mode, cur_mode, new_mode, label_field = 0L;
	char *name_string, *rec_sep_string = "", *record_name = "", *label = NULL_STRING;
	rec_sep = arglist->value.tree[0];
	if(data_mode(rec_sep) != NULL) rec_sep_string = string_value(rec_sep);
	value = arglist->value.tree[1];
	if(!(pattern = value->length>0))value = alcvec(DOUBLE,0L);
	else if(PRECIOUS(value))value = copy_data(value,NULL_ENTRY);
	current = arglist->value.tree[2];
	if(data_mode(current)==NULL) {} /* no row labels */
	if(data_mode(current)==CHAR)label = string_value(current);
	else label_field = long_value(current,ent);
	if(NOT_RECURSIVE(value->mode)){
		current = alcvec(LIST,1L); current->value.tree[0] = value;
		value->length = 0; value = current;
	}
	else for(n=0, p = value->value.tree; n<value->length; n++,p++) (*p)->length = 0;
	for(n=0, name_string = "", item = current = NULL_ENTRY,
		done = named = FALSE;!done;) {
		do_balance = FALSE; mode = S_lex(); do_balance = TRUE;
		switch((int)mode) {
		case EQUAL:
			name_string = string_value(item); item = current = NULL_ENTRY;
			break;
		case END_OF_FILE:
			if(!n && !item) value = alcvec(END_OF_FILE,0L);
			done = TRUE;
			if(item)goto do_item;
			else break;
		case NULL:
			if(rec_sep->mode == NULL) done = TRUE;
			else continue;
		case COMMA:
		do_item:
			if(!item)item = New_vector();
			if(*name_string) {
				current = xact_comp(value,name_string);
				named = TRUE;
			}
			else if(!current && !named && pattern)
				current = value->value.tree[n % value->length];
			if(current) {
				cur_mode = data_mode(current);
				new_mode = coerce_to((int)cur_mode, data_mode(item));
				if(new_mode != cur_mode)
					coevec(current, (int)new_mode, FALSE, FALSE);
				append_el(current,(long)NOARG,item);
				if(!named)current = NULL_ENTRY;
			}
			else { /* a new element */
				item->name = name_string;
				if(n<value->length)value->value.tree[n] = item;
				else append_el(value,(long)NOARG,item);
			}
			item = NULL_ENTRY; n++;
			break;
		case LITERAL:
			if(item) {
				item = coevec(item,CHAR,TRUE,CHECK_IT);
				item->value.Char[0] = encs2("%s %s",item->value.Char[0],
					string_value(yylval.vector_val));
			}
			else item = copy_data(yylval.vector_val,NULL_ENTRY);
			break;
		default:
			if(rec_sep->mode != NULL && name_eq(s_lex_,rec_sep_string)) {
				done = TRUE; goto do_item;
			}
			if(item) {
				item = coevec(item,CHAR,TRUE,CHECK_IT);
				item->value.Char[0] = encs2("%s %s",item->value.Char[0],
					s_lex_);
			}
			else {
				item = alcvec(CHAR,1L);
				item->value.Char[0] = c_s_cpy(s_lex_);
			}
		}
	}
	current = coevec(value,ANY,FALSE,FALSE);
	if(!NOT_RECURSIVE(current->mode)) {
		if(label && *label)
			label_field = x_which_comp(label,current);
		if(label_field >0 && label_field <= current->length) {
			item = current->value.tree[label_field-1];
			record_name = string_value(item);
			item->length = 0; /* will delete */
		}
	}
	for(n=current->length, p = current->value.tree+n-1;n>0; n--, p--){
		item = *p;
		if(item->length == 0) del_comp(current,n);
	}
	value ->name = record_name;
	current = alcvec(LIST,1L); current->value.tree[0] = value;
	return(current);
}
void 
unset_source()
{
	if(yyin!=stdin) {
		flush_input(yyin);
		pop_source();
	}
}

void
set_source(fptr)
char  **fptr;
{
	if(!push_source(*fptr,0))
			Recover("Can't open new source",NULL_ENTRY);
}
