/* read an audit file */
#include <stdio.h>
#include "S.h"
#include "audit.h"
#ifdef Research
#include <sys/ttyio.h>
#include <sys/filio.h>
#else
#include <sgtty.h>
#endif
#ifdef ATT_UNIX
#include <fcntl.h>
#endif
#define MAXBUF	10000
#ifdef DEBUG
#define DEB(x) x
#else
#define DEB(x)
#endif
#define MARKED 1
#define SCANNED 2

#define NPRIME 997	/* hash table size, must be prime */
#define MINFREE 10	/* min free slots in hash table */
char * hash_table[NPRIME];
NAME_ENTRY * hash_info[NPRIME];

void exit();
char buf[MAXBUF];
char *bufp;
int nstmt = 0, interact;
int found_end = 0;

FILE *audit_in;

STMT *current_stmt = NULL;

main(argc,argv)
int argc; char *argv[];
{
	char *getenv();
#ifdef ATT_UNIX
	int flags,c;
#else
	int n;
#endif
	if(argc>2) {
		fprintf(stderr,"Usage: %s [file]\n",argv[0]);
		exit(1);
		}
	if(argc<2){
		audit_in = fopen(".Data/.Audit","r");
		if(audit_in==NULL){
			fprintf(stderr,"Cannot open .Data/.Audit file in current directory, will look in $HOME\n");
			sprintf(buf,"%s/.Data/.Audit",getenv("HOME"));
 			audit_in = fopen(buf,"r");
			if(audit_in==NULL) {
				fprintf(stderr,"%s: Cannot open %s\n",
					argv[0],buf);
				exit(1);
			}
		}
	}
	else {
		audit_in = fopen(argv[1],"r");
		if(audit_in==NULL) {
			fprintf(stderr,"%s: Cannot open file %s\n",
				argv[0],argv[1]);
			exit(1);
		}
	}
	hash_init();
	interact = isatty(0);
	setbuf(stdin,(char *)NULL);
	prompt();

	bufp = buf;
	if(!interact)	/* read entire file first if non interactive */
		do process_file();
		while(found_end == 0);

#ifdef ATT_UNIX
	flags = fcntl(0,F_GETFL,0);
	fcntl(0,F_SETFL,flags|O_NDELAY);
	for(;;){
		c = getc(stdin);
		if(c!=EOF) {
			ungetc(c,stdin);
			process_command();
		}
		else if(interact) process_file();
		else exit(0);
	}
#else
	for(;;){
		ioctl(0,FIONREAD,&n);
		if(n>0) process_command();
		else if(interact) process_file();
		else exit(0);
	}
#endif
}

process_file()
{
	int c;
	
	c = getc(audit_in);	/* this only reached after newline */
	if(c==EOF) {
		clearerr(audit_in);
		found_end = 1; 
		sleep(2);
		c = getc(audit_in);
		if(c==EOF) {clearerr(audit_in); flush_stmt(); return;}
		} 
	if(c!='#') pickup_stmt(c);
	else {
		c = getc(audit_in);
		if(c=='~') audit_note();
		else {
			ungetc(c,audit_in);
			pickup_stmt('#');
			}
		}
}

/* read a statement into current position in buf until reach newline */
pickup_stmt(c)
int c;
{
	*bufp++ = c;
	while(c!='\n'){
		c = getc(audit_in);
		if(c==EOF) { 
			clearerr(audit_in);
			found_end = 1; 
			sleep(2);
			c = getc(audit_in);
			if(c==EOF) {clearerr(audit_in); flush_stmt(); return;}
			}
		if(bufp>=buf+MAXBUF) error("Statement too big -- exceeds max buffer size");
		*bufp++ = c;
		}
}

error(s)
char *s;
{
	fprintf(stderr,"Error: %s\n",s);
	exit(1);
}

flush_stmt()	/* flush previous statement(s) */
{
	if(bufp<=buf) return;
	while(--bufp>=buf && *bufp=='\n')
		*bufp = '\0'; /* omit trailing newlines */
	DEB(if(bufp>buf) fprintf(stdout,"\t__________\n\t%s\n",buf);)
	if(bufp>buf) new_stmt(buf);
	bufp = buf;
}

audit_note()
{
	char *strdup(),*calloc(), type[20];
	int make_hash_entry();
	int c, i;
	NAME_LIST *r;
	long stamp;
	flush_stmt();
	c = getc(audit_in);
	switch(c){ /* read and identify audit trail statements eat newlines */
	case '\n':	/* statement separator */
		break;		
	case 'g':	/* get dataset or function */
		/* should ignore stuff from system database? */
		fscanf(audit_in,"et \"%s %ld %s\n",buf,&stamp,type);
		if(strcmp(type,"\"function\"")==0) break;
		DEB(fprintf(stdout,"~Get %ld, %s\n",stamp,buf);)
		i = make_hash_entry(buf);
		/* check for matching stamps if we wrote the file */
		r = hash_info[i]->put_ptr;
		if(r && stamp && stamp!=hash_info[i]->current_stamp){
			fprintf(stderr,
"Warning stamp mismatch for object %s in statement %d\n  Previous stamp:%ld, This stamp:%ld\n",
			buf,nstmt,hash_info[i]->current_stamp,stamp);
			hash_info[i]->current_stamp = stamp;
			}
		enter_stmt_on_list(&(hash_info[i]->get_ptr),current_stmt);
		if(current_stmt)
			enter_name_on_stmt(i,&(current_stmt->get_ptr));
		break;

	case 'h': 	/* hash */
			fscanf(audit_in,"ash \"%s\n",buf);
			DEB(fprintf(stdout,"~hash: %s\n",buf);)
		break;

	case 'p':	/* put file */
		fscanf(audit_in,"ut \"%s %ld %s\n",buf,&stamp,type);
		DEB(fprintf(stdout,"~Put: %ld, %s\n",stamp,buf);)
		i = make_hash_entry(buf);
		if(current_stmt)
			enter_name_on_stmt(i,&(current_stmt->put_ptr));
		enter_stmt_on_list(&(hash_info[i]->put_ptr),current_stmt);
		if(stamp) hash_info[i]->current_stamp = stamp;
		break;

	case 'e':	/* error */
		fscanf(audit_in,"rror:%*[^\n]\n",buf);
		/* could omit this stmt from audit file, but since it shouldn't
		affect any data objects, why not leave it in */
		DEB(fprintf(stdout,"~Error:\n%s\n",buf);)
		break;

	case 'N':	/* New source, session */
		i=fscanf(audit_in,"ew source: %s\n",buf);
		if(i>0) {
			DEB(fprintf(stdout,"~New Source: %s\n",buf);)
			}
		else {
			fscanf(audit_in,"ession:%*[^\n]\n");
			DEB(fprintf(stdout,"~New session:\n");)
			}
		break;

	case 'a':	/* access */
		fscanf(audit_in,"ccess \"%s mode: %d\n",buf,&i);
		DEB(fprintf(stdout,"~Access: %s, mode %d\n",buf,i);)
		/* check stamp against current for this name */
		/* what else?? */
		break;

	case 'E':	/* End source */
		i = fscanf(audit_in,"nd source: %s\n",buf);
		if(i>0) {
			DEB(fprintf(stdout,"~End Source: %s\n",buf);)
			}
		else {
			fscanf(audit_in,"ession:%*[^\n]\n");
			DEB(fprintf(stdout,"~End session:\n");)
			}
		break;

	case 's':	/* stamp */
		fscanf(audit_in,"tamp %s\n",buf);
		DEB(fprintf(stdout,"~stamp %s\n",buf);)
		if(strcmp(buf,"NULL")==0) *buf = '\0';
		break;
	case 'i':	/* Interrupt */
		fscanf(audit_in,"nterrupt\n");
		DEB(fprintf(stdout,"~Interrupt\n");)
		break;
	default:
		ungetc(c,audit_in);
		fscanf(audit_in,"%[^\n]\n",buf);
/*		fprintf(stderr,"Unknown audit command at statement %d: %s\n",nstmt,buf); */
	}
}

#include <signal.h>
#include <setjmp.h>
jmp_buf sjbuf;

process_command()
{
	STMT *p,*find_stmt();
	int c,i,mask; extern int nhash,nstmt;
	char *strchr();
	int onintr();
	signal(SIGINT,onintr);
	if(setjmp(sjbuf)) goto exit_commands;
	c = getc(stdin);
	switch(c){
	case 'U':	/* undo statements */
		/* read optional stmt number */
		if(fscanf(stdin, "%d", &i)) p = find_stmt(i);
		else p = current_stmt;
		if(p==NULL) fprintf(stderr,"Statement %d not found\n",i);
		else {
			undo_all(p);
			undo_script(p);
		}
		break;
	case 'E':	/* executable script */
		/* read bunch of numbers, mark stmts */
		while(fscanf(stdin, "%d", &i)){
			p = find_stmt(i);
			if(p==NULL) fprintf(stderr,"Statement %d not found\n",i);
			else p->flags = MARKED;
			}
		backtrack_all();
		backtrack_source();
		break;
	case 'L':
	case 'G':
	case 'P':
	case 'B':
		fscanf(stdin, "%s", buf);
		if(*buf>='0' && *buf<='9'){	/* statement number */
			sscanf(buf,"%d",&i);
			p = find_stmt(i);
			if(p==NULL) fprintf(stderr,"Statement %d not found\n",i);
			else {
				fprintf(stdout,"%s\n~~~~~~~~~\n",
					p->statement);
				if(c=='G') print_names("r ",p->get_ptr);
				if(c=='P') print_names(" w",p->put_ptr);
				if(c=='B') backtrack_stmt(p);
				}
			}
		else {	/* name */
			i = lookup(buf);
			if(i<0) fprintf(stderr,"%s not found\n",buf);
			else {
				fprintf(stdout,"%s\n",hash_table[i]);
				if(c=='G') print_stmts(hash_info[i]->get_ptr);
				if(c=='P') print_stmts(hash_info[i]->put_ptr);
				if(c=='B') backtrack_name(hash_info[i]->put_ptr);
				}
			}
		break;
	case 'N':
		fscanf(stdin, "%s", buf);
		mask = 0;
		if(strchr(buf,'N')) mask = 0;
		if(strchr(buf,'G')) mask |= 1;
		if(strchr(buf,'P')) mask |= 2;
		if(strchr(buf,'A')) mask = 4;
		print_hash(mask);
		break;
	case 'S':
		fscanf(stdin, "%s", buf);
		mask = 0;
		if(strchr(buf,'N')) mask = 0;
		if(strchr(buf,'G')) mask |= 1;
		if(strchr(buf,'P')) mask |= 2;
		if(strchr(buf,'A')) mask = 4;
		print_all_stmts(mask);
		break;
	case 'q':
	case EOF:
		exit(0);
	case ' ':
	case '\n':
		fprintf(stderr,"%d statements, %d identifiers\n",
			nstmt,nhash);
		if(c=='\n') ungetc(c,stdin);
		break;
	default:
		fprintf(stderr,"Commands are:\nLookup x, Get x, Put x, Backtrack x, Names, Statements\n");
		fprintf(stderr,"Undo, Executable script, quit\n");
		}
exit_commands:
	while(getc(stdin)!='\n') ;	/* read thru newline */
	fflush(stdout);
	signal(SIGINT, SIG_DFL);
	prompt();
}

onintr()
{
	extern int backtracking;
	if(backtracking){
		signal(SIGINT, SIG_IGN);
		clear_markers();
		}
	fprintf(stderr,"\n ... Interrupt ...\n");
	longjmp(sjbuf,1);
}

new_stmt(buf)
char *buf;
{
	STMT * p; char *strdup(), *calloc();
	nstmt++;
	if(found_end) {
		fprintf(stdout,"#%d\n%s\n",nstmt,buf);
		fflush(stdout);
	}
	p = (STMT *) calloc(1, sizeof(STMT));
	p->stmt_ptr = current_stmt;
	p->statement = strdup(buf);
	p->get_ptr = NULL;
	p->put_ptr = NULL; 
	p->flags = 0;
	p->stmtno = nstmt;
	current_stmt = p;
}

print_stmts(p)
NAME_LIST *p;
{
	for(; p!=NULL; p = p->nxt_list)
		if(p->stmt_ptr)
			fprintf(stdout,"%d: %s\n",p->stmt_ptr->stmtno,p->stmt_ptr->statement);
}

print_all_stmts(mask)
int mask;
{
	int thismask;
	STMT *p;
	for(p=current_stmt; p; p = p->stmt_ptr){
		thismask = 0;
		if(p->get_ptr)thismask |= 1;
		if(p->put_ptr)thismask |= 2;
		if(mask==4 || mask == thismask){
			fprintf(stdout,"~~~~~~~~~~\n~%d: %s\n",p->stmtno,p->statement);
			if(p->get_ptr)print_names("~r ",p->get_ptr);
			if(p->put_ptr)print_names("~ w",p->put_ptr);
			}
		}
}

backtrack_name(n)
NAME_LIST *n;
{
	STMT *p;
	if(n==NULL) return;
	p = n->stmt_ptr;
	if(!p)error("Statement pointer is null");
	backtrack_stmt(p);
	}

int backtracking;

backtrack_stmt(p)
STMT *p;
{
	backtracking = 1;
	p->flags = MARKED;
	backtrack_all();
	backtrack_print();
}

backtrack_print()
{
	STMT *p;
	for(p=current_stmt; p; p = p->stmt_ptr)
		if(p->flags==SCANNED){
			p->flags = 0;
			fprintf(stdout,"%3d: %s\n",p->stmtno,p->statement);
			}
	backtracking = 0;
}

backtrack_all()
{
	SYM_LIST *q;
	STMT *p;
	int changes;
	backtracking = 1;
	do { /* repeatedly scan statements until no flags are MARKED */
		changes = 0;
		for(p=current_stmt; p; p = p->stmt_ptr)
			if(p->flags==MARKED){
				changes++;
				p->flags=SCANNED;
				for(q = p->get_ptr; q; q = q->nxt_list)
					if(q->stmt_ptr && q->stmt_ptr->flags!=SCANNED)
						q->stmt_ptr->flags = MARKED;
				}
	} while(changes);
}

backtrack_source()
{
	int any;
	STMT *p;
	SYM_LIST *q;
	fprintf(stdout,"# ---------- START OF SOURCE FILE -----------\n");
	any = 0;
	for(p=current_stmt; p; p = p->stmt_ptr)
		if(p->flags==SCANNED)
			for(q=p->get_ptr; q; q = q->nxt_list)
				if(q->stmt_ptr==0){
					++any;
					fprintf(stdout, "# \t%s \n", q->name);
					}
	if(any)
		fprintf(stdout,"# The above objects are not defined by this script\n");
	fprintf(stdout,"\n");
	reverse_print(current_stmt);
	backtracking = 0;
	fprintf(stdout,"\n# ---------- END OF SOURCE FILE -----------\n");
}

reverse_print(p)	/* recursive */
STMT *p;
{
	STMT *q;

	q = p;
	if(!p) return;
	for(p = p->stmt_ptr; p; p = p->stmt_ptr)	/* next SCANNED stmt */
		if(p->flags==SCANNED)break;
	if(p) reverse_print(p);
	if(q->flags==SCANNED){
		fprintf(stdout,"%s\n",q->statement);
		q->flags = 0;
		}
}

clear_markers()
{
	STMT *p;
	for(p=current_stmt; p; p = p->stmt_ptr)
		p->flags = 0;
}

int nmarked;

undo_all(pin)
STMT *pin;
{
	SYM_LIST *q;
	STMT *p;
	int changes,i;
	nmarked = 0;
	backtracking = 1;
	for(q = pin->put_ptr; q; q = q->nxt_list)
		if(q->stmt_ptr){
			q->stmt_ptr->flags = MARKED;
			nmarked++;
		}
	do { /* repeatedly scan statements until no flags are MARKED */
		changes = 0;
		for(p=current_stmt; p; p = p->stmt_ptr)
			if(p->flags==MARKED){
				changes++;
				p->flags=SCANNED;
				for(q = p->get_ptr; q; q = q->nxt_list){
					if(q->stmt_ptr==NULL) continue;
					i = lookup(q->name);
					if(i<0) continue;
					if(hash_info[i]->current_stamp != q->stamp && q->stmt_ptr != p)
						q->stmt_ptr->flags = MARKED;
				}
			}
	} while(changes);
}

undo_script(p)
STMT *p;
{
	char *gettail(),*out;
	SYM_LIST *q;
	if(nmarked){
		fprintf(stdout,"function(){ # frame for temporaries\n");
		reverse_print(current_stmt);
		fprintf(stdout,"# permanent assign to recreate vars\n");
	}
	for(q = p->put_ptr; q; q = q->nxt_list){
		out = gettail(q->name);
		if(nmarked) fprintf(stdout,"%s <<- %s\n", out, out);
		else fprintf(stdout,"rm(%s)\n", out);
	}
	if(nmarked) fprintf(stdout,"}()\n");
	backtracking = 0;
}

STMT *
find_stmt(i)
{
	STMT *p;
	if(i>nstmt) return NULL;
	for(p=current_stmt; p; p = p->stmt_ptr)
		if(p->stmtno==i) break;
	return p;
}

print_names(s,p)
char *s; SYM_LIST* p;
{
	for(; p; p = p->nxt_list)
		fprintf(stdout,"%s %s %d\n", s, p->name,
			p->stmt_ptr?p->stmt_ptr->stmtno:0);
}

enter_stmt_on_list(p,stmt)
NAME_LIST **p; STMT *stmt;
{
	NAME_LIST *r;
	for(r = *p; r; r = r->nxt_list)
		if(r->stmt_ptr == stmt) return;
	r = (NAME_LIST *) calloc(1, sizeof(NAME_LIST));
	r->nxt_list = *p;
	r->stmt_ptr = stmt;
	*p = r;
}

enter_name_on_stmt(i,sym)
int i; SYM_LIST **sym;
{
	SYM_LIST *q;
	NAME_LIST *r;

	for(q = *sym; q; q = q->nxt_list)
		if(strcmp(hash_table[i],q->name)==0) return;
	q = (SYM_LIST *) calloc(1, sizeof(SYM_LIST));
	q->name = hash_table[i];
	q->stamp = hash_info[i]->current_stamp;
	r = hash_info[i]->put_ptr;
	q->stmt_ptr = r?r->stmt_ptr:NULL;
	q->nxt_list = *sym;
	*sym = q;
}

unsigned long smash(s)
char *s;
{
	unsigned long sum = 0;
	int i;
	i = 0;
	while(*s++){
		sum ^= *s<<(i*8);
		if(++i>3) i = 0;
		}
	return sum;
}

hash(s)
char *s;
{
	unsigned long smash(),i; int j,incr;
	i = smash(s);
	j = i % NPRIME;
	incr = (i / NPRIME) % NPRIME;
	if(incr==0) incr=1;
	while( hash_table[j] )
		if(strcmp(s,hash_table[j])!=0) j = (j+incr)%NPRIME;
		else break;
	/* the j'th entry is where it belongs */
	return j;
}

hash_init()
{
	int i;
	for(i=0; i<NPRIME; i++) hash_table[i] = NULL;
}

int nhash = 0;
make_hash_entry(s)
char *s;
{
	NAME_ENTRY *p;
	int i;
	char *strdup(),*calloc(),*cp;
	for(cp=s; *cp; cp++);	/* find end */
	*(cp-1) = '\0';	/* zap trailing quote */
	i = hash(s);
	if(hash_table[i]==NULL){	/* a new entry */
		if(++nhash > NPRIME-MINFREE) error("Hash table overflow");
		hash_table[i] = strdup(s);
		p = (NAME_ENTRY *) calloc(1, sizeof(NAME_ENTRY));
		p->get_ptr = NULL;
		p->put_ptr = NULL;
		p->current_stamp = 0;
		hash_info[i] = p;
		}
	return i;
}

print_hash(mask)
int mask;
{
	int i, thismask;
	for(i=0; i<NPRIME; i++)
		if(hash_table[i]){
			thismask = 0;
			if(hash_info[i]->get_ptr) thismask |= 1;
			if(hash_info[i]->put_ptr) thismask |= 2;
			if(mask==4 || mask==thismask)
				fprintf(stdout,"%c%c %s\n",
				hash_info[i]->get_ptr?'r':'-',
				hash_info[i]->put_ptr?'w':'-',
				hash_table[i]);
			}
}

tail(s1,s2)	/* zero if s1 matches tail of s2 */
char *s1,*s2;
{
	char *p1,*p2;
	for(p1=s1; *p1; p1++);
	for(p2=s2; *p2; p2++);
	for(; *p1 == *p2 && p1 >= s1 && p2 >= s2; p1--, p2-- ) ;
	if(p1==s1-1 && p2==s2-1) return 1;	/* exact match */
	if(p1==s1-1) return 0;	/* partial match */
	else return -1;
}

lookup(s)
char *s;
{
	int i,j,partial = -1;
	for(i=0; i<NPRIME; i++)
		if(hash_table[i]){
			j = tail(s,hash_table[i]);
			if(j>0) return(i);
			if(j==0) partial = i;
			}
	return partial;
}

extern int strlen();
extern char *malloc();

char *
strdup(s1) 

   char * s1;

{
   char * s2, *strcpy();

   s2 = malloc((unsigned) strlen(s1)+1) ;
   return(s2==NULL ? NULL : strcpy(s2,s1) );
}

char *
strchr(sp, c)
register char *sp;
int c;
{
	do {
		if (*sp == c)
			return(sp);
	} while (*sp++);
	return(NULL);
}

prompt()
{
	if(interact){
		fprintf(stderr,"audit: ");
		fflush(stderr);
	}
}

char *gettail(s)
char *s;
{
	char *p;
	for(p=s;*s;s++)
		if(*s=='/') p=s+1;
	return(p);
}
