/* -*- C -*- */

%{
#include <assert.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>

#define LITERAL		1
#define VARIABLE	2
#define SYMBOL		3

struct variable
{
  struct variable *next;
  int value;
  char name[0];
};

union
{
  int literal;
  struct variable *variable;
  char symbol;
} yylval;

struct variable *find_variable (char *);

int current_line;

%}

%%

[ \t]		/* do nothing */

\n		current_line++;

[0-9]*		{ yylval.literal = atoi (yytext); return LITERAL; }

[a-z_][a-z0-9_]* { yylval.variable = find_variable (yytext); return VARIABLE; }

[-*/+^|&~()=]	{ yylval.symbol = *yytext; return SYMBOL; }

.		{ fprintf (stderr, 
			 "Scanning error on line %d: Illegal character %c.\n",
			 current_line, *yytext); }

%%

main ()
{
  int ret;
  
  current_line = 1;

  for (;;)
    {
      ret = yylex ();
      switch (ret)
	{
	case LITERAL:
	  printf (":%d:Literal\t%d\n", current_line, yylval.literal);
	  break;
	  
	case VARIABLE:
	  printf (":%d:Variable\t%s\t(%#x)\n", 
		  current_line, yylval.variable->name, (int) yylval.variable);
	  break;
	  
	case SYMBOL:
	  printf (":%d:Symbol\t %c\n", current_line, yylval.symbol);
	  break;

	case 0: /* end of file */
	  exit (0);

	default:
	  abort ();
	}
    }
}

/* Return the variable structure corresponding to NAME.  If the same name
   is presented multiple times, always return the same structure. */
struct variable *
find_variable (char *name)
{
  static struct variable *all_variables = 0; /* A list of all the variables */
  struct variable *var;
  
  /* Uniquify result */
  for (var = all_variables; var; var = var->next)
    if (!strcmp (var->name, name))
      return var;

  /* allocate new structure */
  var = malloc (sizeof (struct variable) + strlen (name) + 1);
  assert (var);

  /* initialize */
  strcpy (var->name, name);
  var->value = 0;
  
  /* hook it in */
  var->next = all_variables;
  all_variables = var;
  
  return var;
}

  
 
  
 


