package antichess;

import java.util.TreeMap;


/**
 * GameClock is a represents a set of timers for use in a game.  
 * GameClock contains a set of GameTimers referenced to by their Player.
 * Only one of these timers are counting down at any one time.  
 * 
 * 
 *@specfield timers						//sequence of GameTimers mapped to by their Player
 *@specfield timerListeners		  		//sequence of a sequence of GameTimerListeners mapped to by their Player
 *										  so a Player->sequence of GameTimerListeners
 *@specfield currentTimer    			//sets which timer is counting down right now if any 
 */
public class GameClock {
	//Fields
	private TreeMap<Player, GameTimer> timerMap;
	private Player currentTimer;
	private static final Player NONE = Player.NONE;
	
	//Abstraction Function
	//Timers = the GameTimers mapped to by timerMap
	//timerListeners = the listeners of all the elements of Timers
	//currentTimer = currentTimer
	
	//Representation Invariants
	//timerMap != null;
	//timerMap does not map any valid keys to null
	//currentTimer is a Player in timerMap.keySet() or NONE
	//the only timer running is currentTimer
	
	
	private void checkRep(){
		if (timerMap == null){ throw new RuntimeException("timerMap is null");}
		if (!(timerMap.containsKey(currentTimer) || currentTimer.equals(NONE))){
			throw new RuntimeException("CurrentTimer references timer not in GameClock");
		}
		int numberRunning = 0;
		for (Player pl : timerMap.keySet()){
			if (timerMap.get(pl) ==null){throw new RuntimeException("timerMap maps value to null");}
			if (timerMap.get(pl).isRunning()){
				numberRunning++;
			}
		}
		if (numberRunning > 1){ 
			throw new RuntimeException("Multiple timers are running");
		}
	}
	
	
	/**
	 * playerList is an array of Players which each clock will be associated with
	 * startTimes is a list of start times for the players in playerList.
	 * Values of the same index correspond to eachother.
	 * 
	 * 
	 * @requires playerList must not contain identical elements
	 * @throws IllegalArgumentException if playerList = null, startTimes = null,
	 *  a value in startTimes is < 0, playerList and startTimes are not the same length
	 *  
	 */
	
	public GameClock(Player[] playerList, long[] startTimes){
		this(playerList, startTimes, startTimes);
	}
	
	
	public GameClock(Player[] playerList, long[] startTimes, long[] currentTimes){
		//checks for the conditions that would cause it to throw and error
		if ((playerList == null) || (startTimes == null)){
			throw new IllegalArgumentException("GameClock constructor null value passed");
		}else if (playerList.length != startTimes.length){
			throw new IllegalArgumentException("GameClock constructor startTimes and playerList have different lengths");
		}
		else {
			for (long st : startTimes){
				if (st < 0){
					throw new IllegalArgumentException("GameClock constructor negative startTime passed");
				}
			}
		}
		
		//loads all of the timers into a map
		timerMap = new TreeMap<Player, GameTimer>();
		for (int i=0; i<playerList.length; i++){
			timerMap.put(playerList[i], new GameTimer(startTimes[i], currentTimes[i], 500));
		}
		
		//sets the current timer to none
		currentTimer = NONE;
		checkRep();
	}
	
	
	/**
	 * @modifies timerListeners
	 * @effects adds listener to timerListeners mapped to by player
	 * @throws IllegalArgumentException if no timers are referenced by player
	 */
	public void addListener(GameTimerListener listener, Player player){
		if (!timerMap.containsKey(player)){
			throw new IllegalArgumentException("No GameTimer referenced by player");
		}
		timerMap.get(player).addTimerListener(listener);
		checkRep();
	}
	
	/**
	 * @modifies currentTimer
	 * @effects currentTimer = timer mapped to by player
	 * @throws IllegalArgumentException if no timers are referenced by player
	 */
	
	public void startTimer(Player player){
		if (!timerMap.containsKey(player)){
			throw new IllegalArgumentException("No GameTimer referenced by player");
		}
		if (currentTimer.equals(NONE)){
			currentTimer = player;
			timerMap.get(player).start();
		}else{
			timerMap.get(currentTimer).stop();
			timerMap.get(player).start();
			currentTimer = player;
		}
		checkRep();
	}

	
	/**
	 * @modifies currentTimer
	 * @effects sets the currentTimer to none
	 */
	public void stopTimer(){
		if (!currentTimer.equals(NONE))
		{
			timerMap.get(currentTimer).stop();
			currentTimer = NONE;
			checkRep();
		}
	}
	
	/**
	 * @return the time remaining on the GameTimer mapped to by player
	 * @throws IllegalArgumentException if no timers are referenced by player
	 */
	public long getTime(Player player){
		if (!timerMap.containsKey(player)){
			throw new IllegalArgumentException("No GameTimer referenced by player");
		}
		return timerMap.get(player).getTime();
	}
	

	

	/**
	 * @modifies player's Timer
	 * @effects sets the Time of the GameTimer mapped to by player to newTime
	 * 			if the GameTimer mapped to by player is not running.
	 * @throws IllegalArgumentException if no timers are referenced by player 
	 * 									or if newTime < 0
	 */
	public void setTime(long newTime, Player player){
		if (!timerMap.containsKey(player)){
			throw new IllegalArgumentException("No GameTimer referenced by player");
		}
		timerMap.get(player).setTime(newTime);
		checkRep();
	}
	
	/**
	 * @returns currentTimer
	 */
	public Player getCurrentPlayer(){
		return currentTimer;
	}
	
	/**
	 * @requires the GameClock have a timer referenced by player
	 * @returns the GameTimer referenced by Player
	 */
	public GameTimer getTimer(Player player){
		return timerMap.get(player);
	}
	
	/**
	 * @returns true if this GameClock has a timer associated with player
	 */
	public boolean hasPlayer(Player player){
		if (timerMap.containsKey(player)){
			return true;
		}
		return false;
	}
}
