
package antichess;

import java.util.List;
import java.util.TreeMap;


/**
 * Game controller handles getting moves from GamePlayers
 * and then making these moves on the board.  The GameController
 * also checks for win conditions on the players.  The clock
 * used as a parameter for the method createGameController has timers
 * for each of the Players.  If the clock does not have a timer associated
 * with the Player then time is not a lose condition for that player. 
 * 
 * 
 * 
 * @author nlharr
 *
 * @specfield board			//Board that the GameController uses when playing the game
 * @specfield playerList	//Sequence of GamePlayers that stores who and in what order are playing the game
 * @specfield clock			//GameClock that the controller uses to determine win conditions based off of time
 * 							//if clock does not have a timer for a specific player the players game is untimed
 * 
 * @specfield currentPlayer //GamePlayer the player who is currently making a move
 */
public class GameController implements Runnable{
	
	//Fields
	private Board currentBoard;
	private TreeMap<Player, GamePlayer> playerMap;
	private GameClock clock;
	private ControllerMaster master;
	
	
	//Abstraction Function
	// board = currentBoard
	// playerList = the values mapped to by playerMap
	// clock = clock
	
	//Representation Invariants
	// currentBoard != null
	// clock != null
	
	
	private void checkRep(){
		if (currentBoard == null) throw new RuntimeException("currentBoard is null");
		if (clock == null) throw new RuntimeException("clock is null");
		
	}
	
	
	//private constructor to prevent people form calling a public constructor
	//and for use with the factory method
	private GameController(Board board, GameClock clock, 
			List<GamePlayer> playerList, ControllerMaster master){

		playerMap = new TreeMap<Player, GamePlayer>();
		for(GamePlayer pl : playerList){
			playerMap.put(pl.getPlayerType(), pl);
		}
		this.currentBoard = board;
		
		this.clock = clock;
		this.master = master;
		
		checkRep();
	}
	
	
	/**
	 * 
	 * Creates a new GameController using board and clock
	 * The time running out is assumed to be a lose condition
	 * for a Player.  If the clock does not have timer for the 
	 * Player then that Player has not lose conditions based on time.
	 * master has control over this GameController.
	 * 
	 * 
	 * @requires board != null, playerList != null, clock != null
	 *			playerList contians at least one element, master!= null
	 * @returns a new GameController operating on board and using clock to determine timebased win conditions
	 */
	public static GameController createGameController(Board board, GameClock clock, 
			List<GamePlayer> playerList, ControllerMaster master){
		return new GameController(board, clock, playerList, master);
	}
	

	
	/**
	 * @modifies board
	 * @effects if time runs out lets the board know that it has lost this way
	 * @return true if the game has ended
	 */
	public synchronized boolean isOver(){
		//stores the number of players WithNoTime
		int playersWithNoTime = 0;
		for (Player pl : playerMap.keySet()){
			if (clock.hasPlayer(pl)){
				if (clock.getTime(pl) == 0){
					playersWithNoTime++;
					
				}
			}
		}
		//checks to see if we have met the timing end conditions
		if (playersWithNoTime == playerMap.keySet().size()){
			currentBoard.endGame(Board.OUTOFTIME, Player.NONE);
		}
		//if there is only one player who isn't out of time then he wins
		else if (playersWithNoTime == playerMap.keySet().size()-1){
			//we search for the player that won
			for (Player pl : playerMap.keySet()){
				if (clock.hasPlayer(pl)){
					if (clock.getTime(pl)!=0){
						//if there is a player with no time and has won
						currentBoard.endGame(Board.OUTOFTIME, pl);
					}
				}else{
					//if there is one untimed player he has won
					currentBoard.endGame(Board.OUTOFTIME, pl);
				}
			}
		}
		return currentBoard.isGameOver();
	}
	
	
	/**
	 * Notifies this GameController that the currentPlayer has a move
	 *
	 */
	public synchronized void notifyControllerHasMove(){
		this.notify();
	}


	
	
	
	/**
	 * @return this.clock
	 */
	public synchronized GameClock getClock() {
		return clock;
	}
	

	
	/**
	 * Handles the operations of the GameController class
	 * this is the loop meant to be invoked in it's own thread
	 */
	public synchronized void run(){
		master.setGameControllerRunning(this);
		
		Move receivedMove;
		GamePlayer currentPlayer;
		
		//add check for gamecontroller
		while (!isOver() && !master.isAwaitingPause()){
			currentPlayer = playerMap.get(currentBoard.getPlayer());
			
			if (clock.hasPlayer(currentBoard.getPlayer())){
				clock.startTimer(currentBoard.getPlayer());
			}else{
				clock.stopTimer();
			}
			
			//checks to see if the board has legal moves
			if (currentBoard.allLegalMoves().size() != 0){
				receivedMove = currentPlayer.getMove(this);
				//tests if the currentPlayer has returned a valid move yet
				//or if the master is wants the GameController to pause
				//if either of these conditions hold the loop is skipped
				while (receivedMove == null && !master.isAwaitingPause()
						&& !isOver()) 
				{
				      try {
				        this.wait(1000);
				      } catch (InterruptedException e) {}
				      receivedMove = currentPlayer.getMove(this);
				      
				}
				
				
				
				try{
					if (receivedMove != null){
						//performs the currentMove
						if (clock.hasPlayer(currentPlayer.getPlayerType())){
							currentBoard.doMove(receivedMove, Long.toString(clock.getTime(currentPlayer.getPlayerType())));
						}else{
							currentBoard.doMove(receivedMove);
						}
					}
				}catch (Exception ex){
					throw new RuntimeException("GamePlayer generated "+ex.getMessage());
				}
			}else{
				try{
					//asks the board to do a null move meaning the player does not have a valid move
					if (clock.hasPlayer(currentPlayer.getPlayerType())){
						currentBoard.doMove(null, Long.toString(clock.getTime(currentPlayer.getPlayerType())));
					}else{
						currentBoard.doMove(null);
					}
				}catch (Exception ex){
						throw new RuntimeException("Board threw error on null move");
				}
			}
			
			//checks to see if the game is over
			if (isOver()){
				clock.stopTimer();
				currentBoard.notifyObservers();
			}
			
		}
		clock.stopTimer();
		master.setGameControllerPaused(this);
		
	}

	
}
